#ifndef OTC_COLLCTN_MAP_C
#define OTC_COLLCTN_MAP_C
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     collctn/map.c
// 
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/tblentry.hh>
#include <OTC/collctn/linklist.hh>
#include <OTC/collctn/hashdefs.hh>
#include <OTC/collctn/kcursor.hh>
#include <OTC/collctn/icursor.hh>
#include <OTC/collctn/kicursor.hh>
#include <OTC/collctn/prbucket.hh>

/* ------------------------------------------------------------------------- */
#ifdef __OSE_TEMPLATES__
template<class T1, class T2> OTC_Map
{
  OSE_DECLARE OTC_PairBucket<T1,T2>;
  OSE_DECLARE OTC_ItemCursor< T2,OTC_PairBucket<T1,T2> >;
  OSE_DECLARE OTC_KeyCursor< T1,OTC_PairBucket<T1,T2> >;
  OSE_DECLARE OTC_KeyItemCursor< T1,T2,OTC_PairBucket<T1,T2> >;
  OSE_IMPLEMENT OTC_Iterator<T2>;
  OSE_IMPLEMENT OTC_Iterator<T1>;
  OSE_IMPLEMENT OTC_Modifier<T2>;
  OSE_IMPLEMENT OTC_HashActions<T1>;
  OSE_IMPLEMENT OTC_RankActions<T1>;
  OSE_IMPLEMENT OTC_PairIterator<T1,T2>;
  OSE_IMPLEMENT OTC_PairModifier<T1,T2>;
  OSE_IMPLEMENT OTC_PairBucket<T1,T2>;
  OSE_IMPLEMENT OTC_ItemCursor< T2,OTC_PairBucket<T1,T2> >;
  OSE_IMPLEMENT OTC_KeyCursor< T1,OTC_PairBucket<T1,T2> >;
  OSE_IMPLEMENT OTC_KeyItemCursor< T1,T2,OTC_PairBucket<T1,T2> >;
};
#endif

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
OTC_Map<T1,T2>::~OTC_Map()
{
  myPopulation = 0;

  // Null out pointer to table so that member functions can check
  // for access to partially destroyed object from destructors of
  // objects deleted.

  delete [] myTable;
  myTable = 0;

  myPairs->unReference();
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
OTC_Map<T1,T2>::OTC_Map(OTC_Map<T1,T2> const& theMap)
  : myPairs(0),
    myTable(0),
    myTableSize(theMap.myTableSize),
    myPopulation(0),
    myLowThreshold(theMap.myLowThreshold),
    myHighThreshold(theMap.myHighThreshold),
    myRankFn(theMap.myRankFn)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myPairs = new ((os_segment*)theLocality) OTC_LinkList;
#else
#if defined(ENV_OSTORE)
  myPairs = new (theLocality,OTC_LinkList::get_os_typespec()) OTC_LinkList;
#else
  myPairs = new OTC_LinkList;
#endif
#endif
  OTCLIB_ASSERT(myPairs != 0);
  myPairs->reference();

#if defined(ENV_OSTORE_DML)
  myTable = new ((os_segment*)theLocality) OTC_TableEntry[myTableSize];
#else
#if defined(ENV_OSTORE)
  myTable = new (theLocality,OTC_TableEntry::get_os_typespec(),myTableSize)
   OTC_TableEntry[myTableSize];
#else
  myTable = new OTC_TableEntry[myTableSize];
#endif
#endif
  OTCLIB_ASSERT(myTable != 0);

  OTC_PairIterator<T1,T2> iter = 0;
  iter = theMap.pairs(OTCLIB_FORWARD,OTCLIB_UNSAFE);
  for (iter.reset(); iter.isValid(); iter.next())
    add(iter.key(),iter.item());
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
OTC_Map<T1,T2>::OTC_Map(int (*theRankFn)(T1 const&,T1 const&))
  : myPairs(0),
    myTable(0),
    myTableSize(OTCLIB_START_TBLSZ),
    myPopulation(0),
    myLowThreshold(OTCLIB_THRESHOLD(OTCLIB_START_TBLSZ,OTCLIB_LOW_THRESHOLD)),
    myHighThreshold(OTCLIB_THRESHOLD(OTCLIB_START_TBLSZ,OTCLIB_HIGH_THRESHOLD)),
    myRankFn(theRankFn)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myPairs = new ((os_segment*)theLocality) OTC_LinkList;
#else
#if defined(ENV_OSTORE)
  myPairs = new (theLocality,OTC_LinkList::get_os_typespec()) OTC_LinkList;
#else
  myPairs = new OTC_LinkList;
#endif
#endif
  OTCLIB_ASSERT(myPairs != 0);
  myPairs->reference();

#if defined(ENV_OSTORE_DML)
  myTable = new ((os_segment*)theLocality) OTC_TableEntry[myTableSize];
#else
#if defined(ENV_OSTORE)
  myTable = new (theLocality,OTC_TableEntry::get_os_typespec(),myTableSize)
   OTC_TableEntry[myTableSize];
#else
  myTable = new OTC_TableEntry[myTableSize];
#endif
#endif
  OTCLIB_ASSERT(myTable != 0);
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
void OTC_Map<T1,T2>::removeAll()
{
  // Make sure we mark all items in index as deleted before killing objects
  // in list in case deleted objects try to access this collection from a
  // destructor.

  myPopulation = 0;
  for (u_int i=0; i<myTableSize; i++)
    myTable[i].markDeleted();

  myPairs->removeAll();

  // If index is greater than the defined floor size then delete it and
  // go back to the start size so that space will be reclaimed.

  if (myTableSize > OTCLIB_FLOOR_TBLSZ)
  {
    delete [] myTable;
    myTableSize = OTCLIB_START_TBLSZ;
    myLowThreshold = OTCLIB_THRESHOLD(myTableSize,OTCLIB_LOW_THRESHOLD);
    myHighThreshold = OTCLIB_THRESHOLD(myTableSize,OTCLIB_HIGH_THRESHOLD);

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
    OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
    myTable = new ((os_segment*)theLocality) OTC_TableEntry[myTableSize];
#else
#if defined(ENV_OSTORE)
    myTable = new (theLocality,OTC_TableEntry::get_os_typespec(),
     myTableSize) OTC_TableEntry[myTableSize];
#else
    myTable = new OTC_TableEntry[myTableSize];
#endif
#endif
    OTCLIB_ASSERT(myTable != 0);
  }
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
OTC_Map<T1,T2>& OTC_Map<T1,T2>::operator=(OTC_Map<T1,T2> const& theMap)
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the map and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  if (&theMap != this)
  {
    removeAll();

    myRankFn = theMap.myRankFn;
    OTC_PairIterator<T1,T2> iter = 0;
    iter = theMap.pairs(OTCLIB_FORWARD,OTCLIB_UNSAFE);
    for (iter.reset(); iter.isValid(); iter.next())
      add(iter.key(),iter.item());
  }

  return *this;
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
void OTC_Map<T1,T2>::add(T1 const& theKey, T2 const& theItem)
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the map and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  // If this addition to the map will mean that we were over the threshold
  // then resize the map before adding in this entry.

  if (population() + 1 > myHighThreshold)
    resize(otclib_hashtable_grow(myTableSize));

  // Search for key.

  int theHashValue = 0;
  int theIndex = -1;

  OTC_Boolean keyExists = search(theKey,theIndex,theHashValue);
  OTCLIB_ENSURE((keyExists == OTCLIB_FALSE),
   "OTC_Map::add() - Key already exists");

  // Add the key and item to the table and list.

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

  OTC_PairBucket<T1,T2>* theBucket;
#if defined(ENV_OSTORE_DML)
    theBucket = new ((os_segment*)theLocality)
     OTC_PairBucket<T1,T2>(theKey,theItem);
#else
#if defined(ENV_OSTORE)
  theBucket = new (theLocality,OTC_PairBucket<T1,T2>::get_os_typespec())
   OTC_PairBucket<T1,T2>(theKey,theItem);
#else
  theBucket = new OTC_PairBucket<T1,T2>(theKey,theItem);
#endif
#endif
  OTCLIB_ASSERT(theBucket != 0);

  OTC_TableEntry theEntry(theHashValue,theBucket);

  myTable[theIndex] = theEntry;
  myPairs->addLast(theBucket);
  myPopulation++;
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
void OTC_Map<T1,T2>::remove(T1 const& theKey)
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the map and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  // Search for key.

  int theHashValue = 0;
  int theIndex = -1;

  OTC_Boolean keyExists = search(theKey,theIndex,theHashValue);
  OTCLIB_ENSURE((keyExists == OTCLIB_TRUE),
   "OTC_Map::remove() - Key does not exist");

  OTC_TableEntry& theEntry = myTable[theIndex];
  theEntry.link()->kill();
  theEntry.markDeleted();
  myPopulation--;

  // Check population against table floor size and low threshold. If we
  // need to, reduce the size of the table.

  if (myTableSize > OTCLIB_FLOOR_TBLSZ && myPopulation < myLowThreshold)
    resize(otclib_hashtable_shrink(myTableSize));
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
OTC_Boolean OTC_Map<T1,T2>::contains(T1 const& theKey) const
{
  int theHashValue = 0;
  int theIndex = -1;

  return search(theKey,theIndex,theHashValue);
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
T2& OTC_Map<T1,T2>::_item(T1 const& theKey) const
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the map and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  // Search for presence of key.

  int theHashValue = 0;
  int theIndex = -1;

  OTC_Boolean keyExists = search(theKey,theIndex,theHashValue);
  OTCLIB_ENSURE((keyExists == OTCLIB_TRUE),
   "OTC_Map::_item() - Key does not exist");

  T2& theItem = ((OTC_PairBucket<T1,T2>*)myTable[theIndex].link())->item();
  return theItem;
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
T2* OTC_Map<T1,T2>::_pItem(T1 const& theKey) const
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the map and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  // Search for presence of key.

  int theHashValue = 0;
  int theIndex = -1;

  OTC_Boolean keyExists = search(theKey,theIndex,theHashValue);

  if (keyExists == OTCLIB_FALSE)
    return 0;

  T2& theItem = ((OTC_PairBucket<T1,T2>*)myTable[theIndex].link())->item();
  return &theItem;
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
T1 const& OTC_Map<T1,T2>::pickKey() const
{
  OTCLIB_ENSURE((population() != 0),
   "OTC_Map::pickKey() - collection must not be empty");

  return ((OTC_PairBucket<T1,T2>*)myPairs->first())->key();
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
T2 const& OTC_Map<T1,T2>::pickItem() const
{
  OTCLIB_ENSURE((population() != 0),
   "OTC_Map::pickItem() - collection must not be empty");

  return ((OTC_PairBucket<T1,T2>*)myPairs->first())->item();
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
OTC_Iterator<T1> OTC_Map<T1,T2>::keys(
 OTC_Direction theDirection,
 OTC_Protection theProtection
) const
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
  if (!theLocality.isTransientSegment())
  {
    os_transaction* theTXN = os_transaction::get_current();
    if (theTXN != 0 && theTXN->get_type() == os_transaction::read_only)
      theProtection = OTCLIB_UNSAFE;

    if (theProtection == OTCLIB_UNSAFE)
      theLocality = os_segment::get_transient_segment();
  }
#endif

  OTC_Cursor<T1>* theIter;
#if defined(ENV_OSTORE_DML)
  theIter = new ((os_segment*)theLocality)
   OTC_KeyCursor< T1,OTC_PairBucket<T1,T2> >(
   myPairs,theDirection,theProtection);
#else
#if defined(ENV_OSTORE)
  theIter = new (theLocality,
   OTC_KeyCursor< T1,OTC_PairBucket<T1,T2> >::get_os_typespec())
   OTC_KeyCursor< T1,OTC_PairBucket<T1,T2> >(myPairs,theDirection,
   theProtection);
#else
  theIter = new OTC_KeyCursor< T1,OTC_PairBucket<T1,T2> >(myPairs,theDirection,
   theProtection);
#endif
#endif
  OTCLIB_ASSERT(theIter != 0);
  return theIter;
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
OTC_Cursor<T2>* OTC_Map<T1,T2>::_items(
 OTC_Direction theDirection,
 OTC_Protection theProtection
) const
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
  if (!theLocality.isTransientSegment())
  {
    os_transaction* theTXN = os_transaction::get_current();
    if (theTXN != 0 && theTXN->get_type() == os_transaction::read_only)
      theProtection = OTCLIB_UNSAFE;

    if (theProtection == OTCLIB_UNSAFE)
      theLocality = os_segment::get_transient_segment();
  }
#endif

  OTC_Cursor<T2>* theIter;
#if defined(ENV_OSTORE_DML)
  theIter = new ((os_segment*)theLocality)
   OTC_ItemCursor< T2,OTC_PairBucket<T1,T2> >(
   myPairs,theDirection,theProtection);
#else
#if defined(ENV_OSTORE)
  theIter = new (theLocality,
   OTC_ItemCursor< T2,OTC_PairBucket<T1,T2> >::get_os_typespec())
   OTC_ItemCursor< T2,OTC_PairBucket<T1,T2> >(myPairs,theDirection,
   theProtection);
#else
  theIter = new OTC_ItemCursor< T2,OTC_PairBucket<T1,T2> >(myPairs,theDirection,
   theProtection);
#endif
#endif
  OTCLIB_ASSERT(theIter != 0);
  return theIter;
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
OTC_PairCursor<T1,T2>* OTC_Map<T1,T2>::_pairs(
 OTC_Direction theDirection,
 OTC_Protection theProtection
) const
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
  if (!theLocality.isTransientSegment())
  {
    os_transaction* theTXN = os_transaction::get_current();
    if (theTXN != 0 && theTXN->get_type() == os_transaction::read_only)
      theProtection = OTCLIB_UNSAFE;

    if (theProtection == OTCLIB_UNSAFE)
      theLocality = os_segment::get_transient_segment();
  }
#endif
 
  OTC_PairCursor<T1,T2>* theIter;
#if defined(ENV_OSTORE_DML)
  theIter = new ((os_segment*)theLocality)
   OTC_KeyItemCursor< T1,T2,OTC_PairBucket<T1,T2> >(
  myPairs,theDirection,theProtection);
#else
#if defined(ENV_OSTORE)
  theIter = new (theLocality,
   OTC_KeyItemCursor< T1,T2,OTC_PairBucket<T1,T2> >::get_os_typespec())
   OTC_KeyItemCursor< T1,T2,OTC_PairBucket<T1,T2> >(myPairs,theDirection,
   theProtection);
#else
  theIter = new OTC_KeyItemCursor< T1,T2,OTC_PairBucket<T1,T2> >(myPairs,
   theDirection,theProtection);
#endif
#endif
  OTCLIB_ASSERT(theIter != 0);
  return theIter;
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
OTC_Boolean OTC_Map<T1,T2>::search(
 T1 const& theKey,
 int& theIndex,
 int& theHashValue
) const
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the map and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  // Get the hash value for <theKey>.

  theHashValue = OTC_HashActions<T1>::hash(theKey);

  // Calculate the start of the search.

  u_int anIndex = otclib_hashtable_mask(theHashValue,myTableSize);

  // Do the search. Note that we cannot simply search till the end of the
  // table but must wrap around to the start and search any entries prior
  // to <theOffset> in the table.

  theIndex = -1;
  int theAvailable = -1;

  for (u_int i=0; i<=myTableSize; i++)
  {
    OTC_TableEntry& anEntry = myTable[anIndex];

    if (anEntry.isEmpty() != OTCLIB_FALSE)
    {
      if (theAvailable == -1)
        theAvailable = anIndex;

      break;
    }
    else if (anEntry.isDeleted() != OTCLIB_FALSE)
    {
      if (theAvailable == -1)
        theAvailable = anIndex;
    }
    else if (anEntry.isOccupied() != OTCLIB_FALSE)
    {
      if (anEntry.hashValue() == theHashValue)
      {
        T1& aKey = ((OTC_PairBucket<T1,T2>*)anEntry.link())->key();
        OTC_Boolean theSame = (rank(theKey,aKey) == 0);
        if (theSame != OTCLIB_FALSE)
        {
          theIndex = anIndex;

          break;
        }
      }
    }
    if (i == 0)
    {
      anIndex = otclib_hashtable_mask(
       anIndex + otclib_hashtable_doublehash(theHashValue,myTableSize),
       myTableSize
      );
    }
    else if (++anIndex == myTableSize)
    {
      anIndex = 0;
    }
  }

  if (theIndex == -1)
  {
    theIndex = theAvailable;
    return OTCLIB_FALSE;
  }

  return OTCLIB_TRUE;
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
void OTC_Map<T1,T2>::resize(u_int theNewSize)
{
  u_int oldTableSize = myTableSize;
  myTableSize = theNewSize;
  myLowThreshold = OTCLIB_THRESHOLD(myTableSize,OTCLIB_LOW_THRESHOLD);
  myHighThreshold = OTCLIB_THRESHOLD(myTableSize,OTCLIB_HIGH_THRESHOLD);
  OTC_TableEntry* oldTable = myTable;

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myTable = new ((os_segment*)theLocality) OTC_TableEntry[myTableSize];
#else
#if defined(ENV_OSTORE)
  myTable = new (theLocality,OTC_TableEntry::get_os_typespec(),
   myTableSize) OTC_TableEntry[myTableSize];
#else
  myTable = new OTC_TableEntry[myTableSize];
#endif
#endif
  OTCLIB_ASSERT(myTable != 0);

  // Traverse through the old table and put each entry in the new table.
  // Note that we cannot simply call <add()> to add the entry into the
  // new table as it will result in another copy of the item being placed
  // in the list. Also note that we do not have to check that an item
  // already exists as we are already guaranteed that they are unique, thus
  // simply search for first empty slot.

  for (u_int j=0; j<oldTableSize; j++)
  {
    OTC_TableEntry& oldEntry = oldTable[j];
    if (oldEntry.isOccupied() != OTCLIB_FALSE)
    {
      int theHashValue = oldEntry.hashValue();
      u_int anIndex = otclib_hashtable_mask(theHashValue,myTableSize);

      for (u_int i=0; i<=myTableSize; i++)
      {
        OTC_TableEntry& anEntry = myTable[anIndex];

        if (anEntry.isEmpty() != OTCLIB_FALSE)
        {
          anEntry = oldEntry;
          break;
        }
	if (i == 0)
	{
	  anIndex = otclib_hashtable_mask(
	   anIndex + otclib_hashtable_doublehash(theHashValue,myTableSize),
	   myTableSize
	  );
	}
	else if (++anIndex == myTableSize)
	{
	  anIndex = 0;
	}
      }
    }
  }

  delete [] oldTable;
}

/* ------------------------------------------------------------------------- */
template<class T1, class T2>
int OTC_Map<T1,T2>::rank(T1 const& key1, T1 const& key2) const
{
  return (myRankFn != 0) ? (*myRankFn)(key1,key2) :
   OTC_RankActions<T1>::rank(key1,key2);
}

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_MAP_C */
