#ifndef OTC_COLLCTN_MAP_HH
#define OTC_COLLCTN_MAP_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     collctn/map.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/prtction.hh>
#include <OTC/collctn/iterator.hh>
#include <OTC/collctn/modifier.hh>
#include <OTC/collctn/hashactn.hh>
#include <OTC/collctn/rankactn.hh>
#include <OTC/collctn/dirction.hh>
#include <OTC/collctn/pritertr.hh>
#include <OTC/collctn/prmodifr.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/map.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/map.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template<class T1, class T2> OTC_Map
{
  OSE_DECLARE OTC_Iterator<T1>;
  OSE_DECLARE OTC_Iterator<T2>;
  OSE_DECLARE OTC_Modifier<T2>;
  OSE_DECLARE OTC_HashActions<T1>;
  OSE_DECLARE OTC_RankActions<T1>;
  OSE_DECLARE OTC_PairIterator<T1,T2>;
  OSE_DECLARE OTC_PairModifier<T1,T2>;
};
#endif

class OTC_LinkList;
class OTC_LinkIterator;
class OTC_TableEntry;

template<class T1, class T2>
class OTC_Map
    // = TITLE
    //	   Class implementing a one way map between two objects.
    //	   
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   This class implements a one way mapping between two objects. The
    //	   first of these objects, the key, must be unique and must be able
    //	   to have a hash value generated for it using the <OTC_HashActions>
    //	   class. It must also be possible to compare the keys. The method
    //	   of comparing keys can be determined by providing a template
    //	   override version of <OTC_RankActions> for the type of the key.
    //	   
    // = NOTES
    //	   When a pointer type is used as a key and the hash value generated
    //	   from it, is based not on the pointer, but on some part of the
    //	   object being pointed at, then it is important that the pointer be
    //	   of type pointer to const object. In other words you should define
    //	   the map as <OTC_Map\<EX_Foo const*,EX_Bar*\>> and not just
    //	   <OTC_Map\<EX_Foo*,EX_Bar*\>>. If this is not done, it would be
    //	   possible for a user to modify parts of the key from which the hash
    //	   value is generated. If the hash value of an object changes, it
    //	   will no longer be accessible.
    //	   
    //	   The <OTC_Bucket> class is used internally to hold both key and
    //	   items. Thus the <OTC_BaseActions> class may be used to provide
    //	   actions to be performed, when items are inserted or removed from
    //	   the map.
    //
    //	   Since an attempt to access or remove an item from the map, which
    //	   isn't there, will raise an exception, it is important to first
    //	   check that an item is actually in the map by calling the
    //	   <contains()> function.
    //	   
    // = SEE ALSO
    //	   <OTC_Iterator>, <OTC_Modifier>, <OTC_HashActions>, <OTC_Bucket>,
    //	   <OTC_BaseActions>, <OTC_RankActions>, <OTC_PairIterator>,
    //	   <OTC_PairModifier>
{
  public:

#if defined(CXX_LC)
    typedef		OTC_Modifier<T2> type1;
    type1		dummy1();
    typedef		OTC_PairModifier<T1,T2> type2;
    type2		dummy2();
#endif

			~OTC_Map();

    // = INITIALISATION

			OTC_Map(int (*theRankFn)(T1 const&,T1 const&)=0);
				// Creates an empty map. <theRankFn> is an
				// optional comparison function to be used in
				// place of <OTC_RankActions> for the key.

			OTC_Map(OTC_Map<T1,T2> const& theMap);
				// Creates a copy of <theMap>.

    // = DESTRUCTION

    void		removeAll();
				// Removes all key/item pairs in this map.

    // = INSERTION

    OTC_Map<T1,T2>&	operator=(OTC_Map<T1,T2> const& theMap);
				// Replaces the contents of this map with
				// the contents of <theMap>.

    void		add(T1 const& theKey, T2 const& theItem);
				// If an item corresponding to <theKey> does
				// not exist in this map, then <theItem> will
				// be entered into this map. If an item
				// corresponding to <theKey> already exists,
				// an exception will be raised.

    // = RETRIEVAL/SEARCH

    void		remove(T1 const& theKey);
				// If an item corresponding to <theKey>
				// exists in this map, it will be removed
				// and deleted. If an item corresponding to
				// <theKey> does not exist, an exception
				// will be raised.

    OTC_Boolean		contains(T1 const& theKey) const;
				// Returns <OTCLIB_TRUE> if this map contains
				// a item corresponding to <theKey>.

    T2&			item(T1 const& theKey)
				{ return _item(theKey); }
				// If an item corresponding to <theKey>
				// exists in this map, a reference to it will
				// be returned. If an item corresponding to
				// <theKey> does not exist, an exception will
				// be raised.

    T2 const&		item(T1 const& theKey) const
				{ return _item(theKey); }
				// If an item corresponding to <theKey>
				// exists in this map, a reference to it will
				// be returned. If an item corresponding to
				// <theKey> does not exist, an exception will
				// be raised.

    T2*			pItem(T1 const& theKey)
				{ return _pItem(theKey); }
				// If an item corresponding to <theKey>
				// exists in this map, a pointer to it will
				// be returned. If an item corresponding to
				// <theKey> does not exist, a null pointer
				// will be returned.

    T2 const*		pItem(T1 const& theKey) const
				{ return _pItem(theKey); }
				// If an item corresponding to <theKey>
				// exists in this map, a pointer to it will
				// be returned. If an item corresponding to
				// <theKey> does not exist, a null pointer
				// will be returned.

    T1 const&		pickKey() const;
				// Returns a key from the collection. If
				// the map is empty, an exception is raised.
				// The function is not guaranteed to
				// return a different key on each call.

    T2 const&		pickItem() const;
				// Returns an item from the collection. If
				// the map is empty, an exception is raised.
				// The function is not guaranteed to
				// return a different item on each call.

    // = QUERY

    u_int		population() const
				{ return myPopulation; }
				// Returns the number of key/item mappings
				// contained in this map.

    OTC_Boolean		isEmpty() const
				{ return myPopulation == 0; }
				// Returns <OTCLIB_TRUE> if the collection
				// is empty.

    // = ITERATION
    //	   By default, iterators will perform reference counts on the
    //	   buckets in the collection as the iterator moves over the items.
    //	   Performing the reference counts ensures that the iterator
    //	   is not corrupted by additions or removals to the collection.
    //	   If an unsafe iterator is required, for example, to avoid
    //	   grabbing a write lock when using ObjectStore, a second
    //	   argument can be passed to the following functions. The value
    //	   of this argument is either <OTCLIB_SAFE> or <OTCLIB_UNSAFE>.
    //	   To get an unsafe iterator, the <OTCLIB_UNSAFE> argument should
    //	   be used.
    //
    //	   The first argument to the following functions indicates the
    //	   direction of traversal of the iterator. Traversal in the
    //	   direction of first to last item is indicated by a value of
    //	   <OTCLIB_FORWARD>. Traversal in the reverse direction is
    //	   indicated by a value of <OTCLIB_BACKWARD>. The default value
    //	   is <OTCLIB_FORWARD>. In the <OTC_Map> class, traversal in
    //	   the forward direction will result in the first key/item being
    //	   accessible being the oldest key/item in the set. Moving the
    //	   iterator will result in successively newer key/items in the set
    //	   being accessible.
    //
    //	   When iterating over items, in the order in which they were
    //	   inserted, be very careful about inserting new items into the set.
    //	   This is because any new items inserted, will also be seen by the
    //	   iterator. If you were inserting a new item for every item seen,
    //	   you would end up with an infinite loop.

    OTC_Iterator<T1>	keys(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Returns an iterator over the keys.

    OTC_Iterator<T2>	items(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const
				{ return _items(theDirection,theProtection); }
				// Returns an iterator over the items.

    OTC_Modifier<T2>	items(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			)
				{ return _items(theDirection,theProtection); }
				// Returns an iterator over the items.

    OTC_PairIterator<T1,T2>	pairs(
				 OTC_Direction theDirection=OTCLIB_FORWARD,
				 OTC_Protection theProtection=OTCLIB_SAFE
				) const
				{ return _pairs(theDirection,theProtection); }
				// Returns an iterator over the key/item
				// pairs in this map.

    OTC_PairModifier<T1,T2>	pairs(
				 OTC_Direction theDirection=OTCLIB_FORWARD,
				 OTC_Protection theProtection=OTCLIB_SAFE
				)
				{ return _pairs(theDirection,theProtection); }
				// Returns an iterator over the key/item
				// pairs in this map.

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  private:

    T2&			_item(T1 const& theKey) const;
				// If an item corresponding to <theKey>
				// exists in this map, a reference to
				// it will be returned. If an item
				// corresponding to <theKey> does not exist,
				// an exception will be raised.

    T2*			_pItem(T1 const& theKey) const;
				// If an item corresponding to <theKey>
				// exists in this map, a pointer to
				// it will be returned. If an item
				// corresponding to <theKey> does not exist,
				// a null pointer is returned.

    OTC_Cursor<T2>*	_items(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Returns a cursor over the items.

    OTC_PairCursor<T1,T2>*     _pairs(
				 OTC_Direction theDirection=OTCLIB_FORWARD,
				 OTC_Protection theProtection=OTCLIB_SAFE
				) const;
				// Returns a cursor over the key/item
				// pairs in this map.

    OTC_Boolean		search(
			 T1 const& theKey,
			 int& theIndex,
			 int& theHashValue
			) const;
				// Searches for <theKey> in the hashtable.
				// If the key is in the hashtable index,
				// <OTCLIB_TRUE> is returned and <theIndex>
				// is set to the location in the table where
				// the key can be found. If the key is not
				// in the table, <OTCLIB_FALSE> is returned
				// and <theIndex> is set to a location in
				// the table where the key could be placed.
				// In both cases, <theHashValue> is set to
				// the hash value of the key.

    void		resize(u_int theNewSize);
				// Change the capacity of the map.

    int			rank(T1 const& key1, T1 const& key2) const;
				// Ranks two items.

    OTC_LinkList*	myPairs;
				// List of pairs.

    OTC_TableEntry*	myTable;
				// Key index table.

    u_int		myTableSize;
				// This must be a power of two.

    u_int		myPopulation;
				// Count of the number of keys in the map.

    u_int		myLowThreshold;
				// Numbers of items which if less than
				// will result in the table being
				// reduced.

    u_int		myHighThreshold;
				// Numbers of items which if greater than
				// will result in the table being
				// increased.

    int			(*myRankFn)(T1 const&,T1 const&);
				// Comparison function to be used in
				// place of <OTC_RankActions> for key.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/map.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_MAP_HH */
