#ifndef OTC_COLLCTN_MLSTITER_HH
#define OTC_COLLCTN_MLSTITER_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/mlstiter.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifndef OTC_COLLCTN_MLIST_HH
#include <OTC/collctn/mlist.hh>
#endif

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/mlstiter.hh"
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/mlstiter.hh"
#else
#pragma interface
#endif
#endif
#endif

/* ------------------------------------------------------------------------- */

template<class T>
class OTC_MListIterator
    // = TITLE
    //	   Iterator over an <OTC_MList>. No modification allowed.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   This class is an iterator specifically for the <OTC_MList> class.
    //	   Modification of items through the iterator is not permitted.
    //	   If modification of items held by an instance of <OTC_MList>,
    //	   through an iterator is required, the <OTC_MListModifier> class
    //	   should be used.
    //
    //	   This iterator can be moved both forward and backwards. The
    //	   life of the iterator, cannot extend beyond the life of the
    //	   list being iterated over.
    //	   
    // = SEE ALSO
    //	   <OTC_MListModifier>
{

  protected:

    T&			_item() const
				{
				  OTCLIB_ENSURE((myIter.isLink()),
				   "OTC_MListIterator::item() - "
				   "iterator not on link");
				  return ((OTC_Bucket<T>*)
				   myIter.link())->item();
				}
				// Returns the item under the current
				// location of the iterator. Raises an
				// exception if there is no item under the
				// iterator.

    void		_remove()
				{ myList.myList.remove(myIter.link()); }
				// If the iterator is located on a valid item
				// in the list, the item is removed from the
				// list.

    void		_addBefore(T const& theItem)
				{
				  if (myIter.isStart())
				    myList.myList.addFirst(
				     myList._link(theItem));
				  else if (myIter.isEnd())
				    myList.myList.addLast(
				     myList._link(theItem));
				  else
				    myList.myList.addBeforeLink(
				     myList._link(theItem),
				     myIter.link());
				}
				// Adds <theItem> in the list before the
				// item where the iterator is currently
				// located. If the iterator is at the
				// start of the list, the item will be
				// added as the first item in the list.

    void		_addAfter(T const& theItem)
				{
				  if (myIter.isStart())
				    myList.myList.addFirst(
				     myList._link(theItem));
				  else if (myIter.isEnd())
				    myList.myList.addLast(
				     myList._link(theItem));
				  else
				    myList.myList.addAfterLink(
				     myList._link(theItem),
				     myIter.link());
				}
				// Adds <theItem> in the list after the
				// item where the iterator is currently
				// located. If the iterator is at the
				// end of the list, the item will be
				// added as the last item in the list.

  public:

    // = INITIALISATION

			OTC_MListIterator(OTC_MListIterator<T> const& theIter)
			  : myList(((OTC_MListIterator<T>&)theIter).myList),
			    myIter(theIter.myIter) {}
				// Creates an iterator which is initially
				// located at the same position as <theIter>.
				// The iterator thought will operator
				// independently of <theIter>.

			OTC_MListIterator(
			 OTC_MList<T> const& theList,
			 OTC_Protection theProtection=OTCLIB_SAFE
			)
			  : myList((OTC_MList<T>&)theList),
			    myIter(theList.myList.list()->items(theProtection))
				{}
				// Initialises the iterator. The iterator
				// will be located over the first item in
				// <theList>, or if the list is empty at the
				// start of the list. The argument
				// <theProtection> determines if reference
				// counting will actually be performed on
				// links in the list. The default of
				// <OTCLIB_SAFE> will result in reference
				// counting being performed. A value of
				// <OTCLIB_UNSAFE> will turn off reference
				// counting. In this later case, the list
				// should not be modified while the iterator
				// exists as the state of the iterator will
				// be corrupted.

    // = MOVEMENT

    void		prev()
				{ myIter.prev(); }
				// Moves the iterator to the previous item.

    void		next()
				{ myIter.next(); }
				// Moves the iterator to the next item.

    void		resetFirst()
				{ myIter.resetFirst(); }
				// Resets the iterator to the first item,
				// or if the list is empty, at the start
				// of the list.

    void		resetLast()
				{ myIter.resetLast(); }
				// Resets the iterator to the last item,
				// or if the list is empty, at the end
				// of the list.

    // = RETRIEVAL

    T const&		item() const
				{ return _item(); }
				// Returns the item under the current
				// location of the iterator. Raises an
				// exception if there is no item under the
				// iterator.

    OTC_Boolean		isValid() const
				{ return myIter.isLink(); }
				// Returns <OTCLIB_TRUE> while there is a
				// valid item under the current location
				// of the iterator.

  private:

    OTC_MListIterator<T>&	operator=(OTC_MListIterator<T> const&)
				{ terminate(); return *this; }
				// Do not define an implementation for this.

    OTC_MList<T>&	myList;
				// Reference to the list being iterated
				// over.

    OTC_LinkIterator	myIter;
				// The real iterator.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_MLSTITER_HH */
