#ifndef OTC_COLLCTN_ORDLIST_C
#define OTC_COLLCTN_ORDLIST_C
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/ordlist.c
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/linklist.hh>
#include <OTC/collctn/icursor.hh>
#include <OTC/collctn/avllnode.hh>

#ifdef __OSE_TEMPLATES__
template<class T> OTC_OrderedList
{
  OSE_DECLARE OTC_ItemCursor< T,OTC_Bucket<T> >;
  OSE_IMPLEMENT OTC_Bucket<T>;
  OSE_IMPLEMENT OTC_ItemCursor< T,OTC_Bucket<T> >;
  OSE_IMPLEMENT OTC_Iterator<T>;
  OSE_IMPLEMENT OTC_RankActions<T>;
};
#endif

/* ------------------------------------------------------------------------- */
template<class T>
OTC_OrderedList<T>::~OTC_OrderedList()
{
  delete myTree;
  myList->unReference();
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_OrderedList<T>::OTC_OrderedList(int (*theRankFn)(T const&,T const&))
  : myRankFn(theRankFn)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myTree = new ((os_segment*)theLocality) OTC_AVLTree;
#else
#if defined(ENV_OSTORE)
  myTree = new (theLocality,OTC_AVLTree::get_os_typespec()) OTC_AVLTree;
#else
  myTree = new OTC_AVLTree;
#endif
#endif
  OTCLIB_ASSERT(myTree != 0);

#if defined(ENV_OSTORE_DML)
  myList = new ((os_segment*)theLocality) OTC_LinkList;
#else
#if defined(ENV_OSTORE)
  myList = new (theLocality,OTC_LinkList::get_os_typespec()) OTC_LinkList;
#else
  myList = new OTC_LinkList;
#endif
#endif
  OTCLIB_ASSERT(myList != 0);
  myList->reference();
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_OrderedList<T>::OTC_OrderedList(OTC_OrderedList<T> const& theList)
  : myRankFn(theList.myRankFn)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myTree = new ((os_segment*)theLocality) OTC_AVLTree;
#else
#if defined(ENV_OSTORE)
  myTree = new (theLocality,OTC_AVLTree::get_os_typespec()) OTC_AVLTree;
#else
  myTree = new OTC_AVLTree;
#endif
#endif
  OTCLIB_ASSERT(myTree != 0);

#if defined(ENV_OSTORE_DML)
    myList = new ((os_segment*)theLocality) OTC_LinkList;
#else
#if defined(ENV_OSTORE)
  myList = new (theLocality,OTC_LinkList::get_os_typespec()) OTC_LinkList;
#else
  myList = new OTC_LinkList;
#endif
#endif
  OTCLIB_ASSERT(myList != 0);
  myList->reference();

  OTC_Iterator<T> iter = 0;
  iter = theList.items(OTCLIB_FORWARD,OTCLIB_UNSAFE);
  for (iter.reset(); iter.isValid(); iter.next())
    add(iter.item());
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_OrderedList<T>::removeAll()
{
  myTree->removeAll();
  myList->removeAll();
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_OrderedList<T>& OTC_OrderedList<T>::operator=(
 OTC_OrderedList<T> const& theList
)
{
  if (this != &theList)
  {
    removeAll();

    myRankFn = theList.myRankFn;
    OTC_Iterator<T> iter = 0;
    iter = theList.items(OTCLIB_FORWARD,OTCLIB_UNSAFE);
    for (iter.reset(); iter.isValid(); iter.next())
      add(iter.item());
  }

  return *this;
}

/* ------------------------------------------------------------------------- */
template<class T>
u_int OTC_OrderedList<T>::add(T const& theItem)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

  OTC_Bucket<T>* theBucket;
#if defined(ENV_OSTORE_DML)
  theBucket = new ((os_segment*)theLocality) OTC_Bucket<T>(theItem);
#else
#if defined(ENV_OSTORE)
  theBucket = new (theLocality,OTC_Bucket<T>::get_os_typespec())
   OTC_Bucket<T>(theItem);
#else
  theBucket = new OTC_Bucket<T>(theItem);
#endif
#endif
  OTCLIB_ASSERT(theBucket != 0);

  OTC_AVLLinkNode* theLinkNode;
#if defined(ENV_OSTORE_DML)
  theLinkNode = new ((os_segment*)theLocality) OTC_AVLLinkNode(theBucket);
#else
#if defined(ENV_OSTORE)
  theLinkNode = new (theLocality,OTC_AVLLinkNode::get_os_typespec())
   OTC_AVLLinkNode(theBucket);
#else
  theLinkNode = new OTC_AVLLinkNode(theBucket);
#endif
#endif
  OTCLIB_ASSERT(theLinkNode);

  u_int theIndex = 0;
  OTC_AVLNode* theNode = myTree->root();
  if (theNode == 0)
  {
    myList->addFirst(theBucket);
    myTree->addRoot(theLinkNode);
  }
  else
  {
    u_int fullCount = 0;
    u_int tmpCount = theNode->count();
    OTC_AVLNode* theParent = theNode;
    while (theNode != 0)
    {
      T& tmpItem = ((OTC_Bucket<T>*)
       ((OTC_AVLLinkNode*)theNode)->link())->item();
      int theRank = rank(theItem,tmpItem);
      theParent = theNode;
      if (theRank < 0)
      {
	theNode = theNode->left();
	if (theNode == 0)
	{
	  ((OTC_AVLLinkNode*)theParent)->link()->addBefore(theBucket);
	  theParent->addBefore(theLinkNode);
	  break;
	}
	else
	  tmpCount = theNode->count();
      }
      else
      {
	theNode = theNode->right();
	if (theNode == 0)
	{
	  ((OTC_AVLLinkNode*)theParent)->link()->addAfter(theBucket);
	  theParent->addAfter(theLinkNode);
	  tmpCount += 1;
	  break;
	}
	else
	{
	  fullCount += tmpCount;
	  tmpCount = theNode->count();
	}
      }
    }
    theIndex = fullCount + tmpCount - 1;
  }

  return theIndex;
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_OrderedList<T>::removeFirst()
{
  OTCLIB_ENSURE((!isEmpty()),
   "OTC_OrderedList::removeFirst() - Collection empty");

  OTC_AVLNode* theNode = myTree->root();
  OTC_AVLNode* theParent = 0;
  while (theNode != 0)
  {
    theParent = theNode;
    theNode = theNode->left();
  }
  theParent->unlink();
  ((OTC_AVLLinkNode*)theParent)->link()->kill();
  delete theParent;
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_OrderedList<T>::removeLast()
{
  OTCLIB_ENSURE((!isEmpty()),
   "OTC_OrderedList::removeLast() - Collection empty");

  OTC_AVLNode* theNode = myTree->root();
  OTC_AVLNode* theParent = 0;
  while (theNode != 0)
  {
    theParent = theNode;
    theNode = theNode->right();
  }
  theParent->unlink();
  ((OTC_AVLLinkNode*)theParent)->link()->kill();
  delete theParent;
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_OrderedList<T>::remove(T const& theItem)
{
  OTCLIB_ENSURE((!isEmpty()),
   "OTC_OrderedList::remove() - Collection empty");

  OTC_AVLNode* theNode = myTree->root();
  while (theNode != 0)
  {
    T& tmpItem = ((OTC_Bucket<T>*)
     ((OTC_AVLLinkNode*)theNode)->link())->item();
    int theRank = rank(theItem,tmpItem);
    if (theRank < 0)
      theNode = theNode->left();
    else if (theRank > 0)
      theNode = theNode->right();
    else
    {
      theNode->unlink();
      ((OTC_AVLLinkNode*)theNode)->link()->kill();
      delete theNode;
      return;
    }
  }

  OTCLIB_ENSURE((theNode != 0),
   "OTC_OrderedList::remove() - No such item");
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_OrderedList<T>::removeItem(u_int theIndex)
{
  OTCLIB_ENSURE((!isEmpty()),
   "OTC_OrderedList::removeItem() - Collection empty");
  OTCLIB_ENSURE((theIndex < myTree->population()),
   "OTC_OrderedList::removeItem() - Index out of range");

  OTC_AVLNode* theNode = myTree->node(theIndex);
  OTCLIB_ASSERT(theNode != 0);
  theNode->unlink();
  ((OTC_AVLLinkNode*)theNode)->link()->kill();
  delete theNode;
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_OrderedList<T>::removeAll(T const& theItem)
{
  removeRange(range(theItem));
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_OrderedList<T>::_removeRange(int theStart, u_int theLength)
{
  if (theLength == 0)
    return;

  OTCLIB_ENSURE((theStart >= 0 && theStart+theLength <= myTree->population()),
    "OTC_OrderedList::removeRange() - Invalid range");

  while (theLength-- > 0)
    removeItem(theStart+theLength);
}

/* ------------------------------------------------------------------------- */
template<class T>
int OTC_OrderedList<T>::index(
 T const& theItem,
 OTC_Occurrence theOccurrence,
 OTC_Placement thePlacement
) const
{
  // Always fail for an empty list.

  if (isEmpty())
    return -1;

  // Start the search.

  OTC_AVLNode* theNode = myTree->root();
  u_int fullCount = 0;
  u_int tmpCount = theNode->count();
  while (theNode != 0)
  {
    T& tmpItem1 = ((OTC_Bucket<T>*)
     ((OTC_AVLLinkNode*)theNode)->link())->item();
    int theRank = rank(theItem,tmpItem1);
    if (theRank < 0)
    {
      u_int theIndex = fullCount + tmpCount - 1;

      // ANY && (GT || GE) - Item being searched is before this item in
      // the list, therefore, if after any item in case of (GT || GE),
      // the current item will do.

      if (theOccurrence == OTCLIB_ANY
       && (thePlacement == OTCLIB_GE || thePlacement == OTCLIB_GT))
      {
	return theIndex;
      }

      if (theNode->left() == 0 && thePlacement != OTCLIB_EQ)
      {
	// LAST && (GE || GT) - If we have got to the last item in
	// the list, we return -1, otherwise we return the last item
	// in the list.

	if (theOccurrence == OTCLIB_LAST
	  && (thePlacement == OTCLIB_GE || thePlacement == OTCLIB_GT))
	{
	  if (theIndex == myTree->population()-1)
	    return -1;

	  return myTree->population()-1;
	}

	// FIRST && (GE || GT) - Return the current item.

	if (theOccurrence == OTCLIB_FIRST
	 && (thePlacement == OTCLIB_GE || thePlacement == OTCLIB_GT))
	{
	  return theIndex;
	}

	// (ANY || LAST) && (LT || LE)  - If at first item in the list,
	// return -1, else return item preceeding this item.

	if ((theOccurrence == OTCLIB_ANY || theOccurrence == OTCLIB_LAST)
	 && (thePlacement == OTCLIB_LT || thePlacement == OTCLIB_LE))
	{
	  if (theIndex == 0)
	    return -1;

	  return theIndex-1;
	}

	// FIRST && (LT || LE) - If at first item in the list, return -1,
	// else return 0.

	if (theOccurrence == OTCLIB_FIRST
	 && (thePlacement == OTCLIB_LT || thePlacement == OTCLIB_LE))
	{
	  if (theIndex == 0)
	    return -1;

	  return 0;
	}
      }

      theNode = theNode->left();
      if (theNode != 0)
	tmpCount = theNode->count();
    }
    else if (theRank > 0)
    {
      u_int theIndex = fullCount + tmpCount - 1;

      // ANY && (LT || LE) - Item being searched is after this item in
      // the list, therefore, if after any item in case of (LT || LE),
      // the current item will do.

      if (theOccurrence == OTCLIB_ANY
       && (thePlacement == OTCLIB_LE || thePlacement == OTCLIB_LT))
      {
	return theIndex;
      }

      if (theNode->right() == 0 && thePlacement != OTCLIB_EQ)
      {
	// FIRST && (LT || LE) - If we have got to the first item in
	// the list, we return -1, otherwise we return first item in the
	// list.

	if (theOccurrence == OTCLIB_FIRST
	  && (thePlacement == OTCLIB_LT || thePlacement == OTCLIB_LE))
	{
	  if (theIndex == 0)
	    return -1;

	  return 0;
	}

	// LAST && (LT && LE) - Return the current item.

	if (theOccurrence == OTCLIB_LAST
	 && (thePlacement == OTCLIB_LT || thePlacement == OTCLIB_LE))
	{
	  return theIndex;
	}

	// (FIRST || ANY) && (GE || GT) - If at the last item in the
	// list return -1, else return suceeding item.

	if ((theOccurrence == OTCLIB_FIRST || theOccurrence == OTCLIB_ANY)
	 && (thePlacement == OTCLIB_GE || thePlacement == OTCLIB_GT))
	{
	  if (theIndex == myTree->population()-1)
	    return -1;

	  return theIndex+1;
	}

	// LAST && (GE || GT) - If at last item in the list return -1,
	// else return last item in the list.

	if (theOccurrence == OTCLIB_LAST
	 && (thePlacement == OTCLIB_GE || thePlacement == OTCLIB_GT))
	{
	  if (theIndex == myTree->population()-1)
	    return -1;

	  return myTree->population()-1;
	}
      }

      theNode = theNode->right();
      if (theNode != 0)
      {
	fullCount += tmpCount;
	tmpCount = theNode->count();
      }
    }
    else
    {
      u_int theIndex = fullCount + tmpCount - 1;

      // ANY && (LE || EQ || GE) - Found item, this will do.

      if (theOccurrence == OTCLIB_ANY && (thePlacement == OTCLIB_LE
       || thePlacement == OTCLIB_EQ || thePlacement == OTCLIB_GE))
      {
	return theIndex;
      }

      // FIRST && LE - Found item. Can return first item in the list.

      if (theOccurrence == OTCLIB_FIRST && thePlacement == OTCLIB_LE)
	return 0;

      // LAST && GE - Found item. Can return last item in the list.

      if (theOccurrence == OTCLIB_LAST && thePlacement == OTCLIB_GE)
	return myTree->population()-1;

      // ANY && LT - Found item. Change this case to LAST && LT.

      if (theOccurrence == OTCLIB_ANY && thePlacement == OTCLIB_LT)
	theOccurrence = OTCLIB_LAST;

      // ANY && GT - Found item. Change this case to FIRST && GT.
      // return -1, else we return last item in the list.

      if (theOccurrence == OTCLIB_ANY && thePlacement == OTCLIB_GT)
	theOccurrence = OTCLIB_FIRST;

      // LAST && LT / FIRST && GT - We need to invert FIRST/LAST for
      // these cases, but remember we have done it so we can check
      // later.

      OTC_Boolean theInverted = OTCLIB_FALSE;

      if (theOccurrence == OTCLIB_LAST && thePlacement == OTCLIB_LT)
      {
	// Becomes FIRST && LT.

	theInverted = OTCLIB_TRUE;
	theOccurrence = OTCLIB_FIRST;
      }
      else if (theOccurrence == OTCLIB_FIRST && thePlacement == OTCLIB_GT)
      {
	// Becomes LAST && GT.

	theInverted = OTCLIB_TRUE;
	theOccurrence = OTCLIB_LAST;
      }

      // FIRST && EQ - FIRST && LT - LAST (INVERTED to FIRST) && LT

      if (theOccurrence == OTCLIB_FIRST)
      {
	OTC_Linkable* theLink = ((OTC_AVLLinkNode*)theNode)->link();
	theLink = theLink->prev();
	while (theLink != 0)
	{
	  if (theLink->isLink() && !theLink->isDead())
	  {
	    T& tmpItem2 = ((OTC_Bucket<T>*)theLink)->item();
	    if (rank(theItem,tmpItem2) != 0)
	      break;
	    theIndex--;
	  }
	  theLink = theLink->prev();
	}

	// FIRST && LT - If item is the first item in the list, we
	// return -1, else we return first item in the list.

	// LAST && LT - If item is the first item in the list, we
	// return -1, else we return item before current. Note
	// that LAST was inverted to FIRST.

	if (thePlacement == OTCLIB_LT)
	{
	  if (theIndex == 0)
	    return -1;

	  if (theInverted != OTCLIB_FALSE)
	    return theIndex-1;

	  return 0;
	}

	// FIRST && EQ

	return theIndex;
      }

      // LAST && EQ - LAST && GT - FIRST (INVERTED to LAST) && LT

      if (theOccurrence == OTCLIB_LAST)
      {
	OTC_Linkable* theLink = ((OTC_AVLLinkNode*)theNode)->link();
	theLink = theLink->next();
	while (theLink != 0)
	{
	  if (theLink->isLink() && !theLink->isDead())
	  {
	    T& tmpItem3 = ((OTC_Bucket<T>*)theLink)->item();
	    if (rank(theItem,tmpItem3) != 0)
	      break;
	    theIndex++;
	  }
	  theLink = theLink->next();
	}

	// LAST && GT - If item is the last item in the list, we
	// return -1, else we return last item in the list.

	// FIRST && GT - If item is the last item in the list, we
	// return -1, else we return item after current. Note
	// that FIRST was inverted to LAST.

	if (thePlacement == OTCLIB_GT)
	{
	  if (theIndex == myTree->population()-1)
	    return -1;

	  if (theInverted != OTCLIB_FALSE)
	    return theIndex+1;

	  return myTree->population()-1;
	}

	// LAST && EQ

	return theIndex;
      }

      // Shouldn't get here.

      return theIndex;
    }
  }

  return -1;
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Range OTC_OrderedList<T>::range(T const& theItem) const
{
  int theStart = index(theItem,OTCLIB_FIRST);
  if (theStart == -1)
    return 0;

  else
  {
    int theEnd = index(theItem,OTCLIB_LAST);
    OTC_Range range(theStart,theEnd-theStart+1);
    return range;
  }
}

/* ------------------------------------------------------------------------- */
template<class T>
T const& OTC_OrderedList<T>::item(u_int theIndex) const
{
  OTCLIB_ENSURE((theIndex < myTree->population()),
   "OTC_OrderedList::item() - Index out of range");

  if (theIndex == 0)
    return first();

  else if (theIndex == myTree->population()-1)
    return last();

  else
  {
    OTC_AVLNode* theNode = myTree->node(theIndex);
    return ((OTC_Bucket<T>*)((OTC_AVLLinkNode*)theNode)->link())->item();
  }
}

/* ------------------------------------------------------------------------- */
template<class T>
T const& OTC_OrderedList<T>::first() const
{
  OTC_Link* aLink = myList->first();
  OTCLIB_ENSURE((aLink != 0),
   "OTC_OrderedList::first() - Collection empty");
 
  return ((OTC_Bucket<T>*)aLink)->item();
}

/* ------------------------------------------------------------------------- */
template<class T>
T const& OTC_OrderedList<T>::last() const
{
  OTC_Link* aLink = myList->last();
  OTCLIB_ENSURE((aLink != 0),
   "OTC_OrderedList::last() - Collection empty");
 
  return ((OTC_Bucket<T>*)aLink)->item();
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Boolean OTC_OrderedList<T>::contains(T const& theItem) const
{
  OTC_AVLNode* theNode = myTree->root();
  while (theNode != 0)
  {
    T& tmpItem = ((OTC_Bucket<T>*)
     ((OTC_AVLLinkNode*)theNode)->link())->item();
    int theRank = rank(theItem,tmpItem);
    if (theRank < 0)
      theNode = theNode->left();
    else if (theRank > 0)
      theNode = theNode->right();
    else
      return OTCLIB_TRUE;
  }
  return OTCLIB_FALSE;
}

/* ------------------------------------------------------------------------- */
template<class T>
u_int OTC_OrderedList<T>::frequency(T const& theItem) const
{
  OTC_AVLNode* theNode = myTree->root();
  while (theNode != 0)
  {
    T& tmpItem1 = ((OTC_Bucket<T>*)
     ((OTC_AVLLinkNode*)theNode)->link())->item();
    int theRank = rank(theItem,tmpItem1);
    if (theRank < 0)
      theNode = theNode->left();
    else if (theRank > 0)
      theNode = theNode->right();
    else
    {
      u_int theCount = 1;
      OTC_Linkable* theLink = 0;
      OTC_Link* theStartLink = ((OTC_AVLLinkNode*)theNode)->link();
      theLink = theStartLink->prev();
      while (theLink != 0)
      {
	if (theLink->isLink() && !theLink->isDead())
	{
	  T& tmpItem2 = ((OTC_Bucket<T>*)theLink)->item();
	  if (rank(theItem,tmpItem2) != 0)
	    break;
	  theCount++;
	}
	theLink = theLink->prev();
      }
      theLink = theStartLink->next();
      while (theLink != 0)
      {
	if (theLink->isLink() && !theLink->isDead())
	{
	  T& tmpItem3 = ((OTC_Bucket<T>*)theLink)->item();
	  if (rank(theItem,tmpItem3) != 0)
	    break;
	  theCount++;
	}
	theLink = theLink->next();
      }
      return theCount;
    }
  }
  return 0;
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Cursor<T>* OTC_OrderedList<T>::_items(
 OTC_Direction theDirection,
 OTC_Protection theProtection
) const
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
  if (!theLocality.isTransientSegment())
  {
    os_transaction* theTXN = os_transaction::get_current();
    if (theTXN != 0 && theTXN->get_type() == os_transaction::read_only)
      theProtection = OTCLIB_UNSAFE;

    if (theProtection == OTCLIB_UNSAFE)
      theLocality = os_segment::get_transient_segment();
  }
#endif

  OTC_Cursor<T>* theIter;
#if defined(ENV_OSTORE_DML)
  theIter = new ((os_segment*)theLocality) OTC_ItemCursor< T,OTC_Bucket<T> >(
   myList,theDirection,theProtection);
#else
#if defined(ENV_OSTORE)
  theIter = new (theLocality,
   OTC_ItemCursor< T,OTC_Bucket<T> >::get_os_typespec())
   OTC_ItemCursor< T,OTC_Bucket<T> >(myList,theDirection,theProtection);
#else
  theIter = new OTC_ItemCursor< T,OTC_Bucket<T> >(myList,theDirection,
   theProtection);
#endif
#endif
  OTCLIB_ASSERT(theIter != 0);
  return theIter;
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Cursor<T>* OTC_OrderedList<T>::_items(
 int theStart,
 u_int theLength,
 OTC_Direction theDirection,
 OTC_Protection theProtection
) const
{
  if (theLength == 0)
    return 0;

  OTCLIB_ENSURE((theStart >= 0 && theStart+theLength <= myTree->population()),
   "OTC_OrderedList::_items() - Invalid range");

  OTC_Linkable* theStartLink;
  OTC_Linkable* theEndLink;

  theStartLink = ((OTC_AVLLinkNode*)myTree->node(theStart))->link();
  theEndLink = ((OTC_AVLLinkNode*)myTree->node(theStart+theLength-1))->link();

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
  if (!theLocality.isTransientSegment())
  {
    os_transaction* theTXN = os_transaction::get_current();
    if (theTXN != 0 && theTXN->get_type() == os_transaction::read_only)
      theProtection = OTCLIB_UNSAFE;

    if (theProtection == OTCLIB_UNSAFE)
      theLocality = os_segment::get_transient_segment();
  }
#endif

  OTC_Linkable* theStartAnchor = 0;
  OTC_Linkable* theEndAnchor = 0;

  if (theProtection == OTCLIB_SAFE)
  {
#if defined(ENV_OSTORE_DML)
    theStartAnchor = new ((os_segment*)theLocality) OTC_Anchor;
    OTCLIB_ASSERT(theStartAnchor != 0);
    theEndAnchor = new ((os_segment*)theLocality) OTC_Anchor;
    OTCLIB_ASSERT(theEndAnchor != 0);
#else
#if defined(ENV_OSTORE)
    theStartAnchor = new (theLocality,OTC_Anchor::get_os_typespec())
     OTC_Anchor;
    OTCLIB_ASSERT(theStartAnchor != 0);
    theEndAnchor = new (theLocality,OTC_Anchor::get_os_typespec())
     OTC_Anchor;
    OTCLIB_ASSERT(theEndAnchor != 0);
#else
    theStartAnchor = new OTC_Anchor;
    OTCLIB_ASSERT(theStartAnchor != 0);
    theEndAnchor = new OTC_Anchor;
    OTCLIB_ASSERT(theEndAnchor != 0);
#endif
#endif
  }

  if (theStartAnchor != 0)
  {
    theStartLink->addBefore(theStartAnchor);
    theStartLink = theStartAnchor;
  }
  else
  {
    theStartLink = theStartLink->prev();
  }

  if (theEndAnchor != 0)
  {
    theEndLink->addAfter(theEndAnchor);
    theEndLink = theEndAnchor;
  }
  else
  {
    theEndLink = theEndLink->next();
  }

  OTC_Cursor<T>* theIter;
#if defined(ENV_OSTORE_DML)
  theIter = new ((os_segment*)theLocality)
   OTC_ItemCursor< T,OTC_Bucket<T> >(_list(),
   theStartLink,theEndLink,theDirection,theProtection);
#else
#if defined(ENV_OSTORE)
  theIter = new (theLocality,
   OTC_ItemCursor< T,OTC_Bucket<T> >::get_os_typespec())
   OTC_ItemCursor< T,OTC_Bucket<T> >(_list(),theStartLink,theEndLink,
   theDirection,theProtection);
#else
  theIter = new OTC_ItemCursor< T,OTC_Bucket<T> >(_list(),theStartLink,
   theEndLink,theDirection,theProtection);
#endif
#endif
  OTCLIB_ASSERT(theIter != 0);

  if (theStartAnchor != 0)
    theStartAnchor->kill();
  if (theEndAnchor != 0)
    theEndAnchor->kill();

  return theIter;
}

/* ------------------------------------------------------------------------- */
template<class T>
int OTC_OrderedList<T>::rank(T const& item1, T const& item2) const
{
  return (myRankFn != 0) ? (*myRankFn)(item1,item2) :
   OTC_RankActions<T>::rank(item1,item2);
}

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_ORDLIST_C */
