#ifndef OTC_COLLCTN_ORDLIST_HH
#define OTC_COLLCTN_ORDLIST_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/ordlist.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/iterator.hh>
#include <OTC/collctn/prtction.hh>
#include <OTC/collctn/dirction.hh>
#include <OTC/collctn/range.hh>
#include <OTC/collctn/rankactn.hh>
#include <OTC/collctn/avltree.hh>
#include <OTC/collctn/bucket.hh>
#include <OTC/collctn/linklist.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/ordlist.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/ordlist.hh"
#else
#pragma interface
#endif
#endif

#ifdef index
#undef index
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template<class T> OTC_OrderedList
{
  OSE_DECLARE OTC_Bucket<T>;
  OSE_DECLARE OTC_Iterator<T>;
  OSE_DECLARE OTC_RankActions<T>;
};
#endif

enum OTC_Occurrence
{
  OTCLIB_FIRST = -1,
  OTCLIB_ANY = 0,
  OTCLIB_LAST = 1
};

enum OTC_Placement
{
  OTCLIB_LT,
  OTCLIB_LE,
  OTCLIB_EQ,
  OTCLIB_GE,
  OTCLIB_GT
};

template<class T>
class OTC_OrderedList
    // = TITLE
    //	   A collection class based on an ordered list.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   <OTC_OrderedList> implements a list of objects, where the order of
    //	   objects is always maintained. This is achieved by ranking objects,
    //	   so that objects of least rank are located at the head of the list.
    //	   The ordering can be dictated by the user by defining an explicit
    //	   version of <OTC_RankActions>, for the type that the list is
    //	   parameterised over.
    //	   
    // = NOTES
    //	   The <OTC_Bucket> class is used internally to hold items in
    //	   the list. Thus, the <OTC_BaseActions> class may be used to provide
    //	   actions to be performed, when items are inserted or removed from
    //	   the list.
    //
    //	   Duplicates are allowed in the list. However users should be aware,
    //	   that the order in which similarly ranked items are ordered is
    //	   undefined. When adding in a duplicate, it may be added before, or
    //	   after the existing item. In addition, when removing an item using
    //	   an instance of an item, if there is a similarly ranked item found
    //	   before the actual one being used as the lookup item, it will be
    //	   removed. If more deterministic control is required, the
    //	   <OTC_AVLTree> class may be used to implement a new list.
    //	   
    //	   When parameterised over a pointer type, where the rank is based on
    //	   a part of the object and not the value of the pointer, it is
    //	   important that the list be parameterised over a pointer to <const>
    //	   object. If this is not done, it would be possible for someone
    //	   to change the object while it is in the list, thus corrupting the
    //	   ordering within the list.
    //	   
    // = SEE ALSO
    //	   <OTC_Iterator>, <OTC_Bucket>, <OTC_BaseActions>,
    //     <OTC_RankActions>, <OTC_AVLTree>
{
  public:

			~OTC_OrderedList();

    // = INITIALISATION

			OTC_OrderedList(int (*theRankFn)(T const&,T const&)=0);
				// Creates an empty list. <theRankFn> is
				// a rank function to be used instead of
				// <OTC_RankActions>.

			OTC_OrderedList(OTC_OrderedList<T> const& theList);
				// Creates a copy of <theList>.

    // = DESTRUCTION

    void		removeAll();
				// Removes all items in the list.

    // = INSERTION

    OTC_OrderedList<T>& operator=(OTC_OrderedList<T> const& theList);
				// Replaces the contents of this list with
				// the items contained in <theList>.
				// Returns a reference to this list.

    u_int		add(T const& theItem);
				// Inserts <theItem> into the list. Note that
				// duplicates are allowed. If you do not want
				// duplicates, you should check by calling
				// the <contains()> function for an existing
				// item, before adding the item in. The index
				// where the item was added will be
				// returned, with <0> being the first
				// location. Note, that with subsequent
				// additions, this index may become invalid.

    // = QUERY

    int			index(
			 T const& theItem,
			 OTC_Occurrence theOccurrence=OTCLIB_ANY,
			 OTC_Placement thePlacement=OTCLIB_EQ
			) const;
				// Returns the index of <theItem> within the
				// list. If there are multiple occurrences of
				// <theItem>, the index of which item is
				// returned, is determined by <theOccurrence>,
				// which can be either <OTCLIB_ANY>,
				// <OTCLIB_FIRST> or <OTCLIB_LAST>, indicating
				// any item, the item with lowest index, or
				// the item with highest index respectively.
				// If <theItem> doesn't exist, then <-1> is
				// returned. <thePlacement> modifies what
				// is being searched for. If <thePlacement>
				// is <OTCLIB_EQ> the search will be as
				// described above. If <OTCLIB_LT>, the above
				// search will apply to the range of items
				// which are less than <theItem>. Ie.,
				// <OTCLIB_ANY> will result in any item which
				// is less than <theItem> being returned.
				// <OTCLIB_FIRST> will result in the first
				// item in the range of items less than
				// <theItem> and <OTCLIB_LAST> the last item
				// in the range of items less than <theItem>.
				// Other values for <thePlacement> are
				// <OTCLIB_GT>, <OTCLIB_LE> and <OTCLIB_GE>.
				// For these values the search is similarly
				// modified so as to apply to the items
				// greater than <theItem>, the items less
				// than and including <theItem>, and the
				// items greater than and including <theItem>.

    OTC_Range		range(T const& theItem) const;
				// Returns the range of indices occupied
				// by <theItem>.

    T const&		item(u_int theIndex) const;
				// Returns a reference to the item in the
				// list at the location <theIndex>. An
				// exception is raised if the index is
				// invalid.

    T const&		first() const;
				// Returns a reference to the first item in
				// the list. If the list is empty, an
				// exception is raised.

    T const&		last() const;
				// Returns a reference to the last item in
				// the list. If the list is empty, an
				// exception is raised.

    T const*		pItem(u_int theIndex) const
				{
				  return theIndex >= myTree->population()
				   ? 0 : &item(theIndex);
				}
				// Returns a pointer to the item in the
				// list at the location <theIndex>. A null
				// pointer is returned if the list is empty
				// or the index is out of range.

    T const*		pFirst() const
				{
				  return myTree->isEmpty() ? 0 :
				   &((OTC_Bucket<T>*)myList->first())->item();
				}
				// Returns a reference to the first item in
				// the list. If the list is empty, a null
				// pointer is returned.

    T const*		pLast() const
				{
				  return myTree->isEmpty() ? 0 :
				   &((OTC_Bucket<T>*)myList->last())->item();
				}
				// Returns a reference to the last item in
				// the list. If the list is empty, a null
				// pointer is returned.

    OTC_Boolean		contains(T const& theItem) const;
				// Returns <OTCLIB_TRUE> if <theItem> exists.

    u_int		frequency(T const& theItem) const;
				// Returns the number of times that <theItem>
				// appears in the list.

    u_int		population() const
				{ return myTree->population(); }
				// Returns the number of items in the list.

    OTC_Boolean		isEmpty() const
				{ return myTree->population() == 0; }
				// Returns <OTCLIB_TRUE> if the collection is
				// empty.

    // = REMOVAL

    void		removeAll(T const& theItem);
				// Removes all items equivalent to <theItem>.

    void		removeFirst();
				// Removes the first item in this list.
				// If the list is empty, an exception is
				// raised.

    void		removeLast();
				// Removes the last item in this list.
				// If the list is empty, an exception is
				// raised.

    void		remove(T const& theItem);
				// Remove first equivalent item to <theItem>
				// found.

    void		removeItem(u_int theIndex);
				// Remove the item at the location
				// <theIndex>. Raises an exception if
				// <theIndex> is invalid.

    void		removeRange(OTC_Range const& theRange)
				{
				  _removeRange(
				   theRange.lower(),
				   theRange.length()
				  );
				}
				// Removes items at locations <theStart>
				// through <theEnd>. If the list is
				// empty, or any of the items do not
				// exist, an exception is raised.

    void		removeRange(u_int theStart, u_int theLength)
				{ _removeRange(theStart,theLength); }
				// Removes items in the range with length
				// <theLength> starting at index <theStart>.
				// If the list is empty, or any of the items
				// do not exist, an exception is raised.

    // = ITERATION
    //     By default, iterators will perform reference counts on the
    //     buckets in the collection as the iterator moves over the items.
    //     Performing the reference counts ensures that the iterator
    //     is not corrupted by additions or removals to the collection.
    //     If an unsafe iterator is required, for example, to avoid
    //     grabbing a write lock when using ObjectStore, a second
    //     argument can be passed to the following functions. The value
    //     of this argument is either <OTCLIB_SAFE> or <OTCLIB_UNSAFE>.
    //     To get an unsafe iterator, the <OTCLIB_UNSAFE> argument should
    //     be used.
    //
    //     The first argument to the following functions indicates the
    //     direction of traversal of the iterator. Traversal in the
    //     direction of first to last item is indicated by a value of
    //     <OTCLIB_FORWARD>. Traversal in the reverse direction is
    //     indicated by a value of <OTCLIB_BACkWARD>. The default value
    //     is <OTCLIB_FORWARD>. In the <OTC_OrderedList> class, traversal
    //     in the forward direction will result in the first item being
    //     available being the least ranked item.

    OTC_Iterator<T>	items(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const
				{ return _items(theDirection,theProtection); }
				// Returns an iterator over the list.

    OTC_Iterator<T>	items(
			 OTC_Range const& theRange,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const
				{
				  return _items(
				   theRange.lower(),theRange.length(),
				   theDirection,theProtection
				  );
				}
				// Returns an iterator over a range of items
				// in the list.

    OTC_Iterator<T>	items(
			 u_int theStart,
			 u_int theLength,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const
				{
				  return _items(
				   theStart,theLength,
				   theDirection,theProtection
				  );
				}
				// Returns an iterator over a range of items
				// in the list.

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  protected:

    void		_removeRange(int theStart, u_int theLength);
				// Removes items in the range with length
				// <theLength>, starting at index <theStart>.
				// If the list is empty, or any of the items
				// do not exist, an exception is raised.

    OTC_Cursor<T>*	_items(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Returns an iterator over the list.

    OTC_Cursor<T>*	_items(
			 int theStart,
			 u_int theLength,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Returns an iterator over a range of items
				// in the list.

    OTC_AVLTree*	_tree() const
				{ return myTree; }
				// Returns the tree being used as index.

    OTC_LinkList*	_list() const
				{ return myList; }
				// Returns the list used to store items.

  private:

    int			rank(T const& item1, T const& item2) const;
				// Ranks two items.

    OTC_AVLTree*	myTree;
				// The tree used to build the search tree
				// on the list.

    OTC_LinkList*	myList;
				// The actual list of items.

    int			(*myRankFn)(T const&,T const&);
				// Comparison function to be used in
				// place of <OTC_RankActions>.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/ordlist.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_ORDLIST_HH */
