#ifndef OTC_COLLCTN_PRIQUEUE_C
#define OTC_COLLCTN_PRIQUEUE_C
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/priqueue.c
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 1993 OTC LIMITED
//     Copyright 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/avltree.hh>
#include <OTC/collctn/linklist.hh>
#include <OTC/collctn/avllnode.hh>
#include <OTC/collctn/holder.hh>

#ifdef __OSE_TEMPLATES__
template<class T> OTC_PriorityQueue
{
  OSE_DECLARE OTC_Holder<T>;
  OSE_IMPLEMENT OTC_Holder<T>;
  OSE_IMPLEMENT OTC_RankActions<T>;
};
#endif

/* ------------------------------------------------------------------------- */
template<class T>
OTC_PriorityQueue<T>::~OTC_PriorityQueue()
{
  delete myTree;
  myList->unReference();
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_PriorityQueue<T>::OTC_PriorityQueue(OTC_QueueType theType)
  : myType(theType), myRankFn(0)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myTree = new ((os_segment*)theLocality) OTC_AVLTree;
#else
#if defined(ENV_OSTORE)
  myTree = new (theLocality,OTC_AVLTree::get_os_typespec()) OTC_AVLTree;
#else
  myTree = new OTC_AVLTree;
#endif
#endif
  OTCLIB_ASSERT(myTree != 0);

#if defined(ENV_OSTORE_DML)
  myList = new ((os_segment*)theLocality) OTC_LinkList;
#else
#if defined(ENV_OSTORE)
  myList = new (theLocality,OTC_LinkList::get_os_typespec()) OTC_LinkList;
#else
  myList = new OTC_LinkList;
#endif
#endif
  OTCLIB_ASSERT(myList != 0);
  myList->reference();
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_PriorityQueue<T>::OTC_PriorityQueue(
 int (*theRankFn)(T const&,T const&),
 OTC_QueueType theType
)
  : myType(theType), myRankFn(theRankFn)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myTree = new ((os_segment*)theLocality) OTC_AVLTree;
#else
#if defined(ENV_OSTORE)
  myTree = new (theLocality,OTC_AVLTree::get_os_typespec()) OTC_AVLTree;
#else
  myTree = new OTC_AVLTree;
#endif
#endif
  OTCLIB_ASSERT(myTree != 0);

#if defined(ENV_OSTORE_DML)
  myList = new ((os_segment*)theLocality) OTC_LinkList;
#else
#if defined(ENV_OSTORE)
  myList = new (theLocality,OTC_LinkList::get_os_typespec()) OTC_LinkList;
#else
  myList = new OTC_LinkList;
#endif
#endif
  OTCLIB_ASSERT(myList != 0);
  myList->reference();
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_PriorityQueue<T>::OTC_PriorityQueue(OTC_PriorityQueue<T> const& theQueue)
  : myType(theQueue.myType), myRankFn(theQueue.myRankFn)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myTree = new ((os_segment*)theLocality) OTC_AVLTree;
#else
#if defined(ENV_OSTORE)
  myTree = new (theLocality,OTC_AVLTree::get_os_typespec()) OTC_AVLTree;
#else
  myTree = new OTC_AVLTree;
#endif
#endif
  OTCLIB_ASSERT(myTree != 0);

#if defined(ENV_OSTORE_DML)
  myList = new ((os_segment*)theLocality) OTC_LinkList;
#else
#if defined(ENV_OSTORE)
  myList = new (theLocality,OTC_LinkList::get_os_typespec()) OTC_LinkList;
#else
  myList = new OTC_LinkList;
#endif
#endif
  OTCLIB_ASSERT(myList != 0);
  myList->reference();

  OTC_LinkIterator iter = theQueue.myList->items(OTCLIB_UNSAFE);
  for (iter.resetFirst(); iter.isLink(); iter.next())
    add(((OTC_Holder<T>*)iter.link())->item());
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Boolean OTC_PriorityQueue<T>::isEmpty() const
{
  return myTree->isEmpty();
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_PriorityQueue<T>::add(T const& theItem)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

  OTC_Holder<T>* theHolder;
#if defined(ENV_OSTORE_DML)
  theHolder = new ((os_segment*)theLocality) OTC_Holder<T>(theItem);
#else
#if defined(ENV_OSTORE)
  theHolder = new (theLocality,OTC_Holder<T>::get_os_typespec())
   OTC_Holder<T>(theItem);
#else
  theHolder = new OTC_Holder<T>(theItem);
#endif
#endif
  OTCLIB_ASSERT(theHolder != 0);

  OTC_AVLLinkNode* theLinkNode;
#if defined(ENV_OSTORE_DML)
  theLinkNode = new ((os_segment*)theLocality) OTC_AVLLinkNode(theHolder);
#else
#if defined(ENV_OSTORE)
  theLinkNode = new (theLocality,OTC_AVLLinkNode::get_os_typespec())
   OTC_AVLLinkNode(theHolder);
#else
  theLinkNode = new OTC_AVLLinkNode(theHolder);
#endif
#endif
  OTCLIB_ASSERT(theLinkNode);

  OTC_AVLNode* theNode = myTree->root();
  if (theNode == 0)
  {
    myList->addFirst(theHolder);
    myTree->addRoot(theLinkNode);
  }
  else
  {
    OTC_AVLNode* theParent = theNode;
    while (theNode != 0)
    {
      T& anItem = ((OTC_Holder<T>*)((OTC_AVLLinkNode*)theNode)->link())->item();
      int theRank = rank(theItem,anItem);
      theParent = theNode;
      if (theRank < 0)
      {
        theNode = theNode->left();
        if (theNode == 0)
        {
          ((OTC_AVLLinkNode*)theParent)->link()->addBefore(theHolder);
          theParent->addBefore(theLinkNode);
          break;
        }
      }
      else
      {
        theNode = theNode->right();
        if (theNode == 0)
        {
          ((OTC_AVLLinkNode*)theParent)->link()->addAfter(theHolder);
          theParent->addAfter(theLinkNode);
          break;
        }
      }
    }
  }
}

/* ------------------------------------------------------------------------- */
template<class T>
T OTC_PriorityQueue<T>::remove()
{
  OTCLIB_ENSURE((!isEmpty()),
   "OTC_PriorityQueue::remove() - Queue empty");

  OTC_AVLNode* theNode = myTree->root();
  OTC_AVLNode* theParent = 0;
  while (theNode != 0)
  {
    theParent = theNode;
    if (myType == OTCLIB_ASCENDING)
      theNode = theNode->left();
    else
      theNode = theNode->right();
  }
  theParent->unlink();
  OTC_AVLLinkNode* theLinkNode = (OTC_AVLLinkNode*)theParent;
  T anItem = ((OTC_Holder<T>*)theLinkNode->link())->item();
  theLinkNode->link()->kill();
  delete theParent;
  return anItem;
}

/* ------------------------------------------------------------------------- */
template<class T>
T const& OTC_PriorityQueue<T>::head() const
{
  OTCLIB_ENSURE((!isEmpty()),
   "OTC_PriorityQueue::head() - Queue empty");

  OTC_Holder<T>* aNode = 0;
  if (myType == OTCLIB_ASCENDING)
    aNode = (OTC_Holder<T>*)myList->first();
  else
    aNode = (OTC_Holder<T>*)myList->last();

  return aNode->item();
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_PriorityQueue<T>::clear()
{
  myTree->removeAll();
  myList->removeAll();
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_PriorityQueue<T>::discard(u_int theCount)
{
  OTCLIB_ENSURE((theCount <= myTree->population()),
   "OTC_PriorityQueue::discard() - Insufficient items");

  while (theCount > 0)
  {
    OTC_AVLNode* theNode = myTree->root();
    OTC_AVLNode* theParent = 0;
    while (theNode != 0)
    {
      theParent = theNode;
      if (myType == OTCLIB_ASCENDING)
	theNode = theNode->left();
      else
	theNode = theNode->right();
    }
    theParent->unlink();
    OTC_AVLLinkNode* theLinkNode = (OTC_AVLLinkNode*)theParent;
    theLinkNode->link()->kill();
    delete theParent;
    theCount--;
  }
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_PriorityQueue<T>& OTC_PriorityQueue<T>::operator=(
 OTC_PriorityQueue<T> const& theQueue
)
{
  OTCLIB_ENSURE((myType == theQueue.myType),
   "OTC_PriorityQueue<T>::operator=() - queues of different types");

  if (&theQueue == this)
    return *this;

  clear();

  OTC_LinkIterator iter = theQueue.myList->items(OTCLIB_UNSAFE);
  for (iter.resetFirst(); iter.isLink(); iter.next())
    add(((OTC_Holder<T>*)iter.link())->item());

  return *this;
}

/* ------------------------------------------------------------------------- */
template<class T>
int OTC_PriorityQueue<T>::rank(T const& item1, T const& item2) const
{
  return (myRankFn != 0) ? (*myRankFn)(item1,item2) :
   OTC_RankActions<T>::rank(item1,item2);
}

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_PRIQUEUE_C */
