#ifndef OTC_COLLCTN_SET_HH
#define OTC_COLLCTN_SET_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     collctn/set.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/iterator.hh>
#include <OTC/collctn/prtction.hh>
#include <OTC/collctn/hashactn.hh>
#include <OTC/collctn/rankactn.hh>
#include <OTC/collctn/dirction.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/set.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/set.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template<class T> OTC_Set
{
  OSE_DECLARE OTC_Iterator<T>;
  OSE_DECLARE OTC_HashActions<T>;
  OSE_DECLARE OTC_RankActions<T>;
};
#endif

class OTC_LinkList;
class OTC_LinkIterator;
class OTC_TableEntry;

template<class T>
class OTC_Set
    // = TITLE
    //	   Class implementing a set of objects.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   <OTC_Set> implements a collection for maintaining a set of objects
    //	   and for determining membership of that set. Once an item is placed
    //	   into the set, it cannot be retrieved. If this behaviour is
    //	   required, one of the map classes should be used.
    //	   
    //	   Commensurate with the properties of a set, no duplicate items are
    //	   allowed. In addition, the items placed into the collection must be
    //	   able to have a hash value generated for them through the use of
    //	   the <OTC_HashActions> class.
    //	   
    // = NOTES
    //	   When a pointer type is used as the item and the hash value
    //	   generated from it is based, not on the pointer, but on some part
    //	   of the object being pointed at, it is important that the pointer
    //	   be of type pointer to const object. You should define the set as
    //	   <OTC_Set\<EX_Foo const*\>> and not just <OTC_Set\<EX_Foo*\>>. If
    //	   this is not done, it would be possible for a user to modify parts
    //	   of the item from which the hash value is generated. If the hash
    //	   value for an object changes, the object will no longer be
    //	   accessible.
    //	   
    //	   The <OTC_Bucket> class is used internally to hold the items. Thus,
    //	   the <OTC_BaseActions> class may be used to provide actions to be
    //	   performed when items are inserted or removed from the set.
    //	   
    //	   Since an attempt to remove an item from the set which is not
    //	   there, will generate an exception, it is important to first check
    //	   that an item is actually in the set, by calling the <contains()>
    //	   function.
    //	   
    // = SEE ALSO
    //	   <OTC_Iterator>, <OTC_Bucket>, <OTC_HashActions>, <OTC_RankActions>
{
  public:

			~OTC_Set();

    // = INITIALISATION

			OTC_Set(int (*theRankFn)(T const&,T const&)=0);
				// Creates an empty set. <theRankFn> is an
				// optional comparison function to be used in
				// placed of <OTC_RankActions>.

			OTC_Set(OTC_Set<T> const& theSet);
				// Creates a copy of <theSet>.

    // = DESTRUCTION

    void		removeAll();
				// Removes all items in this set.

    // = INSERTION

    OTC_Set<T>&		operator=(OTC_Set<T> const& theSet);
				// Replaces the contents of this set with
				// the items contained in <theSet>.
				// Returns a reference to this set.

    void		add(T const& theItem);
				// If <theItem> is not in this set already,
				// it will be added to this set. If <theItem>
				// is in the set already, an exception is
				// raised.

    // = RETRIEVAL/SEARCH

    void		remove(T const& theItem);
				// If <theItem> is in this set, it will be
				// removed and deleted. If <theItem> is not
				// in this set, an exception will be raised.

    OTC_Boolean		contains(T const& theItem) const;
				// Returns <OTCLIB_TRUE> if <theItem> is
				// contained in this set.

    T const&		pickItem() const;
				// Returns an item from the collection. If
				// the set is empty, an exception is raised.
				// The function is not guaranteed to
				// return a different item on each call.

    // = QUERY

    u_int		population() const
				{ return myPopulation; }
				// Returns the number of items in this set.

    OTC_Boolean		isEmpty() const
				{ return myPopulation == 0; }
				// Returns <OTCLIB_TRUE> if the collection is
				// empty.

    // = ITERATION
    //     By default, iterators will perform reference counts on the
    //     buckets in the collection as the iterator moves over the items.
    //     Performing the reference counts ensures that the iterator
    //     is not corrupted by additions or removals to the collection.
    //     If an unsafe iterator is required, for example, to avoid
    //     grabbing a write lock when using ObjectStore, a second
    //     argument can be passed to the following functions. The value
    //     of this argument is either <OTCLIB_SAFE> or <OTCLIB_UNSAFE>.
    //     To get an unsafe iterator, the <OTCLIB_UNSAFE> argument should
    //     be used.
    //
    //     The first argument to the following functions indicates the
    //     direction of traversal of the iterator. Traversal in the
    //     direction of first to last item is indicated by a value of
    //     <OTCLIB_FORWARD>. Traversal in the reverse direction is
    //     indicated by a value of <OTCLIB_BACKWARD>. The default value
    //     is <OTCLIB_FORWARD>. In the <OTC_Set> class, traversal in
    //     the forward direction will result in the first item being
    //     accessible being the oldest item in the set. Moving the
    //     iterator will result in successively newer items in the set
    //     being accessible.
    //
    //	   When iterating over items, in the order in which they were
    //	   inserted, be very careful about inserting new items into the set.
    //	   This is because any new items inserted, may also be seen by the
    //	   iterator. If you were inserting a new item for every item seen,
    //	   you would end up with an infinite loop.

    OTC_Iterator<T>	items(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Returns an iterator over the set.

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  private:

    OTC_Boolean		search(
			 T const& theItem,
			 int& theIndex,
			 int& theHashValue
			) const;
				// Searches for <theItem> in the hashtable.
				// If the item is in the hashtable index,
				// <OTCLIB_TRUE> is returned and <theIndex>
				// is set to the location in the table where
				// the item can be found. If the item is not
				// in the table, <OTCLIB_FALSE> is returned
				// and <theIndex> is set to a location in
				// the table where the item could be placed.
				// In both cases, <theHashValue> is set to
				// the hash value of the item.

    void		resize(u_int theNewSize);
				// Changes the size of the hashtable index.
				// It is assumed that <theNewSize> is
				// sufficient to hold all the items.

    int			rank(T const& item1, T const& item2) const;
				// Ranks two items.

    OTC_LinkList*	myItems;
				// List of items.

    OTC_TableEntry*	myTable;
				// Index into list of items.

    u_int		myTableSize;
				// This must be a power of two.

    u_int		myPopulation;
				// Count of the number of items in the set.

    u_int		myLowThreshold;
				// Number of items which if less than will
				// result in the table size being reduced.

    u_int		myHighThreshold;
				// Number of items which if greater than will
				// result in the table size being increased.

    int			(*myRankFn)(T const&,T const&);
				// Comparison function to be used in
				// place of <OTC_RankActions>.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/set.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_SET_HH */
