#ifndef OTC_COLLCTN_UNIQMAP_HH
#define OTC_COLLCTN_UNIQMAP_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     collctn/uniqmap.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/prtction.hh>
#include <OTC/collctn/iterator.hh>
#include <OTC/collctn/hashactn.hh>
#include <OTC/collctn/rankactn.hh>
#include <OTC/collctn/dirction.hh>
#include <OTC/collctn/pritertr.hh>
#include <OTC/collctn/prvisitr.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/uniqmap.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/uniqmap.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template<class T1, class T2> OTC_UniqMap
{
  OSE_DECLARE OTC_Iterator<T1>;
  OSE_DECLARE OTC_PairIterator<T1,T2>;
  OSE_DECLARE OTC_Iterator<T2>;
  OSE_DECLARE OTC_HashActions<T1>;
  OSE_DECLARE OTC_RankActions<T1>;
  OSE_DECLARE OTC_HashActions<T2>;
  OSE_DECLARE OTC_RankActions<T2>;
  OSE_DECLARE OTC_Visitor<T1>;
  OSE_DECLARE OTC_PairVisitor<T1,T2>;
};
#endif

class OTC_LinkList;
class OTC_LinkIterator;
class OTC_TableEntry;

template<class T1, class T2>
class OTC_UniqMap
    // = TITLE
    //	   Class implementing a one to one mapping between two objects.
    //	   
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   This class implements a one to one mapping between two objects.
    //	   Since the map works both ways, both keys and items must be unique
    //	   within there respective indexes. In addition, both key and item
    //	   must be able to have a hash value generated for them, through
    //	   the use of the <OTC_HashActions> class.
    //
    // = NOTES
    //	   When a pointer type is used as either a key or item, and the hash
    //	   value generated from it, is based not on the pointer, but on some
    //	   part of the object being pointed at, it is important that the
    //	   pointer be of type pointer to const object. You should define the
    //	   map as <OTC_UniqMap\<EX_Foo const*,EX_Bar const*\>> and not just
    //	   <OTC_Map\<EX_Foo*,EX_Bar*\>>. If this is not done, it would be
    //	   possible for a user to modify parts of the key or item from which
    //	   the hash value is generated. If the hash value for an object is
    //	   changed, the key or item will no longer be accessible.
    //	   
    //	   The <OTC_Bucket> class is used internally to hold both key and
    //	   items. Thus the <OTC_BaseActions> class may be used to provide
    //	   actions to be performed when items are inserted or removed from
    //	   the map.
    //
    //	   Since an attempt to access or remove objects which are not in the
    //	   map will raise an exception, it is important first to check that
    //	   an object is actually in the map using the appropriate
    //	   <contains()> function.
    //	   
    // = SEE ALSO
    //	   <OTC_Iterator>, <OTC_Bucket>, <OTC_BaseActions>,
    //     <OTC_HashActions>, OTC_RankActions>
{
  public:

			~OTC_UniqMap();

#if defined(CXX_LC)

    typedef		int (*type1)(T1 const&,T1 const&);
    typedef		int (*type2)(T2 const&,T2 const&);

			OTC_UniqMap(
			 type1 theKeyRankFn=0,
			 type2 theItemRankFn=0
			);
				// Creates an empty map. <theKeyRankFn>
				// and <theItemRankFn> are rank functions,
				// which if provided will override the
				// <OTC_RankActions> class.
#endif

    // = INITIALISATION

#if !defined(CXX_LC)
			OTC_UniqMap(
			 int (*theKeyRankFn)(T1 const&,T1 const&)=0,
			 int (*theItemRankFn)(T2 const&,T2 const&)=0
			);
				// Creates an empty map. <theKeyRankFn>
				// and <theItemRankFn> are rank functions,
				// which if provided will override the
				// <OTC_RankActions> class.
#endif

			OTC_UniqMap(OTC_UniqMap<T1,T2> const& theMap);
				// Creates a copy of <theMap>.

    // = DESTRUCTION

    void		removeAll();
				// Removes all key/item pairs in this map.

    // = INSERTION

    OTC_UniqMap<T1,T2>& operator=(OTC_UniqMap<T1,T2> const& theMap);
				// Replace the contents of this map with
				// a copy of the contents of <theMap>.
				// Returns a reference to this map.

    void		add(T1 const& theKey, T2 const& theItem);
				// If an item corresponding to <theKey> does
				// not exist in this map and <theItem> does
				// not already exist in the map, then
				// <theItem> will be entered into this map.
				// If an item corresponding to <theKey>
				// already exiss, or <theItem> already
				// exists, an exception will be raised.

    // = RETRIEVAL/SEARCH

    void		removeKey(T1 const& theKey);
				// If an item corresponding to <theKey>
				// exists in this map, the key/item pair will
				// be removed. If an item corresponding to
				// <theKey> does not exist, an exception will
				// be raised.

    void		removeItem(T2 const& theItem);
				// If a key corresponding to <theItem> exists
				// in this map, the key/item pair will be
				// removed. If a key corresponding to
				// <theItem> does not exist, an exception
				// will be raised.

    OTC_Boolean		containsKey(T1 const& theKey) const;
				// Returns <OTCLIB_TRUE> if this map contains
				// an item corresponding to <theKey>.

    OTC_Boolean		containsItem(T2 const& theItem) const;
				// Returns <OTCLIB_TRUE> if this map contains
				// a key corresponding to <theItem>.

    T2 const&		item(T1 const& theKey) const;
				// If an item corresponding to <theKey>
				// exists in this map, a reference to
				// the item will be returned. If an item
				// corresponding to <theKey> does not exist,
				// an exception will be raised.

    T2 const*		pItem(T1 const& theKey) const;
				// If an item corresponding to <theKey>
				// exists in this map, a pointer to
				// the item will be returned. If an item
				// corresponding to <theKey> does not exist,
				// a null pointer is returned.

    T1 const&		key(T2 const& theItem) const;
				// If a key corresponding to <theItem>
				// exists in this map, a reference to the key
				// will be returned. If a key corresponding
				// to <theItem> does not exist, an exception
				// will be raised.

    T1 const*		pKey(T2 const& theItem) const;
				// If a key corresponding to <theItem>
				// exists in this map, a pointer to the key
				// will be returned. If a key corresponding
				// to <theItem> does not exist, a null
				// pointer will be returned.

    T1 const&		pickKey() const;
				// Returns a key from the collection. If the
				// map is empty, an exception is raised. The
				// function is not guaranteed to return a
				// different key on each call.

    T2 const&		pickItem() const;
				// Returns a item from the collection. If the
				// map is empty, an exception is raised. The
				// function is not guaranteed to return a
				// different item on each call.

    // = QUERY

    u_int		population() const
				{ return myPopulation; }
				// Returns the number of key/item mappings
				// contained in this map.

    OTC_Boolean		isEmpty() const
				{ return myPopulation == 0; }
				// Returns <OTCLIB_TRUE> if the collection is
				// empty.

    // = ITERATION
    //     By default, iterators will perform reference counts on the
    //     buckets in the collection as the iterator moves over the items.
    //     Performing the reference counts ensures that the iterator
    //     is not corrupted by additions or removals to the collection.
    //     If an unsafe iterator is required, for example, to avoid
    //     grabbing a write lock when using ObjectStore, a second
    //     argument can be passed to the following functions. The value
    //     of this argument is either <OTCLIB_SAFE> or <OTCLIB_UNSAFE>.
    //     To get an unsafe iterator, the <OTCLIB_UNSAFE> argument should
    //     be used.
    //
    //     The first argument to the following functions indicates the
    //     direction of traversal of the iterator. Traversal in the
    //     direction of first to last item is indicated by a value of
    //     <OTCLIB_FORWARD>. Traversal in the reverse direction is
    //     indicated by a value of <OTCLIB_BACKWARD>. The default value
    //     is <OTCLIB_FORWARD>. In the <OTC_Map> class, traversal in
    //     the forward direction will result in the first key/item being
    //     accessible being the oldest key/item in the set. Moving the
    //     iterator will result in successively newer key/items in the set
    //     being accessible.
    //
    //     When iterating over items, in the order in which they were
    //     inserted, be very careful about inserting new items into the set.
    //     This is because any new items inserted, will also be seen by the
    //     iterator. If you were inserting a new item for every item seen,
    //     you would end up with an infinite loop.

    OTC_Iterator<T1>	keys(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Returns an iterator over the keys. The
				// order in which keys in the map are visited
				// is determined by <theDirection>. If
				// <theDirection> is set to <OTCLIB_FORWARD>,
				// keys will be visited in the order in
				// which they were inserted. If
				// <theDirection> is <OTCLIB_BACKWARD>,
				// keys will be visited in the reverse order.
				// The default direction is forward.

    OTC_Iterator<T2>	items(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Returns an iterator over the items. The
				// order in which items in the map are visited
				// is determined by <theDirection>. If
				// <theDirection> is set to <OTCLIB_FORWARD>,
				// items will be visited in the order in
				// which they were inserted. If
				// <theDirection> is <OTCLIB_BACKWARD>,
				// items will be visited in the reverse order.

    OTC_PairIterator<T1,T2>	pairs(
				 OTC_Direction theDirection=OTCLIB_FORWARD,
				 OTC_Protection theProtection=OTCLIB_SAFE
				) const;
				// Returns an iterator over the key/item
				// pairs. The order in which key/item pairs
				// in the map are visited is determined by
				// <theDirection>. If <theDirection> is set
				// to <OTCLIB_FORWARD>, key/item pairs will
				// be visited in the order in which they were
				// inserted. If <theDirection> is
				// <OTCLIB_BACKWARD>, key/item pairs will be
				// visited in the reverse order. The default
				// direction is forward.

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  private:

    OTC_Boolean		searchKey(
			 T1 const& theKey,
			 int& theIndex,
			 int& theHashValue
			) const;
				// Searches for <theKey> in the hashtable.
				// If the key is in the hashtable index,
				// <OTCLIB_TRUE> is returned and <theIndex>
				// is set to the location in the table where
				// the key can be found. If the key is not
				// in the table, <OTCLIB_FALSE> is returned
				// and <theIndex> is set to a location in
				// the table where the key could be placed.
				// In both cases, <theHashValue> is set to
				// the hash value of the key.

    OTC_Boolean		searchItem(
			 T2 const& theItem,
			 int& theIndex,
			 int& theHashValue
			) const;
				// Searches for <theItem> in the hashtable.
				// If the item is in the hashtable index,
				// <OTCLIB_TRUE> is returned and <theIndex>
				// is set to the location in the table where
				// the item can be found. If the item is not
				// in the table, <OTCLIB_FALSE> is returned
				// and <theIndex> is set to a location in
				// the table where the item could be placed.
				// In both cases, <theHashValue> is set to
				// the hash value of the item.

    void		resize(u_int theNewSize);
				// Change the capacity of the map.

    int			rankKey(T1 const& key1, T1 const& key2) const;
				// Rank two keys.

    int			rankItem(T2 const& item1, T2 const& item2) const;
				// Ranks two items.

    OTC_LinkList*	myPairs;
				// List of key/item pairs.

    OTC_TableEntry*	myKeyTable;
				// Key index table.

    OTC_TableEntry*	myItemTable;
				// Item index table.

    u_int		myTableSize;
				// Must be a power of two.

    u_int		myPopulation;
				// Count of the number of keys in the map.

    u_int		myLowThreshold;
				// Number of items which if less than will
				// result in the table size being decreased.

    u_int		myHighThreshold;
				// Number of items which if greater than will
				// result in the table size being increased.

    int			(*myKeyRankFn)(T1 const&,T1 const&);
				// Comparison function to be used in
				// place of <OTC_RankActions> for keys.

    int			(*myItemRankFn)(T2 const&,T2 const&);
				// Comparison function to be used in
				// place of <OTC_RankActions> for items.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/uniqmap.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_UNIQMAP_HH */
