#ifndef OTC_COLLCTN_VECTOR_HH
#define OTC_COLLCTN_VECTOR_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/vector.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/range.hh>
#include <OTC/collctn/copyactn.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/vector.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/vector.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template<class T> OTC_Vector
{
  OSE_DECLARE OTC_CopyActions<T>;
};
#endif

template<class T>
class OTC_Vector
    // = TITLE
    //     Wrapper around a conventional C vector.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     The class <OTC_Vector> is a wrapper around a conventional C style
    //     vector. As with the traditional vector, indexes into the vector
    //     commence at <0> and extend up to one less than the size of the
    //     vector.
    //     
    //     If the vector is to hold a data type possessing a constructor,
    //     the data type must have a constructor which accepts no arguments.
    //     If the data type being held, doesn't automatically initialise
    //     itself through a constructor, it is the user's responsibility
    //     to initialise the vector. For example, if the vector is to hold
    //     pointers, the user will have to initialise each location in
    //     the vector to <0> before use. In addition to having to initialise
    //     the vector when holding pointers, it is also the user's
    //     responsibility to delete each object pointed at, when the vector is
    //     destroyed, if those objects would no longer be referenced.
{
  public:

			~OTC_Vector()
				{ delete [] myVector; }

    // = CLASS INITIALISATION

			OTC_Vector(u_int theSize);
				// Creates a vector of size <theSize>.
				// <theSize> must be greater than zero.

			OTC_Vector(OTC_Vector<T> const& theVector);
				// Creates a vector which is a copy of
				// <theVector>.

			OTC_Vector(
			 OTC_Vector<T> const& theVector,
			 OTC_Range const& theRange
			);
				// Creates a vector which is a copy of the
				// region of <theVector> described by
				// <theRange>. The length of the region
				// described by <theRange> must be greater
				// than zero.

			OTC_Vector(
			 OTC_Vector<T> const& theVector,
			 u_int theStart,
			 u_int theLength
			);
				// Creates a vector which is a copy of the
				// region of <theVector> described by
				// <theStart> and <theLength>. <theLength>
				// must be greater than zero.

			OTC_Vector(T const* theVector, u_int theSize);
				// Creates a vector of size <theSize>
				// which contains a copy of the items
				// in <theVector>. <theSize> must
				// be greater than zero.

    // = ASSIGNMENT

    OTC_Vector<T>&	operator=(OTC_Vector<T> const& theVector);
				// Replaces the contents of this vector with
				// a copy of contents of <theVector>. Except
				// in the case of self assignment, new memory
				// will always be allocated to hold the
				// new data. The old memory held by this
				// vector will be deleted. If the vector holds
				// pointers to objects, the objects being
				// pointed at are not deleted.

    // = ACCESSING

			operator T*()
				{ return myVector; }
				// Conversion operator. Returns a pointer to
				// the underlying buffer. This particular
				// conversion operator will only be used on a
				// non-const instance of this class and will
				// allow modification of the underlying
				// buffer using the pointer returned.

			operator T const*() const
				{ return myVector; }
				// Conversion operator. Returns a pointer
				// to the underlying buffer. This particular
				// conversion operator will always be used
				// for a const instance of this class, or a
				// non-const instance of this class, when the
				// conversion is to a pointer to a vector of
				// const objects. It is not possible to
				// modify the underlying vector using the
				// pointer returned.

    T*			buffer()
				{ return myVector; }
				// Returns a pointer to the underlying
				// buffer. This particular version of
				// <buffer()> will only be used on a
				// non-const instance of this class, and will
				// allow modification of the underlying
				// buffer using the pointer returned.

    T const*		buffer() const
				{ return myVector; }
				// Returns a pointer to the underlying
				// buffer. This particular version of
				// <buffer()> will always be used for a const
				// object, or a non-const instance of this
				// class, when the conversion is to a pointer
				// to a vector of const objects. It is not
				// possible to modify the underlying vector
				// using the pointer returned.

    u_int		size() const
				{ return mySize; }
				// Returns the size of the vector.

    T&			operator[](u_int theIndex)
				{
				  OTCLIB_ENSURE((theIndex < mySize),
				   "OTC_Vector::operator[] - invalid index");
				  return myVector[theIndex];
				}
				// Returns a reference to the object held
				// in the vector at index <theIndex>. This
				// version of <operator[]()> will only be
				// used on a non-const instance of this
				// class. The reference to the object held,
				// will allow modification of the actual
				// object. <theIndex> must be less than the
				// size of the vector.

    T const&		operator[](u_int theIndex) const
				{
				  OTCLIB_ENSURE((theIndex < mySize),
				   "OTC_Vector::operator[] - invalid index");
				  return myVector[theIndex];
				}
				// Returns a reference to the object held
				// in the vector at index <theIndex>. This
				// version of <operator[]()> will only be
				// used on a const instance of this
				// class, or a non-const instance, when
				// a reference to a const object is required.
				// The reference to the object held will not
				// allow modification of the actual object.
				// <theIndex> must be less than the size of
				// the vector.

    // = RESIZING

    void		resize(u_int theSize);
				// Resizes the vector to <theSize>. Unless
				// <theSize> is the same as the current size,
				// a new block of memory is always allocated,
				// even when the size of the vector is being
				// decreased. If the size of the vector is
				// increased, the new section is not
				// explicitly initialised, thus unless the
				// vector holds class objects which perform
				// their own initialisation you will need to
				// initialise it yourself.

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  private:

    u_int		mySize;
				// The size of the vector.

    T*			myVector;
				// The actual buffer space used by the vector.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/vector.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_VECTOR_HH */
