#ifndef OTC_DEBUG_TRACER_HH
#define OTC_DEBUG_TRACER_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     debug/tracer.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifndef OTC_DEBUG_TRCESTRM_HH
#include <OTC/debug/trcestrm.hh>
#endif
#ifndef OTC_THREAD_MUTEX_HH
#include <OTC/thread/mutex.hh>
#endif

#include <iostream.h>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/debug/tracer.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef OTCLIB_TRACE
#define OTCLIB_MARKBLOCK(lvl,function) \
 OTC_Tracer _otclib_mark_block(function,__FILE__,__LINE__,int(lvl)); \
 if (&_otclib_mark_block);
#define OTCLIB_DOTRACE(function) OTCLIB_MARKBLOCK(1,function)
#define OTCLIB_TRACER(lvl) \
  if (!OTC_Tracer::enabled(int(lvl))) ; else otclib_tracer()
#else
#define OTCLIB_MARKBLOCK(lvl,function)
#define OTCLIB_DOTRACE(function)
#define OTCLIB_TRACER(lvl) \
 if (1) ; else cerr
#endif

class OTC_LogStream;

class OTC_Tracer
    // = TITLE
    //     Class for tracing execution path through a function.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     Used to trace execution path through a function, by printing out
    //     messages when an instance of the class is created, and
    //     automatically, when the class is destroyed on function exit.
    //
    //     An example of how the class is used.
    //     
    // = BEGIN<CODE>
    //     main()
    //     {
    //       OTC_Tracer tracer("main()");
    //       tracer() << "Some info." << endl;
    //     }
    // = END<CODE>
    //
    //     This would produce the following output.
    //
    // = BEGIN<INDENT>
    // = BEGIN<NOFILL>
    //     @enter - main()
    //     Some info.
    //     @exit - main()
    // = END<NOFILL>
    // = END<INDENT>
    //
    //     To encourage you to leave debugging statements in your code, a
    //     number of macros have been provided, that achieve the above
    //     result, but that can be conditionally compiled into your code,
    //     only when required. These are the <OTCLIB_DOTRACE> and
    //     <OTCLIB_TRACER> macros and are used in the following manner.
    //     
    // = BEGIN<CODE>
    //     main()
    //     {
    //       OTCLIB_DOTRACE("main()");
    //       OTCLIB_TRACER(1) << "Hello World" << endl;
    //     }
    // = END<CODE>
    //
    //     It is not necessary to use the <OTCLIB_DOTRACE> macro in
    //     a block to be able to use the <OTCLIB_TRACER> macro. The
    //     <OTCLIB_TRACER> can be used in a block by itself.
    //
    //     To compile the debugging statements into your code, the
    //     preprocessor symbol <OTCLIB_TRACE> must be defined, ie.,
    //     <-DOTCLIB_TRACE> must be supplied as an argument to the compiler.
    //     
    //     The argument to the <OTCLIB_TRACER> macro represents a debug
    //     level. If the debug level set in the <OTC_Tracer> class is greater
    //     than the value of the argument, and the argument is positive,
    //     the line will be run. When the <OTCLIB_DOTRACE> macro is used, the
    //     information generated when the <OTC_Tracer> class is created and
    //     destroyed is at level <1>. Instead of the <OTCLIB_DOTRACE> macro,
    //     the <OTCLIB_MARKBLOCK> macro can be used. This macro takes
    //     two arguments. The first argument is a trace level. The second
    //     argument is the string which will be passed to the construtor
    //     of the <OTC_Tracer> class. The <OTCLIB_DOTRACE> macro has the
    //     same affect as the <OTCLIB_MARKBLOCK> macro, using <1> as the
    //     first argument. A level of <0> indicates that the information
    //     should always be displayed. Higher values should be used for
    //     successive levels of verbosity.
    //     
    //     The initial debug level use to to determine what is displayed will
    //     be taken from the environment variable <OTCLIB_TRACELEVEL> or will
    //     be set to <0> if this isn't defined or the value in the variable
    //     was negative, ie., by default, no trace output is displayed. If
    //     you use this feature a lot, ie., you are a code developer, you may
    //     wish to permanently set this environment variable to <1> or
    //     higher.
    //     
    //     By default, the output from the <OTC_Tracer> class is sent to
    //     <stderr> via the <clog> stream. You can divert the trace output to
    //     a file by setting the environment variable <OTCLIB_TRACEFILE> to
    //     be the name of the file. If the file cannot be opened, the output
    //     will still be sent to <stderr>.
    //     
    //     You can also divert the trace output to a particular file
    //     descriptor. If you wish to do this, the environment variable
    //     <OTCLIB_TRACEFD> should be set to the number of the file
    //     descriptor. This can be used to divert the trace output through a
    //     process that filters or highlights information.
    //     
    //     Note that if both <OTCLIB_TRACEFD> and <OTCLIB_TRACEFILE> are
    //     defined, use of the file descriptor will take precedence;
    //     output will not be sent to both places.
    //
    //     If you are using the <OTCLIB_TRACER> and <OTCLIB_DOTRACE> macros,
    //     you can enable the generation of additional trace information
    //     containing file and line information. This is enabled by setting
    //     the environment variable <OTCLIB_TRACEINFO>. The additional
    //     information is output, just before the <@enter> line for a
    //     function. The format of the additional information will be:
    //     
    // = BEGIN<CODE>
    //     @location - "file.cc", line 42
    // = END<CODE>
    //     
    //
    // = NOTES
    //     Use of the <OTC_Tracer> class cannot be made totally thread safe
    //     as thread conflicts can probably occur in the stream returned to
    //     which output is being directed. I am not sure about this. Also,
    //     the feature whereby instances of the <OTC_Tracer> class are
    //     chained together so as to be able to dump out a rudimentary stack
    //     trace, is disabled when threads are available.
    //     
    // = SEE ALSO
    //     <ostream>
{
  public:

    // = CONSTRUCTION

			OTC_Tracer(
			 char const* thePrototype,
			 char const* theFile,
			 u_int theLine,
			 int theEnable=0
			);
				// <thePrototype> should be a string
				// describing the function and its arguments.
				// This string will be printed out with a
				// message indicating that the function has
				// been entered. <theFile> and <theLine>
				// should identify the file and line at which
				// the <OTC_Tracer> class was created. These
				// would normally be passed automatically,
				// by the <OTCLIB_DOTRACE> macro, the symbols
				// <__FILE__> and <__LINE__>. When this
				// constructor is used, the message indicating
				// entry and exit of the function is only
				// displayed if enabled. To enable output,
				// <theEnable> should be <0> or a positive
				// value. To disable output, <theEnable>
				// should be negative.

			OTC_Tracer(char const* thePrototype);
				// <thePrototype> should be a string
				// describing the function and its arguments.
				// This string will be printed out with a
				// message indicating that the function has
				// been entered.

    // = DESTRUCTION

			~OTC_Tracer();
				// Prints out a message to say that the
				// function is being exited.

    // = STREAM ACCESS

    ostream&		operator()() const
				{ return otclib_tracer(); }
				// Returns the stream to which the trace
				// output will be sent.

    // = FUNCTION NAME

    char const*		prototype() const
				{ return myPrototype; }
				// Returns the prototype.

    // = TRACE STATE

    static OTC_Boolean	enabled(int theLevel);
				// Returns <OTCLIB_TRUE> if tracing is
				// enabled for <theLevel> given the state
				// of <OTC_Tracer> as set by functions
				// below.

    // = GLOBAL TRACE
    //     Enabling of a global trace can be done by setting the environment
    //     variable <OTCLIB_GLOBALTRACE> as well as through the following
    //     functions.

    static OTC_Boolean	globalTrace();
				// Returns <OTCLIB_TRUE> if a global
				// tracing is enabled, regardless
				// of other settings.

    static void		enableGlobalTrace();
				// Enables a global trace.

    static void		disableGlobalTrace();
				// Disables a global trace.

    // = TRACE LEVEL
    //     A global trace will override the effect of these settings and will
    //     result in everything being displayed.

    static int		level();
				// Returns the debug level.

    static void		setLevel(int theLevel);
				// Set the debug level.

    // = TRACE INDENT
    //    Trace indenting is by default turned on. You can turn identing
    //    off by setting the environment variable <OTCLIB_NOTRACEINDENT>.

    static OTC_Boolean	traceIndent();
				// Returns <OTCLIB_TRUE> if trace indenting
				// is enabled.

    static void		enableTraceIndent();
				// Enables trace indenting.

    static void		disableTraceIndent();
				// Disables trace indenting.

    // = STACK TRACE

    static OTC_Tracer const*	last();
				// Returns a pointer to the most recently
				// created instance of this class, or <0> if
				// none are currently in existence.

    OTC_Tracer const*	prev() const
				{ return myPrev; }
				// Returns a pointer to the instance
				// of this class created prior to this
				// instance or <0> if there wasn't one.

    // = STAND ALONE TRACE

    friend ostream&	otclib_tracer();
				// Allows you to do away with having either a
				// <OTCLIB_DOTRACE> or <OTCLIB_MARKBLOCK>
				// macro in a function.

    // = STREAMS

    static ostream*	stream();
				// Returns the stream to which trace output
				// is currently being directed. If the
				// trace facility hasn't yet been initialised
				// this will return <0>.

    static void		setStream(ostream* theStream);
				// Set the trace facility to use <theStream>
				// for output. If the trace is being diverted
				// to the logger, this has no affect until
				// such that diversion of trace output to
				// the logger is disabled.

    static OTC_Boolean	logTrace();
				// Returns <OTCLIB_TRUE> if trace output
				// is being diverted to the logger.

    static void		enableLogTrace();
				// Enables diversion of trace output to the
				// logger.

    static void		disableLogTrace();
				// Disables diversion of trace output to
				// the logger.

  private:

			OTC_Tracer(OTC_Tracer const&);
				// Do not define an implementation for this.

   OTC_Tracer&		operator=(OTC_Tracer const&);
				// Do not define an implementation for this.

    static void		initialise();
				// Reads values of various environment
				// variables to set up class.

    char const*		myPrototype;
				// The string describing the function and its
				// arguments.

    OTC_Boolean		myEnable;
				// <OTCLIB_TRUE> if entry and exit tags
				// should be displayed.

    static OTC_Mutex	_mutex;
				// Lock for threads. Use recursive
				// lock as streambuf associated with
				// trace stream can callback to
				// <OTC_Tracer> from inside of the
				// locks held by the constructor and
				// destructor.

    static int		globInit;
				// Indicates if static data has been
				// initialised. Has value <0> if not
				// initialised.

    static int		globLevel;
				// The debug level. Used by <OTCLIB_TRACER>
				// macro to determine if line should be
				// executed or not.

    static OTC_Boolean	globTraceAll;
				// Indicates whether all tracing is turned
				// on regardless of other settings.

    static OTC_Boolean	globInfo;
				// Indicates whether additional line and
				// file information should be output.

    static ostream*	globStream;
				// The stream to which output will be sent.

    static OTC_Boolean	globLogTrace;
				// Set to <OTCLIB_TRUE> if trace is
				// being diverted to the logger.

    static OTC_LogStream*	globLogStream;
				// Instance of log stream if trace output
				// is to be sent there.

    static char*	globLogBuf;
				// Character buffer for log stream.

    static OTC_Boolean	globIndent;
				// Indicates if trace indenting is to occur.

    static OTC_TraceStream*	globTraceStream;
				// Indenting stream class.

    static OTC_Tracer*	globLast;
				// Pointer to last instance of this class
				// created.

    OTC_Tracer*		myPrev;
				// Pointer to instance of this class
				// created prior to this instance.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_DEBUG_TRACER_HH */
