#ifndef OTC_DISPATCH_ALARM_HH
#define OTC_DISPATCH_ALARM_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     dispatch/alarm.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/dispatch/event.hh>
#include <OTC/thread/nrmutex.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/dispatch/alarm.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Job;

class OTC_AlarmSubscription;

class OTCEV_Alarm : public OTC_Event
    // = TITLE
    //     Event object to notify of a clock reaching a particular time.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     <OTCEV_Alarm> is a derived version of <OTC_Event> specifically for
    //     notifying agents that a clock has reached a particular time. The
    //     class also provides the interface for registration of interest in
    //     alarms by agents.
    //     
    // = NOTES
    //     Time is measured as the number of seconds since the epoch. In most
    //     systems the epoch is January 1, 1970.
    //     
    //     A value of <0> will never be used as an alarm ID.
    //
    // = SEE ALSO
    //     <OTC_Event>, <OTC_EVAgent>
{
  public:

			~OTCEV_Alarm();

    // = CONSTRUCTION

			OTCEV_Alarm(int theAlarmId, long theTime)
			  : myAlarmId(theAlarmId), myTime(theTime)
				{}
				// Creates an event object for the alarm with
				// ID number <theAlarmId>. <theTime> is the
				// time at which the alarm was triggered,
				// measured as the number of seconds since
				// the epoch.

    // = QUERY

   int			alarm() const
				{ return myAlarmId; }
				// Returns the ID number of the alarm.

    long		time() const
				{ return myTime; }
				// Returns the time at which the alarm was
				// triggered, measured as the number of
				// seconds since the epoch.

    // = IDENTIFICATION

    void*		type() const;
				// Returns a value that uniquely identifies
				// this type of event.

    static void*	typeId()
				{ return &globTypeId; }
				// Returns a value that uniquely identifies
				// this type of event.

    // = DEBUGGING

    void		dump(ostream& outs) const;
				// Dumps info about the event onto <outs>.

    // = SUBSCRIPTION

    static int		set(int theAgentId, long theTime);
				// Registers that the agent identified by ID
				// <theAgentId> is interested in being
				// notified when time reaches <theTime>
				// seconds since the epoch. An ID is returned
				// which can be used to cancel the alarm.
				// If an alarm is already registered to expire
				// at <theTime>, the new alarm will be expired
				// after the existing alarm.

    static void		cancel(int theAlarmId);
				// Cancels any interest in the alarm
				// with ID <theAlarmId>.

    static void		cancelAgent(int theAgentId);
				// Cancels all alarms that the agent with
				// ID <theAgentId> is interested in.

    static OTC_Boolean	active(int theAlarmId);
				// Returns <OTCLIB_TRUE> if the alarm with ID
				// <theAlarmId> has yet to be triggered.

    // = SCHEDULING

    static long		period();
				// Returns the time in seconds until the
				// next alarm is due to expire.

    static OTC_Job*	pending();
				// Returns a job for the next pending
				// alarm or <0> if no alarms are pending.

  private:

			OTCEV_Alarm(OTCEV_Alarm const&);
				// Do not define an implementation for this.

    OTCEV_Alarm&	operator=(OTCEV_Alarm const&);
				// Do not define an implementation for this.

    static OTC_NRMutex	_mutex;
				// Lock for threads.

    static int		globAlarmIdCount;
				// Counter for allocation of alarm IDs.

    static OTC_AlarmSubscription*	globSubscriptions;
				// List of agents interested in alarms.

    static int		globTypeId;
				// Anchor for use as a type identifier.

    int			myAlarmId;
				// ID of the alarm which was triggered.

    long		myTime;
				// Time, in seconds since the epoch, at
				// which the alarm was triggered
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_DISPATCH_ALARM_HH */
