#ifndef OTC_DISPATCH_DISPATCH_HH
#define OTC_DISPATCH_DISPATCH_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     dispatch/dispatch.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/dispatch/jobqueue.hh>
#include <OTC/thread/nrmutex.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/dispatch/dispatch.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

enum OTC_DispatcherState
{
  OTCLIB_WAITING=0,
  OTCLIB_RUNNING,
  OTCLIB_STOPPED
};

class OTC_Dispatcher
    // = TITLE
    //     Dispatcher of jobs.
    //
    // = CLASS TYPE
    //     Static
    //
    // = DESCRIPTION
    //     <OTC_Dispatcher> is the collection point for jobs to be executed.
    //     To schedule jobs, you should use the <schedule()> function. When
    //     you run the dispatcher by calling <run()>, it will continually
    //     poll the job queue for the next available job, execute the job
    //     and then destroy it. When the job queue returns that there are no
    //     more jobs, the dispatcher will return from <run()>. The dispatcher
    //     will also return from the <run()> routine, if stopped prematurely
    //     as a result of the <stop()> function being called.
    //     
    //     The default job queue adds new jobs to the end of the queue, and
    //     returns jobs for execution from the front of the queue. If you
    //     require a more sophisticated queueing mechanism, you will need
    //     to derive a new queue from <OTC_JobQueue> and pass it to the
    //     dispatcher when you call <initialise()>, before queueing any jobs.
    //     Even if you do not supply your own job queue, you must still call
    //     initialise before and jobs are queued. An example of a more
    //     sophisticated job queue would be one that, when returning the next
    //     job would first check to see if any signals had occurred. If a
    //     signal had occurred, it would return a job for the delivery of an
    //     event corresponding to that signal, rather than returning the job
    //     at the head of the queue. The same queue could also use system
    //     <select()> or <poll()> function, to try and generate new jobs
    //     corresponding to I/O and timer events.
    //
    // = SEE ALSO
    //     <OTC_JobQueue>, <OTC_Job>
{
  public:

    friend class	OTC_DispatcherDummy;

    // = INITIALISATION
    //     One of the following functions must be called before and
    //     jobs are scheduled to be run.

    static void		initialise();
				// Initialises the dispatcher to use a
				// standard job queue.

    static void		initialise(
			 OTC_JobQueue* theJobQueue,
			 void (*theFunc)()=0
			);
				// Initialise the dispatcher but supply your
				// own job queue by supplying <theJobQueue>.
				// It is by defining your own derived version
				// of <OTC_JobQueue> that you can modify the
				// scheduling algorithm. If this function is
				// called more than once, an exceptions is
				// raised. If <theFunc> is supplied, that
				// function will be called prior to each job
				// being executed.

    // = EXECUTION

    static int		dispatch(int theActions=0, int theOptions=0);
				// Provided that the dispatcher hasn't been
				// stopped, a single job is executed. If the
				// dispatcher had already been stopped, a
				// value of <-1> is returned. If there was no
				// job to execute a value of <0> is returned.
				// If there was no job to execute, and
				// <theActions> has a value of <0>, the
				// dispatcher will be placed into the stopped
				// state. If a job was executed, a value of
				// <1> is returned. Except for the value <0>,
				// the meaning of the value of <theActions>
				// is dependent on the type of job queue. A
				// value of <0> will always indicate that all
				// sources should be checked for jobs. The
				// meaning of the second argument,
				// <theOptions> is also dependent on the
				// type of the job queue. It is provided
				// as a means of passing in any auxiliary
				// information needed for it to work out
				// what to run.

    static int		run();
				// Executes any jobs which have been
				// scheduled, until there are no more jobs to
				// run, or the dispatcher is stopped through
				// a call to <stop()>. If the dispatcher
				// runs out of jobs, then <0> is returned.
				// If the dispatcher is stopped, then <-1>
				// is returned.

    static void		stop();
				// Will cause <run()> to return to the
				// caller, even if there are still jobs
				// to be executed.

    static void		reset();
				// If the dispatcher is in the stopped
				// state, resets it to the waiting state.
				// Raises an exception if called when
				// the dispatcher is not in the stopped
				// state.

    static OTC_Boolean	isWaiting();
				// Returns <OTCLIB_TRUE> if the dispatcher
				// is currently waiting to be run, otherwise
				// returns <OTCLIB_FALSE>.

    static OTC_Boolean	isRunning();
				// Returns <OTCLIB_TRUE> if the dispatcher
				// is currently running, otherwise returns
				// <OTCLIB_FALSE>. A value of <OTCLIB_FALSE>
				// is also returned if the dispatcher has
				// been stopped, but the <run()> routine
				// has not yet returned.

    static OTC_Boolean	isStopped();
				// Returns <OTCLIB_TRUE> if the dispatcher is
				// has been stopped, otherwise returns
				// <OTCLIB_FALSE>. It is possible that this
				// will return <OTCLIB_TRUE> while <run()>
				// is still executing as <stop()> may have
				// only been called by the current job and
				// thus <run()> hasn't had an opportunity
				// to return.

    // = SCHEDULING

    static void		schedule(OTC_Job* theJob, int theOptions=0);
				// Add <theJob> to the queue of jobs to
				// be executed. <theOptions> is any
				// special options which may be understood
				// by the particular queueing mechanism
				// in use.

  public:

    // Following is kept public for backwards compatability only.

    static OTC_JobQueue*	queue();
				// Returns a pointer to the job queue.
				// If the dispatcher has not yet been
				// initialised, then <0> is returned.

  private:

			OTC_Dispatcher();
				// Do not define an implementation for this.

			OTC_Dispatcher(OTC_Dispatcher const&);
				// Do not define an implementation for this.

    OTC_Dispatcher&	operator=(OTC_Dispatcher const&);
				// Do not define an implementation for this.

    static OTC_NRMutex	_mutex;
				// Lock for threads.

    static OTC_DispatcherState	globState;
				// Initially set to <OTCLIB_WAITING>. When
				// <run()> is called, or the first time that
				// <dispatch()> is called this is set to
				// <OTCLIB_RUNNING>. If <stop()> is called,
				// this is set to <OTCLIB_STOPPED>. When in
				// the stopped state, calling <reset()> sets
				// this back to <OTCLIB_WAITING>.

    static void		(*globFunc)();
				// Function to be called prior to each
				// job being executed.

    static OTC_JobQueue*	globJobQueue;
				// Queue to which jobs are added and from which
				// they are later obtained for execution.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_DISPATCH_DISPATCH_HH */
