#ifndef OTC_DISPATCH_EVENT_HH
#define OTC_DISPATCH_EVENT_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     dispatch/event.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/memory/cmmnpool.hh>
#include <OTC/thread/nrmutex.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/dispatch/event.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class ostream;
class OTC_EVAgent;
class OTC_Message;

class OTC_Event
    // = TITLE
    //     Base class for any event object.
    //
    // = CLASS TYPE
    //     Abstract
    //
    // = DESCRIPTION
    //     <OTC_Event> is the base class for any event objects, which are
    //     to be delivered to an agent.
    //
    //     Space for derived classes will be allocated from <OTC_CommonPool>.
    //     
    // = SEE ALSO
    //     <OTC_CommonPool>, <OTC_EVAgent>, <OTC_Dispatcher>
{
  public:

#if !defined(ENV_NOCOMMONPOOL)
    void*		operator new(size_t theSize)
				{ return OTC_CommonPool::allocate(theSize); }

    void		operator delete(void* theMem, size_t theSize)
				{ OTC_CommonPool::release(theMem,theSize); }
#endif

  public:

    // = IDENTIFICATION

    virtual void*	type() const = 0;
				// Must be redefined in a derived class
				// to return a <void*> which uniquely
				// identifies the event type. Normally,
				// a pointer to a static member variable
				// of a derived class would be used as the
				// return value.

    // = NOTIFICATION

    void		deliver(OTC_EVAgent* theAgent);
				// Deliver the event to <theAgent>
				// immediately. No attempt should be made to
				// use the event object after calling this
				// function, as the agent probably will
				// have destroyed the event object.

    void		deliver(int theAgentId);
				// Deliver the event to the agent with ID
				// <theAgentId> immediately. No attempt
				// should be made to use the event object
				// after calling this function, as the agent
				// probably will have destroyed the event
				// object.

    void		queue(int theAgentId, int theOptions=0);
				// Creates a job, and schedules the job with
				// the dispatcher for delivery of the event
				// at some later time, to the agent with
				// ID number <theAgentId>. <theOptions>
				// are any special options which may be
				// understood by the queueing mechanism
				// currently being used by the dispatcher.

    // = CLONING

    virtual OTC_Event*	clone();
				// Returns an instance of the event object,
				// which is substitutable for this event
				// object. By default, this function will
				// increment an internal reference count and
				// return a pointer to the same object. If an
				// event contains modifiable data, the derived
				// class should override this function to
				// create an actual copy of the event object
				// and return the copy.

    // = MARHSALLING

    virtual OTC_Message*	deflate() const;
				// Can be redefined in a derived class to
				// convert the event into a serialised form.
				// Default implementation returns <0>
				// indicating that event cannot be converted
				// into a serialised form.

    // = DESTRUCTION

    void		destroy();
				// Destroys the object, returning the space
				// it occupies back to the memory pool.
				// Actually, this function will only destroy
				// the event object if the internal reference
				// count, when decremented, reaches zero.

    // = DEBUGGING

    virtual void	dump(ostream& outs) const = 0;
				// Must be redefined in a derived class to
				// dump to <outs> a representation of the
				// event for debugging purposes. This
				// function will be called to dump out
				// information about an event if it has to be
				// discarded.

    friend ostream&	operator<<(ostream& outs, OTC_Event const& theEvent)
				{ theEvent.dump(outs); return outs; }
				// Dumps a representation of <theEvent>
				// onto the stream <outs>.

  protected:

    // = NON DELIVERY

    virtual void	cancelSource(int theAgentId);
				// By default does nothing. Is called with
				// the ID agent <theAgentId>, when an event
				// is being discarded because the agent does
				// not exist. A derived event should override
				// this function to cancel the subscription
				// which caused the event to occur so that it
				// will not keep reoccuring.

  protected:

			OTC_Event()
			  : myRefCount(1)
				{}

    virtual		~OTC_Event();

  private:

			OTC_Event(OTC_Event const&) {}
				// Do not define an implementation for this.

    OTC_Event&		operator=(OTC_Event const&);
				// Do not define an implementation for this.

    u_int		myRefCount;
				// The number of references to this event.

    static OTC_NRMutex	_mutex;
				// Lock for threads. Used to protect
				// reference counting code.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_DISPATCH_EVENT_HH */
