#ifndef OTC_DISPATCH_IOEVENT_HH
#define OTC_DISPATCH_IOEVENT_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     dispatch/ioevent.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/dispatch/event.hh>
#include <OTC/thread/nrmutex.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/dispatch/ioevent.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Job;

class OTC_IOEventSubscription;

enum OTC_IOEventMask
{
  OTCLIB_POLLIN = 0x0001,
  OTCLIB_POLLPRI = 0x0002,
  OTCLIB_POLLOUT = 0x0004,
  OTCLIB_POLLERR = 0x0008,
  OTCLIB_POLLHUP = 0x0010,
  OTCLIB_POLLNVAL = 0x0020
};

class OTCEV_IOEvent : public OTC_Event
    // = TITLE
    //     Event object to notify of pending conditions on a file descriptor.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     <OTCEV_IOEvent> is a derived version of <OTC_Event> specifically
    //     for notifying agents of pending conditions on a file descriptor.
    //
    // = SEE ALSO
    //     <OTC_Event>, <OTC_EVAgent>
{
  public:

			~OTCEV_IOEvent();

    // = CONSTRUCTION

			OTCEV_IOEvent(int theFd, int theEvents)
			  : myFd(theFd), myEvents(theEvents)
				{}
				// Creates an event object indicating that
				// <theEvents> are pending on file descriptor
				// <theFd>.

    // = QUERY

    int			events() const
				{ return myEvents; }
				// Returns the events pending on the file
				// descriptor.

    int			fd() const
				{ return myFd; }
				// Returns the number of the file descriptor.

    static int		agent(int theFd);
				// Returns the ID of the agent subscribed to
				// <theFd> or <0> if no agent is subscribed
				// to that file descriptor.

    // = IDENTIFICATION

    void*		type() const;
				// Returns a value that uniquely identifies
				// this type of event.

    static void*	typeId()
				{ return &globTypeId; }
				// Returns a value that uniquely identifies
				// this type of event.

    // = DEBUGGING

    void		dump(ostream& outs) const;
				// Dumps info about the event onto <outs>.

    // = SUBSCRIPTION
    //     In the following functions, <theEvents> is a bit mask formed from
    //     a bitwise AND of the individual events that are of interest.
    //     Individual events which you may subscribe to are: input, output
    //     and priority input. These are selected using the variables:
    //     <OTCLIB_POLLIN>, <OTCLIB_POLLOUT> and <OTCLIB_POLLPRI>.

    static void		subscribe(int theAgentId, int theFd, int theEvents);
				// Registers that the agent with ID number
				// <theAgentId>, is interested in being
				// notified, when <theEvents> are pending on
				// the file descriptor <theFd>. If another
				// agent is already subscribed to <theFd>,
				// an exception is raised. If the agent is
				// already subscribed to the file descriptor,
				// the set of events will be added to those
				// already subscribed to.

    static void		unsubscribe(
			 int theAgentId,
			 int theFd,
			 int theEvents=~0
			);
				// Cancels interest by the agent with ID
				// number <theAgentId> in <theEvents> on the
				// file descriptor <theFd>. If <theEvents>
				// is <~0>, it cancels interest in any events
				// on that file descriptor by that agent.

    static void		unsubscribeFd(int theFd);
				// Cancels the subscription of any agent
				// subscribed to <theFd> has.

    static void		unsubscribeAgent(int theAgentId);
				// Cancels all interest that the agent with
				// ID number <theAgentId>, has in any file
				// descriptors.

    // = SCHEDULING

    static OTC_Job*	job(int theFd, int theEvents);
				// Returns a job for delivery to the agent
				// interested in <theFd> of <theEvents>.
				// Returns <0> if there is no subscription
				// to that file descriptor.

    static int		events(int theFd);
				// Returns the set of events on the file
				// descriptor <theFd>, in which any agents
				// are interested.

    static int		maxFd()
				{
				  return globSubscriptions == 0
				   ? -1 : globLimit;
				}
				// Returns the number of the highest numbered
				// file descriptor in which any agent is
				// interested. Returns <-1> if there are no
				// agents interested in any file descriptors.

  protected:

    // = NON DELIVERY

    void		cancelSource(int theAgentId);
				// Unsubscribes the agent with ID
				// <theAgentId> to which this event was to be
				// delivered, from all file descriptor
				// events. This function will be called when
				// the agent to which this event was to be
				// delivered does not exist.

  private:

			OTCEV_IOEvent(OTCEV_IOEvent const&);
				// Do not define an implementation for this.

    OTCEV_IOEvent&	operator=(OTCEV_IOEvent const&);
				// Do not define an implementation for this.

    static int		nfds();
				// Returns the highest numbered file
				// descriptor the operating system can
				// manage.

    static OTC_NRMutex	_mutex;
				// Lock for threads.

    static int		globLimit;
				// Highest numbered file descriptor that an
				// agent is interested in.

    static OTC_IOEventSubscription*	globSubscriptions;
				// List of subscriptions by agents.

    static int		globTypeId;
				// Anchor for use as type idenitifier.

    int			myFd;
				// File descriptor which event has occurred on.

    int			myEvents;
				// Events which occurred on file descriptor.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_DISPATCH_IOEVENT_HH */
