#ifndef OTC_DISPATCH_JOBQUEUE_HH
#define OTC_DISPATCH_JOBQUEUE_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     dispatch/jobqueue.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/dispatch/job.hh>
#include <OTC/collctn/linklist.hh>
#include <OTC/collctn/vlink.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/dispatch/jobqueue.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_JobQueue
    // = TITLE
    //     Queue for holding jobs.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     <OTC_JobQueue> holds jobs on behalf of the dispatcher. The default
    //     implementation of the job queue adds new jobs to the end of the
    //     queue and returns jobs from the front of the queue.
    //     
    //     If you need to redefine the scheduling algorithm, you should
    //     derive your own queue from this class and redefine <next()>.
    //     An example of a derived queue would be one that, before returning
    //     a job from the queue, would check to see if any signals had
    //     occurred that agents were interested in. Signals would be given a
    //     higher priority than jobs already queued, so a new job
    //     corresponding to the signal would be created and returned, instead
    //     of getting the job from the front of the queue. The same queue
    //     could also use the system <select()> or <poll()> function to
    //     generate new jobs corresponding to I/O or timer events.
    //     
    // = SEE ALSO
    //     <OTC_Dispatcher>, <OTC_Job>
{
  public:

    // = DESTRUCTION

    virtual		~OTC_JobQueue();
				// Goes through the job queue, destroying
				// any jobs.

    // = QUEUEING

    OTC_Boolean		isEmpty() const;
				// Returns <OTCLIB_TRUE> if there are
				// no jobs in the queue.

    virtual void	add(OTC_Job* theJob, int theOptions=0);
				// Adds <theJob> to the end of the queue.
				// Generates an exception if <theJob>
				// is <0>. <theOptions> is any options
				// which are understood by a derived queue
				// class.

    virtual OTC_Job*	next(int theActions=0, int theOptions=0);
				// Returns the job at the head of the queue.
				// Can be redefined in a derived class to
				// look for jobs elsewhere besides this
				// queue. The value of <theActions> is
				// dependent on the type of the derived
				// class. A derived class must take <0>
				// as meaning that all sources should be
				// checked for jobs. The value of
				// <theOptions> is also dependent on the type
				// of the derived class. It is provided as a
				// means of passing in any extra information
				// to the job queue needed for it to
				// determine what the next job to run should
				// be.

  private:

    OTC_LinkList	myJobs;
				// List of jobs.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_DISPATCH_JOBQUEUE_HH */
