#ifndef OTC_DISPATCH_TIMEOUT_HH
#define OTC_DISPATCH_TIMEOUT_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     dispatch/timeout.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/dispatch/event.hh>
#include <OTC/thread/nrmutex.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/dispatch/timeout.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Job;

class OTC_TimeoutSubscription;

class OTCEV_Timeout : public OTC_Event
    // = TITLE
    //     Event object to notify that a timer has expired.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     <OTCEV_Timeout> is a derived version of <OTC_Event> specifically
    //     for notifying agents that a timer has expired.
    //
    // = NOTES
    //     A value of <0> will never be used as a timer ID.
    //
    // = SEE ALSO
    //     <OTC_EVAgent>, <OTC_Event>
{
  public:

			~OTCEV_Timeout();

    // = CONSTRUCTION

			OTCEV_Timeout(int theTimerId)
			  : myTimerId(theTimerId)
				{}
				// Creates a timeout event object for
				// notification that the timer with ID number
				// <theTimerId> has expired.

    // = QUERY

    int			timer() const
				{ return myTimerId; }
				// Returns the ID number of the timer which
				// expired.

    // = IDENTIFICATION

    void*		type() const;
				// Returns a value that uniquely identifies
				// this type of event.

    static void*	typeId()
				{ return &globTypeId; }
				// Returns a value that uniquely identifies
				// this type of event.

    // = DEBUGGING

    void		dump(ostream& outs) const;
				// Dumps info about the event onto <outs>.

    // = SUBSCRIPTION

    static int		start(int theAgentId, long thePeriod);
				// Registers that the agent identified by ID
				// number <theAgentId> should be notified
				// when a time period of <thePeriod>
				// milliseconds has expired. A timer ID
				// number is returned. The timer ID number
				// can be used to cancel a timer using
				// <cancel()>. If a timer is already
				// scheduled to go off at the same time as
				// this timer will expire, this timer will
				// be expired after the existing timer.
				// <thePeriod> must be a positive value,
				// or an exception will be raised.

    static void		cancel(int theTimerId);
				// Cancels the timer with ID number
				// <theTimerId>.

    static void		cancelAgent(int theAgentId);
				// Cancels any timers in which the agent
				// with ID number <theAgentId> is interested.

    static OTC_Boolean	active(int theTimerId);
				// Returns <OTCLIB_TRUE> if the timer with
				// ID number <theTimerId> has not yet expired.

    // = SCHEDULING

    static long		period();
				// Returns <-1> if there are no timers
				// active, otherwise the number of
				// milliseconds until the next timer is due
				// to expire.

    static OTC_Job*	pending();
				// Returns a job for the next pending
				// timeout or <0> if no timeouts are
				// pending.

  private:

			OTCEV_Timeout(OTCEV_Timeout const&);
				// Do not define an implementation for this.

    OTCEV_Timeout&	operator=(OTCEV_Timeout const&);
				// Do not define an implementation for this.

    static OTC_NRMutex	_mutex;
				// Lock for threads.

    static int		globTimerIdCount;
				// Counter for allocating timer ID.

    static OTC_TimeoutSubscription*	globSubscriptions;
				// List of subscriptions to timeouts.

    static int		globTypeId;
				// Anchor for use as a type identifier.

    int			myTimerId;
				// The timer which expired.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_DISPATCH_TIMEOUT_HH */
