#ifndef OTC_FILES_PATHNAME_HH
#define OTC_FILES_PATHNAME_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     files/pathname.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1992 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/text/string.hh>
#include <OTC/text/sobject.hh>
#include <OTC/text/record.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/files/pathname.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Pathname : public OTC_SObject
    // = TITLE
    //	   Class for accessing information about a pathname.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   The <OTC_Pathname> class is intended to hold the pathname
    //	   of a file and allow interogation of whether or not the file
    //	   exists and what access the user is allowed.
    //
    // = NOTES
    //	   Trailing and leading whitespace is not removed from the pathname
    //	   by this class. If you need to remove the whitespace then it must
    //	   be done before initialisation of the class. One way of doing
    //	   this would be to use <OTC_String::trim()>.
    //	   
    //	   The functions <isAccessable()>, <isReadable()>, <isWritable()> and
    //	   <isExecutable()> make use of <access(1)>. Thus the results
    //	   generated are based on the real user ID and supplementary
    //	   group ID (including the real group ID) of the executing
    //	   program.
    //
    //     On non UNIX platforms, a <\\> is not interpreted as a directory
    //     separator. A leading drive name will be recognised though.
    //
    //	   An <OTC_BaseActions> and <OTC_HashActions> class are defined for
    //	   <OTC_Pathname>.
    //	   
    // = SEE ALSO
    //	   <access(2)>, <dirname(1)>, <basename(1)>
{
    friend istream&	operator>>(istream& ins, OTC_Pathname& thePath)
				{ ins >> thePath.myPath; return ins; }

  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

			~OTC_Pathname();

    // = INITIALISATION

			OTC_Pathname();
				// Creates an empty pathname.

			OTC_Pathname(char const* thePath);
				// Creates a pathname using <thePath>.

			OTC_Pathname(OTC_String const& thePath);
				// Creates a pathname using <thePath>.

			OTC_Pathname(OTC_Pathname const& thePath);
				// Creates a pathname using <thePath>.

    OTC_Pathname&	operator=(OTC_Pathname const& thePath)
				{ myPath = thePath.myPath; return *this; }
				// Initialises this path to <thePath>.

    OTC_Pathname&	operator=(OTC_String const& thePath)
				{ myPath = thePath; return *this; }
				// Initialises this path to <thePath>.

    // = QUERY

    OTC_String const&	path() const
				{ return myPath; }
				// Returns the pathname.

			operator char const*() const
				{ return path(); }
				// Returns the pathname.

    OTC_Boolean		isAccessable() const;
				// Returns <OTCLIB_TRUE> if the file exists
				// and all directories in the path are
				// searchable. In the case of <OTCLIB_FALSE>
				// being returned, if <errno> is <ENOENT>
				// then the file truly doesn't exist. If
				// <errno> has some other value then a
				// problem was encountered in accessing the
				// file.

    OTC_Boolean		isWritable() const;
				// Returns <OTCLIB_TRUE> if the file is
				// writable.

    OTC_Boolean		isReadable() const;
				// Returns <OTCLIB_TRUE> if the file is
				// readable.

    OTC_Boolean		isExecutable() const;
				// Returns <OTCLIB_TRUE> if the file is
				// executable.

    // Following functions are currently only available under UNIX
    // like systems.

    OTC_Pathname	absolute() const;
				// Returns the pathname as an absolute path.
				// If the path was already absolute then
				// it is returned unchanged. If it was
				// relative, the current working directory is
				// prepended to it and the result returned.
				// In the case of the pathname being ".", the
				// current working directory is returned
				// without a trailing "/.". If the pathname
				// is empty, an empty pathname is returned.

    OTC_Pathname	dirname() const;
				// Behaves like <dirname(1)>, returning
				// the directory portion of the pathname.

    OTC_Pathname	basename() const;
				// Behaves like <basename(1)>, returning
				// the filename portion of the pathname.

    OTC_String		basename(char const* theSuffix) const;
				// Behaves like <basename(1)>, returning the
				// filename portion of the pathname less
				// <theSuffix> if the file did in fact end in
				// that suffix.

    // Note that if the pathname is empty both <isAbsolute()> and
    // <isRelative()> will return <OTCLIB_FALSE>.

    OTC_Boolean		isAbsolute() const;
				// Returns <OTCLIB_TRUE> if the pathname
				// commences with a </>.

    OTC_Boolean		isRelative() const;
				// Returns <OTCLIB_TRUE> if the pathname
				// doesn't commence with a </>.

    // = EXTENSION

    OTC_Pathname&	operator+=(char theChar)
				{ myPath += theChar; return *this; }
				// Extend the pathname with <theChar>.

    OTC_Pathname&	operator+=(char const* theString)
				{ myPath += theString; return *this; }
				// Extend the pathname with <theString>.

    OTC_Pathname&	operator+=(OTC_String const& theString)
				{ myPath += theString; return *this; }
				// Extend the pathname with <theString>.

    OTC_Pathname&	operator+=(OTC_Pathname const& thePath)
				{ myPath += thePath.myPath; return *this; }
				// Extend the pathname with <thePath>.

    // = ADDITION

    OTC_TString		operator+(char theChar)
				{ return myPath + theChar; }
				// Returns the path with <theChar> appended.
				// This object is not modified.

    OTC_TString		operator+(char const* theString)
				{ return myPath + theString; }
				// Returns the path with <theString> appended.
				// This object is not modified.

    OTC_TString		operator+(OTC_String const& theString)
				{ return myPath + theString; }
				// Returns the path with <theString> appended.
				// This object is not modified.

    OTC_TString		operator+(OTC_Pathname const& thePath)
				{ return myPath + thePath; }
				// Returns the path with <thePath> appended.
				// This object is not modified.

  protected:

    OTC_RString		rawString() const;

  private:

    OTC_String		myPath;
				// The path.
};

/* ------------------------------------------------------------------------- */

// = STREAMS
// Both <operator\<\<()> and <operator\>\>()> are overloaded for operation
// with streams. The extraction operator behaves the same as that for the
// <OTC_String> class, namely it will extract up to the first whitespace
// character in the input. Also, if <ios::skipws> is defined it will first
// skip leading whitespace.

inline ostream& operator<<(ostream& outs, OTC_Pathname const& thePath)
{
  outs << thePath.path();
  return outs;
}

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
OSE_MARK_TEMPLATE OTC_RankActions<OTC_Pathname>;
OSE_MARK_TEMPLATE OTC_HashActions<OTC_Pathname>;
#endif

#if defined(CXX_OS)
typedef OTC_RankActions<OTC_Pathname> otc_files_pathname_hh_typedef1;
#pragma ObjectStore exclude_instantiations OTC_RankActions<OTC_Pathname>;
typedef OTC_HashActions<OTC_Pathname> otc_files_pathname_hh_typedef2;
#pragma ObjectStore exclude_instantiations OTC_HashActions<OTC_Pathname>;
#endif

#if !defined(CXX_OS)
class OTC_RankActions<OTC_Pathname>
{
  public:
    static int		rank(OTC_Pathname const& p1, OTC_Pathname const& p2)
				{
				  return OTC_RankActions<OTC_String>::rank(
				   p1.path(),p2.path());
				}
};

class OTC_HashActions<OTC_Pathname>
{
  public:
    static int		hash(OTC_Pathname const& p)
				{
				  return OTC_HashActions<OTC_String>::hash(
				   p.path());
				}
};
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_FILES_PATHNAME_HH */
