#ifndef OTC_FILES_STAT_HH
#define OTC_FILES_STAT_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     files/stat.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 1992 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/OTC.h>

#include <sys/types.h>
#include <sys/stat.h>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/files/stat.hh"
#else
#pragma interface
#endif
#endif

#if !defined(S_IREAD) && defined(S_IRUSR)
#define S_IREAD S_IRUSR
#endif
#if !defined(S_IWRITE) && defined(S_IWUSR)
#define S_IWRITE S_IWUSR
#endif
#if !defined(S_IEXEC) && defined(S_IXUSR)
#define S_IEXEC S_IXUSR
#endif

#if !defined(S_IFMT) && defined(_IFMT)
#define S_IFMT _IFMT
#endif
#if !defined(S_IFDIR) && defined(_IFDIR)
#define S_IFDIR _IFDIR
#endif
#if !defined(S_IFREG) && defined(_IFREG)
#define S_IFREG _IFREG
#endif
#if !defined(S_IFCHR) && defined(_IFCHR)
#define S_IFCHR _IFCHR
#endif

#if !defined(S_IFMT) && defined(_S_IFMT)
#define S_IFMT _S_IFMT
#endif
#if !defined(S_IFDIR) && defined(_S_IFDIR)
#define S_IFDIR _S_IFDIR
#endif
#if !defined(S_IFREG) && defined(_S_IFREG)
#define S_IFREG _S_IFREG
#endif
#if !defined(S_IFCHR) && defined(_S_IFCHR)
#define S_IFCHR _S_IFCHR
#endif

#if !defined(S_IFMT) && defined(__IBMCPP__)
#define S_IFMT 0x17000
#endif

/* ------------------------------------------------------------------------- */

typedef struct stat otclib_stat;

class OTC_Stat
    // = TITLE
    //	   Class wrapper around the system stat structure.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DECSCRIPTION
    //	   Encapsulates the system stat structure to make deriving information
    //	   about a file easier. Note that it is still necessary to use the
    //	   system <stat()> function to fill out information in the class.
    //
    // = EXAMPLE
    //
    // = BEGIN<CODE>
    //	   OTC_Stat info;
    //	   if (stat("/tmp",info) == 0)
    //	   {
    //	     if (info.isDirectory())
    //	       cout << "Is a directory" << endl;
    //	   }
    // = END<CODE>
    //
    // = SEE ALSO
    //	   <stat(2)>
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

			operator otclib_stat*()
				{ return &myStat; }

			OTC_Stat();

			OTC_Stat(OTC_Stat const& theStat);

    OTC_Stat&		operator=(OTC_Stat const& theStat);

    // = ATTRIBUTES

    dev_t		dev() const
				{ return myStat.st_dev; }
				// Returns the device id for device which
				// the file resides upon.

    u_short		mode() const
				{ return myStat.st_mode; }
				// Returns the files mode bits.

    u_short		permissions() const
				{ return mode() & 0777; }
				// Returns the bits of the mode
				// giving the file permissions.

    u_short		type() const
				{ return mode() & S_IFMT; }
				// Returns the bits of the mode
				// giving the file type.

    off_t		size() const
				{ return myStat.st_size; }
				// Returns the size of the file.

    time_t		atime() const
				{ return myStat.st_atime; }
				// Returns the last access time of the file.

    time_t		mtime() const
				{ return myStat.st_mtime; }
				// Returns the last modification time of the
				// file.

    time_t		ctime() const
				{ return myStat.st_ctime; }
				// Returns the last time which the status of
				// the file was changed.

    // The following simplify questions commonly asked about files.

    OTC_Boolean		isDirectory() const
				{ return (type() & S_IFDIR) == S_IFDIR; }
				// Returns <OTCLIB_TRUE> if file is a
				// directory.

    OTC_Boolean		isRegularFile() const
				{ return (type() & S_IFREG) == S_IFREG; }
				// Returns <OTCLIB_TRUE> if file is a regular
				// file.

    OTC_Boolean		isCharacterSpecial() const
				{ return (type() & S_IFCHR) == S_IFCHR; }
				// Returns <OTCLIB_TRUE> if file is a character
				// special file.

#if defined(S_IFIFO)
    OTC_Boolean		isFifo() const
				{ return (type() & S_IFIFO) == S_IFIFO; }
				// Returns <OTCLIB_TRUE> if file is a fifo.
#endif

#if defined(S_IFBLK)
    OTC_Boolean		isBlockSpecial() const
				{ return (type() & S_IFBLK) == S_IFBLK; }
				// Returns <OTCLIB_TRUE> if file is a block
				// special file.
#endif

    OTC_Boolean		isUserRead() const
				{ return mode() & S_IREAD; }
				// Returns <OTCLIB_FALSE> if not readable by
				// the user who owns the file.

    OTC_Boolean		isUserWrite() const
				{ return mode() & S_IWRITE; }
				// Returns <OTCLIB_FALSE> if not writable by
				// the user who owns the file.

    OTC_Boolean		isUserExec() const
				{ return mode() & S_IEXEC; }
				// Returns <OTCLIB_FALSE> if not executable
				// by the user who owns the file.

#if defined(SYS_UNIX)

    // Following functions are only available on UNIX like systems.
    // Some functions are not available on older SystemV UNIX systems.

    ino_t		ino() const
				{ return myStat.st_ino; }
				// Returns the inode of the file.

    short		nlink() const
				{ return myStat.st_nlink; }
				// Returns the number of hardlinks to the
				// file.

    uid_t		uid() const
				{ return myStat.st_uid; }
				// Returns the owners user id.

    gid_t		gid() const
				{ return myStat.st_gid; }
				// Returns the group user id.

#ifdef HAVE_ST_RDEV
    dev_t		rdev() const
				{ return myStat.st_rdev; }
				// Returns the device type.
#endif

#ifdef HAVE_ST_BLKSIZE
    long		blksize() const
				{ return myStat.st_blksize; }
				// Returns the optimal blocksize for file
				// system i/o operations.
#endif

#ifdef HAVE_ST_BLOCKS
    long		blocks() const
				{ return myStat.st_blocks; }
				// Returns the number of blocks allocated to
				// the file.
#endif

    // The following simplify questions commonly asked about files.

#ifdef S_IFLNK
    OTC_Boolean		isSymbolicLink() const
				{ return (type() & S_IFLNK) == S_IFLNK; }
				// Returns <OTCLIB_TRUE> if file is a
				// symbolic link.
#endif

#ifdef S_IFSOCK
    OTC_Boolean		isSocket() const
				{ return (type() & S_IFSOCK) == S_IFSOCK; }
				// Returns <OTCLIB_TRUE> if file is a socket.
#endif

    OTC_Boolean		isSetUid() const
				{ return (mode() & S_ISUID) == S_ISUID; }
				// Returns <OTCLIB_TRUE> if file is setuid.

    OTC_Boolean		isSetGid() const
				{ return (mode() & S_ISGID) == S_ISGID; }
				// Returns <OTCLIB_TRUE> if file is setgid.

    OTC_Boolean		isSaveTxt() const
				{ return (mode() & S_ISVTX) == S_ISVTX; }
				// Returns <OTCLIB_TRUE> if file is has its
				// stickybit set.

    OTC_Boolean		isGroupRead() const
				{ return mode() & 0040; }
				// Returns <OTCLIB_FALSE> if not readable by
				// the users in the group of the file.

    OTC_Boolean		isGroupWrite() const
				{ return mode() & 0020; }
				// Returns <OTCLIB_FALSE> if not writable by
				// the users in the group of the file.

    OTC_Boolean		isGroupExec() const
				{ return mode() & 0010; }
				// Returns <OTCLIB_FALSE> if not executable
				// by the users in the group of the file.

    OTC_Boolean		isOtherRead() const
				{ return mode() & 0004; }
				// Returns <OTCLIB_FALSE> if not readable by
				// any users.

    OTC_Boolean		isOtherWrite() const
				{ return mode() & 0002; }
				// Returns <OTCLIB_FALSE> if not writable by
				// any users.

    OTC_Boolean		isOtherExec() const
				{ return mode() & 0001; }
				// Returns <OTCLIB_FALSE> if not executable by
				// any users.

#endif

  private:

    otclib_stat		myStat;
				// Stat structure.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_FILES_STAT_HH */
