#ifndef OTC_MEMORY_ARENA_HH
#define OTC_MEMORY_ARENA_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     memory/arena.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/memory/mpobject.hh>
#include <OTC/memory/align.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/memory/arena.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_ArenaBlock : public OTC_MPObject
    // = TITLE
    //     Class to keep track of arena block information.
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

			OTC_ArenaBlock() {}

			~OTC_ArenaBlock();

    char*		block;

    char*		free;

    size_t		size;

    OTC_ArenaBlock*	next;
};

class OTC_Arena
    // = TITLE
    //     A memory allocator which concatenates requests into blocks.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     <OTC_Arena> is a memory allocator which obtains blocks
    //     of memory using <operator new()> and then parcels the
    //     memory out in pieces. All memory allocated must be freed
    //     at the same time.
    //
    // = NOTES
    //     If a request for memory greater than the block size is
    //     requested, it will be allocated directly from the free
    //     store.
    //
    //     The default block size is <2040>. A weird value calculated
    //     by looking at size of blocks allocated by GNU malloc and
    //     BSD malloc. Sun malloc used first fit, so the size doesn't
    //     matter too much when using it.
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    // = CONSTRUCTION

			OTC_Arena(size_t theAlign=OTC_Alignment::ofDouble());
				// Initialises the class. The block size used
				// for allocating memory will be <2040> bytes
				// unless overridden by the environment
				// variable <OTCLIB_ARENABLOCKSIZE>. When the
				// amount of free space in a block falls
				// below <16> bytes, the class will stop
				// trying to allocate memory from that block.
				// The slop value can be overridden by
				// setting the environment variable
				// <OTCLIB_ARENASLOPSIZE>. Memory returned
				// will be aligned according to <theAlign>.
				// The default is to align memory to that
				// required by the type <double>. If
				// <theAlign> is <0> an exception is raised.

			OTC_Arena(
			 size_t theBlockSize,
			 size_t theSlop,
			 size_t theAlign=OTC_Alignment::ofDouble()
			);
				// Initialises the class. <theBlockSize>
				// should be the minimum amount of memory
				// allocated from the free store. When the
				// amount of free space in a block decreases
				// below <theSlop>, the class stops trying to
				// allocate from that block. Memory returned
				// will be aligned according to <theAlign>.
				// The default is to align memory to that
				// required by the type <double>.
				// If <theAlign> is <0> an exception is
				// raised.

    // = DESTRUCTION

			~OTC_Arena();
				// Returns all memory to the free store.

    // = ALLOCATION

    void*		allocate(size_t theSize);
				// Returns a piece of memory of size
				// <theSize>.

  private:

			OTC_Arena(OTC_Arena const&);
				// Do not define an implementation for this.

    OTC_Arena&		operator=(OTC_Arena const&);
				// Do not define an implementation for this.

    void*		allocateNewBlock(size_t theSize);
				// Allocates a new block of memory and
				// allocate <theSize> memory out of it.

    size_t		myBlockSize;
				// Minimum block size in which allocations
				// of free store memory should be made.

    size_t		mySlop;
				// When the amount of free space in a block
				// decreases below this amount we stop
				// trying to allocate more strings from it.

    size_t		myAlign;
				// Memory alignment.

    OTC_ArenaBlock*	myFull;
				// The blocks of memory which are full.

    OTC_ArenaBlock*	myFree;
				// The blocks of memory which still hold
				// some free space.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_MEMORY_ARENA_HH */
