#ifndef OTC_MEMORY_POOL_HH
#define OTC_MEMORY_POOL_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     memory/pool.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/OTC.h>

#include <stddef.h>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/memory/pool.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Pool
    // = TITLE
    //     A fast memory allocator.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     This is a fast memory allocator modelled after a Pool class by Dag
    //     Bruck. which was in turn modelled after the Pool class in the USL
    //     C++ Class Library.
{
  private:

    struct OTC_PoolElement
	// The beginning of a pool element within a block.
    {
      OTC_PoolElement*	next;
    };

    struct OTC_PoolBlock
        // The beginning of a block of allocated pool memory.
    {
      OTC_PoolBlock*	next;
    };

    OTC_PoolBlock*	myBlocks;
				// List of memory blocks from which elements
				// are allocated.

    size_t		myBlockSize;
				// The size of a memory block.

    size_t		myHeaderSize;
				// Offset into memory of first element.

    OTC_PoolElement*	myElements;
				// The list of free pool elements.

    size_t		myElementSize;
				// The size of a pool element. This number is
				// rounded up to a multiple of some small
				// number (to ensure alignment).

    size_t		myElementCount;
				// The number of pool elements which fit
				// in a memory block.

    size_t		myCapacity;
				// The total number of elements managed by
				// this pool. This includes both allocated
				// and free.

    size_t		myAllocated;
				// Count of the number of pool elements
				// currently allocated to users.

  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    // = CONSTRUCTION
    //     Note that the constructors do not allocate an initial block of
    //     memory. The first block will only be allocated the first time
    //     <allocate()> is called. If the maximum block size is not of
    //     sufficient size to hold one block and enough space for internal
    //     requirements, more than the maximum size will be allocated.

			OTC_Pool(size_t theSize);
				// Creates a pool for blocks of memory of
				// <theSize>. If the environment variable
				// <OTCLIB_POOLBLOCKSIZE> is set to an
				// integer value, it will be used as the
				// the maximum block size when allocating
				// new blocks of memory from the operating
				// system. If the environment variable is not
				// defined, a default of <2040> bytes will be
				// used.

			OTC_Pool(size_t theSize, size_t theBlockSize)
			  : myBlocks(0),
			    myElements(0),
			    myCapacity(0),
			    myAllocated(0)
				{ initialise(theSize,theBlockSize); }
				// Creates a pool for blocks of memory of
				// <theSize>. The value <theBlockSize>, will
				// be used as the maximum block size when
				// allocating new blocks of memory from the
				// operating system.

    // = DESTRUCTION

			~OTC_Pool();
				// Returns all memory allocated by the pool
				// to the operating system, unless there is
				// still memory allocated to the user, in
				// which case an exception is raised.

    // = QUERY

    size_t		elementSize() const
				{ return myElementSize; }
				// Returns the maximum size of the objects
				// which can allocated by the pool. Note
				// that if smaller object are allocated,
				// the mory returned will still be of the
				// size indicated by this function.

    size_t		blockSize() const
				{ return myBlockSize; }
				// Returns the block size used when
				// allocating more memory from the operating
				// system.

    size_t		elementCount() const
				{ return myElementCount; }
				// Returns the number of objects of the size
				// managed by this memory pool, which can
				// fit into each block of memory which is
				// allocated from the operating system.

    size_t		capacity() const
				{ return myCapacity; }
				// Returns the total number of objects which
				// can be managed by this pool. This is the
				// sum of the free object elements and those
				// which have already been allocated.

    size_t		allocated() const
				{ return myAllocated; }
				// Returns the number of objects currently
				// allocated from this pool.

    // = ALLOCATION/DEALLOCATION

    void*		allocate()
				{
				  if (myElements == 0)
				    grabMemory();
				  OTC_PoolElement* e = myElements;
				  myElements = e->next;
				  myAllocated++;
				  return e;
				}
				// A new element is allocated from the pool
				// and returned. If there are no free
				// elements, more memory will be allocated
				// from the operating system.

    void		release(void* theMemory)
				{
				  OTCLIB_ASSERT(myAllocated != 0);
				  OTC_PoolElement* e;
				  e = (OTC_PoolElement*)theMemory;
				  e->next = myElements;
				  myElements = e;
				  myAllocated--;
				}
				// The element of the pool addressed by
				// <theMemory> is freed. The element must
				// have been allocated from this pool. If the
				// memory was not from this pool, the results
				// are undefined.

    void		grabMemory(size_t theSize=0);
				// Grabs a new block of memory from the
				// operating system and splits it up in to
				// elements that are then inserted into the
				// free list. The size of the block of memory
				// allocated will be the larger of <theSize>
				// and the default block size.

  private:

			OTC_Pool(OTC_Pool const&);
				// Do not define an implementation for this.

    OTC_Pool&		operator=(OTC_Pool const&);
				// Do not define an implementation for this.

    void		initialise(size_t theSize, size_t theBlockSize);
				// Initialises the element size, block size
				// and element count for this pool, given
				// that the user wants blocks of memory of
				// size <theSize>, and wants allocations of
				// memory from the operating system to be in
				// the region of <theBlockSize>.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_MEMORY_POOL_HH */
