#ifndef OTC_MEMORY_VECPTR_HH
#define OTC_MEMORY_VECPTR_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     memory/vecptr.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/OTC.h>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/vecptr.hh"
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/vecptr.hh"
#else
#pragma interface
#endif
#endif
#endif

/* ------------------------------------------------------------------------- */

template<class T>
class OTC_VecPtr
    // = TITLE
    //	   Smart pointer to help manage resources.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   Designed such that the array of objects being managed is only
    //	   created when required. Also deletes the array of objects when the
    //	   pointer class is destroyed.
{
  private:

    T*			_item() const
				{
				  if (myItem == 0)
				  {
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
				    OTC_Locality theLocality =
				     OTC_Locality::of(this);
#endif
#if defined(ENV_OSTORE_DML)
				    ((OTC_VecPtr<T>*)this)->myItem =
				     new ((os_segment*)theLocality) T[mySize];
#else				     
#if defined(ENV_OSTORE)
				    ((OTC_VecPtr<T>*)this)->myItem = new
				     (theLocality,OTC_TypeSpec<T>::typespec())
				     T[mySize];
#else				      
				    ((OTC_VecPtr<T>*)this)->myItem =
				     new T[mySize];
#endif				     
#endif
				    OTCLIB_ASSERT(myItem != 0);
				  }
				  return myItem;
				}
				// Returns the array of objects being pointed
				// at. If the array of objects had not already
				// been created, it will be created.
  public:

    // = CONSTRUCTION

			OTC_VecPtr(u_int theSize)
			  : mySize(theSize), myItem(0) {}
				// Initialises the class to an undefined
				// state. The array of objects is only
				// created when it is required. <theSize> is
				// the desired length of the array.

    // = DESTRUCTION

			~OTC_VecPtr()
				{ if (myItem != 0) delete [] myItem; }
				// Deletes the array of objects if it had
				// be created.

    // = ACCESS

    T*			item()
				{ return _item(); }
				// Returns the array of objects being pointed
				// at. If the array of objects had not already
				// been created, it will be created.

    T const*		item() const
				{ return _item(); }
				// Returns the array of objects being pointed
				// at. If the array of objects had not already
				// been created, it will be created.

    T&			operator[](u_int theIndex)
				{ return item()[theIndex]; }
				// Returns a reference to the object in
				// the array at position <theIndex>.
				// No bounds checking is performed.

    T const&		operator[](u_int theIndex) const
				{ return item()[theIndex]; }
				// Returns a reference to the object in
				// the array at position <theIndex>.
				// No bounds checking is performed.

			operator T*()
				{ return item(); }
				// Returns the array of objects being pointed
				// at.

			operator T const*() const
				{ return item(); }
				// Returns the array of objects being pointed
				// at.

    T&			operator*()
				{ return *item(); }
				// Returns a reference to the first object in
				// the vector being held. If no item is held,
				// an exception is generated.

    T const&		operator*() const
				{ return *item(); }
				// Returns a reference to the first object in
				// the vector being held. If no item is held,
				// an exception is generated.

    // = QUERY

    u_int		size() const
				{ return mySize; }
				// Returns the size of the array.

    OTC_Boolean		isUndefined() const
				{ return myItem == 0; }
				// Returns <OTCLIB_TRUE> if the array of
				// items has not yet been created.

  private:

			OTC_VecPtr(OTC_VecPtr<T> const&)
				{ terminate(); }
				// Do not define an implementation for this.

    OTC_VecPtr<T>&	operator=(OTC_VecPtr<T> const&)
				{ terminate(); return *this; }
				// Do not define an implementation for this.

    u_int		mySize;
				// The size of the array.

    T*			myItem;
				// The array of items being held.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_MEMORY_VECPTR_HH */
