#ifndef OTC_OSTORE_LOCALITY_HH
#define OTC_OSTORE_LOCALITY_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     ostore/locality.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifndef OTC_OSTORE_OSTORE_HH
#include <OTC/ostore/ostore.hh>
#endif

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)

#include <OTC/ostore/storage.hh>

/* ------------------------------------------------------------------------- */

enum OTC_LocalityType
{
  OTCLIB_SEGMENT,
  OTCLIB_CLUSTER,
  OTCLIB_STORAGE
};

class OTC_Storage;

union OTC_LocalityUnion
{
  void* _void;
  os_segment* _segment;
  os_object_cluster* _cluster;
  OTC_Storage* _storage;
};

class OTC_Locality
    // = TITLE
    //     Pointer object which can hold either a segment or cluster pointer.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     Bundles up ObjectStore segment and cluster pointers into one object
    //     so as to simplify code which needs to work for both segment and
    //     cluster levels of storage locality. A global <operator new()>
    //     is provided which accepts an instance of <OTC_Locality>.
{
  public:

			~OTC_Locality() {}

    // = INITIALISATION

			OTC_Locality()
			  : _type(OTCLIB_SEGMENT)
				{
				  _ptr._segment =
				   os_segment::get_transient_segment();
				}
				// Initialises this class to refer the
				// transient segment.

			OTC_Locality(OTC_Locality const& theLocality)
			  : _type(theLocality._type)
				{ _ptr._void = theLocality._ptr._void; }
				// Initialises this class to refer to the
				// same locality as <theLocality>.

			OTC_Locality(os_segment* theSegment)
			  : _type(OTCLIB_SEGMENT)
				{ _ptr._segment = theSegment; }
				// Initialises this class to refer to
				// <theSegment>.

#if !defined(ENV_OSTORE_DML)
			OTC_Locality(os_object_cluster* theCluster)
			  : _type(OTCLIB_CLUSTER)
				{ _ptr._cluster = theCluster; }
				// Initialises this class to refer to
				// <theCluster>.

			OTC_Locality(OTC_Storage* theStorage)
			  : _type(OTCLIB_STORAGE)
				{ _ptr._storage = theStorage; }
				// Initialises this class to refer to
				// <theStorage>.
#endif

    OTC_Locality&	operator=(OTC_Locality const& theLocality)
				{
				  _type = theLocality._type;
				  _ptr._void = theLocality._ptr._void;
				  return *this;
				}
				// Sets this class to refer to the same
				// locality as <theLocality>.

    OTC_Locality&	operator=(os_segment* theSegment)
				{
				  _type = OTCLIB_SEGMENT;
				  _ptr._segment = theSegment;
				  return *this;
				}
				// Sets this class to refer to <theSegment>.

#if !defined(ENV_OSTORE_DML)
    OTC_Locality&	operator=(os_object_cluster* theCluster)
				{
				  _type = OTCLIB_CLUSTER;
				  _ptr._cluster = theCluster;
				  return *this;
				}
				// Sets this class to refer to <theCluster>.

    OTC_Locality&	operator=(OTC_Storage* theStorage)
				{
				  _type = OTCLIB_STORAGE;
				  _ptr._storage = theStorage;
				  return *this;
				}
				// Sets this class to refer to <theStorage>.
#endif

    // = QUERY

    OTC_Boolean		isSegment() const
				{ return _type == OTCLIB_SEGMENT; }
				// Returns <OTCLIB_TRUE> if the locality
				// refers to a segment.

    OTC_Boolean		isTransientSegment() const
				{
				  return _ptr._segment ==
				   os_segment::get_transient_segment();
				 }
				// Returns <OTCLIB_TRUE> if the locality
				// refers to the transient segment.

#if !defined(ENV_OSTORE_DML)
    OTC_Boolean		isCluster() const
				{ return _type == OTCLIB_CLUSTER; }
				// Returns <OTCLIB_TRUE> if the locality
				// refers to a cluster.

    OTC_Boolean		isStorage() const
				{ return _type == OTCLIB_STORAGE; }
				// Returns <OTCLIB_TRUE> if the locality
				// refers to a storage manager.
#endif

#if defined(ENV_OSTORE_DML)
			operator os_segment*() const;
				// Returns the segment pointer. Will
				// be zero if this class doesn't refer to
				// a segment.
#endif

    os_segment*		segment() const
				{ return isSegment() ? _ptr._segment : 0; }
				// Returns the segment pointer. Will
				// be zero if this class doesn't refer to
				// a segment.

#if !defined(ENV_OSTORE_DML)
    os_object_cluster*	cluster() const
				{ return isCluster() ? _ptr._cluster : 0; }
				// Returns the cluster pointer. Will
				// be zero if this class doesn't refer to
				// a cluster.

    OTC_Storage*	storage() const
				{ return isStorage() ? _ptr._storage : 0; }
				// Returns the storage manager pointer. Will
				// be zero if this class doesn't refer to
				// a storage manager.
#endif

    OTC_Boolean		operator==(OTC_Locality const& theLocality) const
				{ return _ptr._void == theLocality._ptr._void; }
				// Returns <OTCLIB_TRUE> if <theLocality>
				// is the same as this locality.

    OTC_Boolean		operator!=(OTC_Locality const& theLocality) const
				{ return _ptr._void != theLocality._ptr._void; }
				// Returns <OTCLIB_TRUE> if <theLocality>
				// is not the same as this locality.

    static OTC_Locality	of(void const* theMem);
				// Returns the locality in which the
				// address <theMem> is located.

  private:

    OTC_LocalityUnion	_ptr;
				// Union containing pointer to the
				// locality.

    OTC_LocalityType	_type;
				// The type of the locality.
};

#if defined(ENV_OSTORE)
extern void* operator new(size_t, OTC_Locality const&, os_typespec*);
extern void* operator new(size_t, OTC_Locality const&, os_typespec*, os_int32);
#endif

/* ------------------------------------------------------------------------- */

#endif

#endif /* OTC_OSTORE_LOCALITY_HH */
