#ifndef OTC_TEXT_REGEXP_HH
#define OTC_TEXT_REGEXP_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     text/regexp.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/text/pattern.hh>
#include <OTC/thread/nrmutex.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/text/regexp.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

enum { OTCLIB_NSUBEXP=10 };

struct OTC_RegexpStruct
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    char const*		startp[OTCLIB_NSUBEXP];
    char const*		endp[OTCLIB_NSUBEXP];
    char		regstart;
    char		reganch;
    char*		regmust;
    int			regmlen;
    char*		program;
};

class OTC_Regexp : public OTC_Pattern
    // = TITLE
    //	   Class to perform <egrep> style pattern matching.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   Refer to the <egrep(1)> manual page for the format of
    //	   the pattern.
{
  private:

    char const*		regparse;

    int			regnpar;

    char		regdummy;

    char*		regcode;

    long		regsize;

    char const*		reginput;

    char const*		regbol;

    char const**	regstartp;

    char const**	regendp;

    OTC_RegexpStruct*	regcomp(char const*);

    int			regexec(OTC_RegexpStruct*, char const*);

    void		regerror(char const* mesg)
				{ myError = mesg; }

    char*		reg(int, int*);

    char*		regbranch(int*);

    char*		regpiece(int*);

    char*		regatom(int*);

    char*		regnode(char);

    char*		regnext(char*);

    void		regc(char);

    void		reginsert(char, char*);

    void		regtail(char*, char*);

    void		regoptail(char*, char*);

    int			regtry(OTC_RegexpStruct*, char const*);

    int			regmatch(char*);

    int			regrepeat(char*);

  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

			~OTC_Regexp();

    // = CONSTRUCTION

			OTC_Regexp(char const* thePattern);
				// Compiles <thePattern>.

    // = MATCHING

    OTC_Boolean		match(char const* theString);
				// Returns <OTCLIB_TRUE> if the pattern matched
				// against <theString>.

    u_int		start() const;
				// Returns the index of the first character
				// in the matched portion of the string.

    u_int		start(u_int theIndex) const;
				// Returns the index of the first character
				// in the tagged portion of the matched
				// string indicated by <theIndex>.

    u_int		length() const;
				// Returns the length of the matched portion
				// of the string.

    u_int		length(u_int theIndex) const;
				// Returns the length of the matched tagged
				// portion of the string indicated by
				// <theIndex>.

    OTC_Range		range() const
				{ return OTC_Pattern::range(); }
				// Returns a range object for the matched
				// portion of the string.

    OTC_Range		range(u_int theIndex) const;
				// Returns a range object for the matched
				// tagged portion of the string indicated by
				// <theIndex>.

    // = ERRORS

    OTC_Boolean		isValid() const;
				// Returns <OTCLIB_TRUE> if the pattern was
				// valid.

    char const*		error() const
				{ return myError; }
				// If the pattern was not valid, returns a
				// string describing the problem with the
				// pattern.

    // = PATTERNS
    //	   Some commonly used regular expression are defined as constant
    //	   character strings. These are:

    static OTC_Regexp&	whiteSpace();
				// Matches white space.

    static OTC_Regexp&	optWhiteSpace();
				// Optionally matches white space.

    static OTC_Regexp&	nonWhiteSpace();
				// Matches non-white space.

    static OTC_Regexp&	integerValue();
				// Matches integer number.

    static OTC_Regexp&	realValue();
				// Matches double number.

    static OTC_Regexp&	alphabetic();
				// Matches alpha characters.

    static OTC_Regexp&	lowerCase();
				// Matches lower case characters.

    static OTC_Regexp&	upperCase();
				// Matches upper case characters.

    static OTC_Regexp&	alphaNumeric();
				// Matches alphanumeric characters.

    static OTC_Regexp&	identifier();
				// Matches an identifier.

    static OTC_Regexp&	matchingQuotes();
				// Matches string delineated with double
				// quotes. Doesn't handle <\"> in
				// the string.

    static OTC_Regexp&	quotedString();
				// Matches string delineated with double
				// quotes. Handles <\"> in string.

  protected:

    // = COMPILATION

    void		compile();
				// Compiles the pattern.

  private:

			OTC_Regexp(OTC_Regexp const&);

    OTC_Regexp&		operator=(OTC_Regexp const&);

    char const*		myString;
				// The string being matched against.

    OTC_RegexpStruct*	myRegexp;
				// Information on the compiled pattern.

    char const*		myError;
				// The error message if the pattern is not
				// valid.

    static OTC_NRMutex	_mutex;
				// Lock for threads.

    static OTC_Regexp*	myWhiteSpace;
				// Matches white space.

    static OTC_Regexp*	myOptWhiteSpace;
				// Optionally matches white space.

    static OTC_Regexp*	myNonWhiteSpace;
				// Matches non-white space.

    static OTC_Regexp*	myIntegerValue;
				// Matches integer number.

    static OTC_Regexp*	myRealValue;
				// Matches double number.

    static OTC_Regexp*	myAlphabetic;
				// Matches alpha characters.

    static OTC_Regexp*	myLowerCase;
				// Matches lower case characters.

    static OTC_Regexp*	myUpperCase;
				// Matches upper case characters.

    static OTC_Regexp*	myAlphaNumeric;
				// Matches alphanumeric characters.

    static OTC_Regexp*	myIdentifier;
				// Matches an identifier.

    static OTC_Regexp*	myMatchingQuotes;
				// Matches string delineated with double
				// quotes. Doesn't handle <"> in
				// the string.

    static OTC_Regexp*	myQuotedString;
				// Matches string delineated with double
				// quotes. Handles <"> in string.

};

/* ------------------------------------------------------------------------- */

#endif /* OTC_TEXT_REGEXP_HH */
