#ifndef OTC_TEXT_RSTRING_HH
#define OTC_TEXT_RSTRING_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     text/rstring.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 1994 TELSTRA CORPORATION LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/memory/mpobject.hh>
#include <OTC/thread/nrmutex.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/text/rstring.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class ostream;

enum OTC_StringBuffering
    // = TITLE
    //     Indicates whether memory should have additional space for
    //     buffering when allocated.
{
  OTCLIB_BUFFERED,
  OTCLIB_UNBUFFERED
};

enum OTC_BufferingType
    // = TITLE
    //     Indicates strategy used for buffering memory.
{
  OTCLIB_16BUFFERING,
  OTCLIB_MMBUFFERING,
  OTCLIB_NOBUFFERING
};

class OTC_StringData : public OTC_MPObject
    // = TITLE
    //     Holder of memory for a string.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     Holds the actual memory for a string. Also contains capacity,
    //     length and  reference count information. This data needs to
    //     be in a separate class so reference count can still be adjusted
    //     for a const string object.
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

			OTC_StringData() {}

			~OTC_StringData();

    u_int		capacity;
				// The capacity of the memory allocated.
				// This must always be at least one greater
				// that the length.

    u_int		length;
				// The amount of memory in use. This does
				// not count the null terminator.

    u_int		count;
				// The number of references to this piece of
				// memory.

    short		locked;
				// Whether the memory is currently locked
				// for use.

    short		symbol;
				// Whether the memory represents a symbol.

    char*		data;
				// The actual memory.
};

class OTC_RString
    // = TITLE
    //	   Envelope class for holder of string memory.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     The <OTC_RString> class is a wrapper for <OTC_StringData>. This
    //     class implements the functionality for maintaining the string
    //     memory and the delayed copy mechanism. The <OTC_StringData> class
    //     holds all the data required. This class ensures that the string is
    //     null terminated on initial allocation and reallocation. If you
    //     overwrite the null terminator it is your own problem.
    //
    // = NOTES
    //     Thread locking is performed around code related to the
    //     manipulation of the reference count used for the delayed
    //     copy mechanism. This is necessary as two threads could
    //     validly access the shared data before a delayed copy is
    //     broken. Lock is done by function not for each
    //     instance of the class. The functions where this locking
    //     occurs is <operator=()> and <reallocate()>. Because locking
    //     is done by function, this may or may not reduce possible
    //     concurrency when there is large amounts of string resizing
    //     and assignment.
{
    friend		class OTC_Symbol;

  public:

			~OTC_RString();

    // = INITIALISATION

			OTC_RString(
			 u_int theLength,
			 OTC_StringBuffering theType=OTCLIB_BUFFERED
			);
				// Allocates string memory for string of
				// length <theLength>. If <theType> is
				// <OTCLIB_UNBUFFERED>, only enough memory
				// to hold the string and a null terminator
				// will initially be allocated. If <theType>
				// is <OTCLIB_BUFFERED>, the default,
				// additional memory may be allocated to try
				// to reduce memory allocations on changes in
				// the length of the string.

			OTC_RString(
			 u_int theLength,
			 void const* theObject,
			 OTC_StringBuffering theType=OTCLIB_BUFFERED
			);
				// Allocates string memory for string of
				// length <theLength>. If <theType> is
				// <OTCLIB_UNBUFFERED>, only enough memory
				// to hold the string and a null terminator
				// will initially be allocated. If <theType>
				// is <OTCLIB_BUFFERED>, the default,
				// additional memory may be allocated to try
				// to reduce memory allocations on changes in
				// the length of the string. The additional
				// argument of <theObject> indicates which
				// object internal memory should be allocated
				// with when ObjectStore is being used.

			OTC_RString(OTC_RString const& theString);
				// Results in this class referencing the same
				// memory as <theString>.

    // = QUERY

    u_int		length() const
				{ return myData->length; }
				// Returns the length of the string.
				// Does not include the null terminator.

    u_int		capacity() const
				{ return myData->capacity-1; }
				// Returns the capacity of the string.
				// Doesn't include the cell which is
				// reserved for the null terminator.

    OTC_Boolean		isShared() const
				{ return (myData->count != 1); }
				// Returns <OTCLIB_TRUE> if this class
				// references a chunk of memory which is also
				// being referenced by another instance of
				// the class.

    OTC_Boolean		isLocked() const
				{ return myData->locked; }
				// Returns <OTCLIB_TRUE> if the memory is
				// currently locked for exclusive use.

    OTC_Boolean		isSymbol() const
				{ return myData->symbol; }
				// Returns <OTCLIB_TRUE> if the memory
				// represents a symbol. Only really useful to
				// <OTC_Symbol>.

    // = ACCESS
    //     Note that the following functions do nothing special if another
    //     instance of the class is referencing the same string. If
    //     modifications are going to be made to the memory, either
    //     <sync()> or <reallocate()> should be invoked to force a separate
    //     copy to be made.
    //
    //     If you overwrite the null terminator added by this class via
    //     these functions, it is your problem.

    char const*		string() const;
				// Returns a pointer to the string.

    char*		string();
				// Returns a pointer to the string.

    char		operator[](u_int theIndex) const
				{ return myData->data[theIndex]; }
				// Returns a reference to the character at
				// position <theIndex> into the string. No
				// bounds checking is performed.

    char&		operator[](u_int theIndex)
				{ return myData->data[theIndex]; }
				// Returns a reference to the character at
				// position <theIndex> into the string. No
				// bounds checking is performed.

    // = ALLOCATION

    void		reallocate(u_int theLength);
				// Similar to invoking <realloc(3)>, this
				// function will result in a new string being
				// allocated with sufficient size to hold
				// <theLength> characters. If <theLength> is
				// less than or equal to the current capacity
				// of this string and the string is not also
				// being referenced by another instance of
				// this class, no reallocation will be
				// made, instead the same piece of memory
				// will be used, the length adjusted and a
				// new null terminator inserted. If a new
				// string is allocated, the contents of
				// the old string will be copied into the
				// new. In doing this, instances of this
				// class which previously shared the same
				// string as this class will no longer do so,
				// instead they will continue to reference
				// the old string.

    void		sync()
				{ reallocate(myData->length); }
				// If the string is shared, this forces a
				// copy to be made. It is equivalent to
				// invoking <reallocate(length())>.

    // = LOCKING

    void		lock();
				// Locks the memory for exclusive use.
				// If the memory is shared, a copy will
				// be made first. If the memory is already
				// locked an exception is raised.

    void 		unlock();
				// Unlocks the memory from exclusive use.
				// Raises an exception if the memory is not
				// locked.

    // = ASSIGNMENT

    OTC_RString const&	operator=(OTC_RString const& theString);
				// Results in the current string being
				// discarded and the one held by <theString>
				// to be referenced instead. The old string
				// will only be deleted if no other instances
				// of the class reference it.

    // = STREAMS OUTPUT

    friend ostream&	operator<<(ostream& outs, OTC_RString const& theString);
				// Dumps <theString> to the stream <outs>.
				// Width and justification specifications
                                // are honoured.

  private:

			OTC_RString(OTC_StringData* theData);
				// Uses <theData> as it except that the
				// reference count will be incremented.

    u_int		capacity(u_int theCapacity) const;
				// Returns an adjusted capacity. This is to
				// facilitate performance improvement through
				// buffering. Doesn't modify the current
				// capacity, just works out what it should
				// be for a suggested value of <theCapacity>.

    OTC_StringData*	data() const
				{ return myData; }
				// Returns pointer to string data. Provided
				// for <OTC_Symbol> when doing quick
				// conversion of string to symbol.

    static OTC_NRMutex	_mutex;
				// Lock for threads. This one is for global
				// data.

    static OTC_NRMutex	_sdmutex;
				// Lock for threads. This one is to protect
				// code manipulating shared string data.

    static OTC_Boolean	myInitialised;
				// Indicates if buffering strategy has been
				// set.

    static OTC_BufferingType	myBufType;
				// Strategy used for memory allocation.

    OTC_StringData*	myData;
				// Actual data.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_TEXT_RSTRING_HH */
