#ifndef OTC_TEXT_STRING_HH
#define OTC_TEXT_STRING_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     text/string.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 1994 TELSTRA CORPORATION LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/text/rstring.hh>
#include <OTC/text/sobject.hh>
#include <OTC/text/cstring.hh>
#include <OTC/text/tstring.hh>
#include <OTC/text/cmptype.hh>
#include <OTC/collctn/baseactn.hh>
#include <OTC/collctn/hashactn.hh>
#include <OTC/collctn/rankactn.hh>
#include <OTC/collctn/range.hh>
#include <OTC/collctn/bitsetc.hh>
#include <OTC/thread/nrmutex.hh>

#include <iostream.h>
#include <string.h>

#ifdef index
#undef index
#endif

#ifdef rindex
#undef rindex
#endif

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/text/string.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Capacity
    // = TITLE
    //     Class for holding string capacity.
{
  public:

			OTC_Capacity(u_int theCapacity)
			  : myCapacity(theCapacity) {}

    u_int		capacity() const
				{ return myCapacity; }

  private:

    u_int		myCapacity;
};

class OTC_Length
    // = TITLE
    //     Class for holding string length.
{
  public:

			OTC_Length(u_int theLength)
			  : myLength(theLength) {}

    u_int		length() const
				{ return myLength; }

  private:

    u_int		myLength;
};

class OTC_String
    // = TITLE
    //	   A class for holding a sequence of characters.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //     The <OTC_String> class is designed to hold a sequence of
    //     characters, including null characters. The class uses delayed copy
    //     to improve average performance.
    //     
    //	   The delayed copy mechanism means that when an assignment of one
    //	   string class is made to another, or when a new class is created
    //	   using another, they initially share the same underlying character
    //	   buffer. The number of instances of the string class referencing
    //	   the buffer is maintained through reference counting. Only when an
    //	   operation is performed which would modify the buffer is a private
    //	   copy made for that class. This scheme avoids unnecessary copies
    //	   being made of strings.
    //
    // = NOTES
    //	   Except where special behaviour is mentioned, a nil pointer is
    //	   dealt with as if it were a null terminated string.
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    // Use these at your own peril. They are part of the implementation
    // and thus are subject to change.

    static int		rank(
			 char const* theString1,
			 u_int theLength1,
			 char const* theString2,
			 u_int theLength2,
			 OTC_CmpType theType=OTCLIB_EXACTMATCH
			);
				// Returns an integer denoting the rank of
				// <theString1> and <theString2>. The
				// arguments <theLength1> and <theLength2>
				// and the respective lengths of the strings.
				// <theType> is to denote whether case
				// should be ignored when comparing
				// the strings. Valid values for <theType>
				// are <OTCLIB_EXACTMATCH> and
				// <OTCLIB_IGNORECASE>.

    static int		hash(char const* theString, u_int theNum)
				{ return otclib_hash(theString,theNum); }
				// Returns a hash value based on the
				// first <theNum> characters of <theString>.

    virtual		~OTC_String();

    // = INITIALISATION

			OTC_String();
				// Creates an empty string with a notionally
				// undefined value. The string will behave
				// exactly like an empty string, but
				// <isUndefined()> will return true until the
				// value of the string has been set.

			OTC_String(OTC_Capacity theCapacity);
				// Creates an empty string with enough
				// underlying capacity to hold a string of
				// <theCapacity>. This is equivalent to
				// the old <OTC_String> constructor
				// taking <u_int>.

			OTC_String(OTC_Length theLength);
				// Creates a string of length <theLength>.
				// The string is not initialised and
				// will be filled with garbage. This is
				// equivalent to the old <OTC_Buffer>
				// constructor taking <u_int>.

			OTC_String(
			 OTC_Length theLength,
			 OTC_Capacity theCapacity
			);
				// Creates a string of length <theLength>.
				// The string is not initialised and
				// will be filled with garbage. The
				// underlying buffer capacity will at the
				// same time be set to at least
				// <theCapacity>.

			OTC_String(char const* theString);
				// Creates a string holding a copy of the
				// null terminated string <theString>. If
				// <theString> is <0>, an empty string is
				// created.

			OTC_String(char const* theString, u_int theNum);
				// Creates a string holding a copy of the
				// first <theNum> characters of <theString>.
				// If <theString> is <0>, and <theNum> is
				// not <0>, an exception will be raised.

			OTC_String(char theChar, u_int theNum=1);
				// Creates a string containing <theNum>
				// occurrences of <theChar>. If <theNum>
				// is <0> an empty string is created.

			OTC_String(OTC_String const& theString);
				// Creates a string which references the same
				// character string as held by <theString>. A
				// copy is only made of the string when an
				// attempt is made to modify it.

			OTC_String(OTC_String const& theString, u_int theNum);
				// Creates a string holding a copy of the
				// first <theNum> characters of <theString>.
				// If the length of <theString> is less
				// than <theNum>, an exception is raised.

			OTC_String(OTC_SObject const& theString);
				// Creates a string which references the same
				// character string obtained from
				// <theString>. A copy is only made of the
				// string when an attempt is made to modify
				// it.

			OTC_String(OTC_CString const& theString);
				// Creates a string which references the same
				// character string as held by <theString>. A
				// copy is only made of the string when an
				// attempt is made to modify it. <OTC_CString>
				// is a constant string.

			OTC_String(OTC_RString const& theString);
				// Creates a string which references the same
				// character string as held by <theString>. A
				// copy is only made of the string when an
				// attempt is made to modify it. <OTC_RString>
				// is the raw string used to implement this
				// class.

			OTC_String(OTC_TString const& theString);
				// Creates a string which references the same
				// character string as held by <theString>. A
				// copy is only made of the string when an
				// attempt is made to modify it. Instances of
				// <OTC_TString> are returned from the
				// <operator+()> defined for strings.

    // = STREAM OPERATIONS

    friend istream&	operator>>(istream& ins, OTC_String& theString);
				// If the flag <ios::skipws> is set in <ins>,
				// any whitespace at the start of <ins> is
				// discarded. Characters are then read in and
				// placed into <theString>. Reading stops
				// when further whitespace or <EOF> is
				// encountered. If <ios::skipws> is not
				// defined and there is initial white space,
				// an empty string is returned. If a field
				// width is specified in <ins> it will be
				// honoured.

    friend ostream&	operator<<(ostream& outs, OTC_String const& theString);
				// Outputs <theString> onto <outs>. Any field
				// widths and justification flags set in
				// <outs> are honoured.

    // The following give similar functionality to <istream::get()> and
    // <istream::getline()>.

    static OTC_String	get(istream& ins, char theDelim=EOL);
				// Similiar to <istream::get()> except that
				// it is not required to specify a maximum
				// length for a buffer. In other words, any
				// number of characters up to but not
				// including <theDelim> are read from <ins>,
				// the result being returned as a string.
				// <theDelim> is left in <ins>.

    static OTC_String&	get(
			 OTC_String& theString,
			 istream& ins,
			 char theDelim=EOL
			);
				// Similiar to <istream::get()> except that
				// it is not required to specify a maximum
				// length for a buffer. In other words, any
				// number of characters up to but not
				// including <theDelim> are read from <ins>
				// and appended to <theString>. <theDelim> is
				// left in <ins>. Characters are appended to
				// <theString>. A reference to <theString> is
				// returned.

    static OTC_String	getline(istream& ins, char theDelim=EOL);
				// Similar to <istream::getline()> except
				// that it is not required to specify a
				// maximum length for a buffer. In other
				// words, any number of characters up to but
				// not including <theDelim> are read from <ins>,
				// the result being returned as a string.
				// <theDelim> is extracted from <ins>.

    static OTC_String&	getline(
			 OTC_String& theString,
			 istream& ins,
			 char theDelim=EOL
			);
				// Similar to <istream::getline()> except
				// that it is not required to specify a
				// maximum length for a buffer. In other
				// words, any number of characters up to but
				// not including <theDelim> are read from
				// <ins> and appended to <theString>.
				// <theDelim> is extracted from <ins>.
				// Characters are appended to <theString>. A
				// reference to <theString> is returned.

    // The following give similar functionality to <istream::read()>.

    static OTC_String	read(istream& ins, u_int theNum);
				// Similar to <istream::read()>. The string
				// into which the characters are read is
				// returned. The length of the string
				// returned should be checked to determine
				// how many characters were actually read.

    static OTC_String&	read(
			 OTC_String& theString,
			 istream& ins,
			 u_int theNum
			);
				// Similar to <istream::read()>. Characters
				// read in are appended to <theString>. The
				// length of the string returned should be
				// checked to determine how many characters
				// were actually read. A reference to
				// <theString> is returned.

    // The following stream functions give the same functionality as do the
    // member functions of the same name in the Rogue Wave string class.
    // These have been added to ease transition from the Rogue Wave string
    // class to this string class. The member functions do also present a
    // simpler interface than the stream functions above. Note though that
    // the value returned is the string class and not the stream from which
    // characters are being read. This is different from the Rogue Wave
    // string class.

    OTC_String&		readFile(istream& theStream);
				// Reads characters from <theStream>,
				// replacing the contents of this string
				// in the process. Reading of characters
				// from <theStream> will stop when <EOF>
				// is reached.

    OTC_String&		readLine(
			 istream& theStream,
			 OTC_Boolean theSkipWhite=OTCLIB_TRUE
			);
				// Reads characters from <theStream>,
				// replacing the contents of this string
				// in the process. Reading of characters
				// from <theStream> will stop when a newline
				// character is encountered, or the end of
				// the stream is reached. The newline
				// character is removed from the stream but
				// is not stored in this string. If
				// <theSkipWhite> is <OTCLIB_TRUE>, the
				// default, any leading whitespace is skipped
				// by applying the <iostream> manipulator
				// <ws>. Note that a newline is regarded as
				// whitespace, thus blank lines would be
				// skipped.

    OTC_String&		readString(istream& theStream);
				// Reads characters from <theStream>,
				// replacing the contents of this string in
				// the process. Reading of characters from
				// <theStream> will stop when a null
				// character is encountered, or the end of
				// the stream is reached. The null character
				// is removed from the stream but is not
				// stored in this string.

    OTC_String&		readToDelim(istream& theStream, char theDelim);
				// Reads characters from <theStream>,
				// replacing the contents of this string in
				// the process. Reading of characters from
				// <theStream> will stop when the delimiter
				// character <theDelim> is encountered, or
				// the end of the stream is reached. The
				// delimiter character is removed from the
				// stream but is not stored in this string.

    OTC_String&		readToken(istream& theStream);
				// Leading whitespace in <theStream> is
				// skipped. Characters are then read from
				// <theStream>, replacing the contents of
				// this string in the process. Reading
				// of characters from <theStream> will stop
				// when further whitespace is encountered
				// or the end of the stream is reached.

    // The following extends Rogue Wave style interface to reading set
    // number of characters from a stream.

    OTC_String&		readData(istream& theStream, u_int theNum);
				// Reads characters from <theStream>,
				// replacing the contents of this string in
				// the process. Reading of characters from
				// <theStream> will stop when <theNum>
				// characters is read or the end of the
				// stream is reached. You should check the
				// length of the string to determine how
				// many characters were actually read.

    // = ACCESS
    //     It is not guaranteed that pointers returned by the following
    //     functions are valid after subsequent modifications to the
    //     underlying buffer, as modifications may result in new buffer space
    //     being allocated and the previous space being deleted. If a valid
    //     buffer is required, a copy should be made. See <duplicate()> about
    //     making copies.
    //     
    //     The return value of the following functions should not be cast
    //     from non const, to circumvent the restrictions in place to stop
    //     indirect modifications to the buffer, as the changes could effect
    //     more than one class due to the delayed copy mechanism.

    char const*		string() const;
				// Returns a pointer to the underlying buffer
				// used to hold characters. If the string
				// is empty, a pointer to an empty but null
				// terminated string will be returned.

    			operator char const*() const;
				// Returns a pointer to the underlying buffer
				// used to hold characters. If the string
				// is empty, a pointer to an empty but null
				// terminated string will be returned.

    char const*		data() const;
				// Returns a pointer to the underlying buffer
				// used to hold characters. If the string
				// is empty, a pointer to an empty but null
				// terminated string will be returned.
				// This function is only provided to make a
				// transition from using the Rogue Wave
				// string class to this string class easier.
				// You should use <string()> instead.

    char const*		buffer() const;
				// Returns a pointer to the underlying buffer
				// used to hold characters. If the string
				// is empty, a <0> pointer will be returned.

    char*		buffer();
				// Returns a pointer to the underlying buffer
				// used to hold characters. If the string is
				// empty, a <0> pointer will be returned.
				// Note that this variant of the <buffer()>
				// function is only used on non const
				// instances of the class. The returned
				// pointer can be used to directly write into
				// the buffer. It is guaranteed that any
				// delayed copy will be broken before the
				// pointer is returned. If however, you
				// overwrite the end of the buffer and
				// destroy the null terminator, it is your
				// own problem.

    // WARNING. WARNING. WARNING. Ye who play with the following functions
    // better know what they are doing.

    OTC_RString const&	rawString() const
				{ return myData; }
				// Returns a reference to the raw string
				// information.

    OTC_RString&	rawString()
				{ return myData; }
				// Returns a reference to the raw string
				// information.

    // = INDEXING
    //     Whether the object is const or not, it is not possible to
    //     access the null terminator which is automatically added
    //     at the end of the string.

    char		operator[](u_int theIndex) const;
				// Returns the character at the location in
				// this string given by <theIndex>. If
				// <theIndex> is greater than or equal to the
				// length of this string, an exception is
				// raised.

    char&		operator[](u_int theIndex);
				// Returns a reference to the character at
				// the location in this string given by
				// <theIndex>. If <theIndex> is greater than
				// or equal to the length of this string, an
				// exception is raised. When this function is
				// invoked, any delayed copy will be broken
				// first.

    // = RESIZING

    void		length(u_int theLength);
				// Resizes the string. Old data in the string
				// is preserved for whatever range would be
				// valid in the new string. If the length of
				// the string is increased, any new portion
				// of memory is not initialised.

    void		resize(u_int theLength);
				// Resizes the string. Old data in the string
				// is preserved for whatever range would be
				// valid in the new string. If the length of
				// the string is increased, any new portion
				// of memory is not initialised. Kept for
				// backward compatibility with the old
				// <OTC_Buffer>, use <length(u_int)> instead.

    void		capacity(u_int theCapacity);
				// Resizes the underlying buffer to increase
				// its size so that it can hold a string of
				// length <theCapacity>. The length of the
				// current string and its contents are
				// unchanged. If the underlying buffer is
				// shared, this operation will result in the
				// buffer being reallocated, the capacity
				// being the greater of what is required
				// to hold the current string and the value
				// <theCapacity>. If the underlying buffer is
				// not shared, a reallocation will only occur
				// if <theCapacity> is greater than the
				// existing capacity.

    // = COPYING

    char*		duplicate() const;
				// Returns a copy of the underlying buffer.
				// Note that it is the users responsibility
				// to delete the copy. Only enough
				// space to hold the string and the null
				// terminator is allocated. Also, the
				// memory is always obtained from transient
				// memory in situations where a database
				// product is being used.

    OTC_String		clone() const
				{ return *this; }
				// Returns a clone of this string as a new
				// instance of an <OTC_String> object.
				// Actually, the delayed copy mechanism is
				// used, meaning that once a modification
				// is made separate memory for the string
				// will be allocated.

    // = QUERIES

    u_int		length() const;
				// Returns the length of this string.
				// The length does not include the null
				// terminator.

    u_int		size() const;
				// Provided for backward compatibility with
				// <OTC_Buffer> only. Use <length()> instead.

    OTC_Boolean		isEmpty() const;
				// Returns <OTCLIB_TRUE> if the length of
				// this string is <0>.

    OTC_Boolean		isNull() const;
				// Returns <OTCLIB_TRUE> if the length of
				// this string is <0>. This function is only
				// provided to ease the transition from using
				// the Rogue Wave string class to this string
				// class. Use <isEmpty()> instead.

    OTC_Boolean		isUndefined() const;
				// Returns <OTCLIB_TRUE> if the value of
				// the string has not as yet been set. Note
				// that <isEmpty()> will also return
				// <OTCLIB_TRUE> if the string has not as
				// yet been set.

    u_int		capacity() const;
				// Returns the maximum capacity of the buffer
				// being used to hold the current string. The
				// value returned does not include the cell
				// reserved for holding the null terminator.

    // = NULL STRING

    static OTC_String const&	nullString();
				// Returns a reference to an empty string.
				// Can be used when it is necessary to return
				// an empty string where the return type of
				// the function is a reference but you
				// do not have an actual instance to return.

    static OTC_String const&	nullBuffer()
				{ return nullString(); }
				// Returns a reference to an empty string.
				// Can be used when it is necessary to return
				// an empty string where the return type of
				// the function is a reference but you
				// do not have an actual instance to return.
				// Provided only for backward compatibility
				// with <OTC_Buffer>. Use <nullString()>
				// instead.

    static OTC_String const&	undefinedString();
				// Returns a reference to an empty string.
				// This string is different to that returned
				// by <nullString()>. When a string has been
				// set to the value of <undefinedString()>
				// the <isUndefinedString()> function will
				// return true.

    // = REPLACEMENT
    //     In the following functions, <theStart> is an index. Valid values
    //     for <theStart> are <0> through <length()>. Values outside this
    //     range will result in an exception. <theLength> is the number of
    //     characters to be replaced. If <theStart> plus <theLength> is
    //     greater than <length()>, an exception will be raised. If
    //     <theLength> is equal to <0>, insertion is performed at that point.
    //     Both <theStart> and <theLength> equal to <0> is equivalent to
    //     prepending something to the string. <theStart> equal to <length()>
    //     and <theLength> equal to <0> is equivalent to appending something
    //     to the string. Replacing a range of characters with a null
    //     terminated string, a zero number of characters, or a null pointer,
    //     is equivalent to removing that sequence of characters. <theStart>
    //     equal to <0> and <theLength> equal to <length()> is equivalent to
    //     assignment.
    //     
    //     Any functions where arguments of <theStart> and <theLength> are
    //     expected have also been overloaded such that an instance of the
    //     <OTC_Range> object can be supplied instead. In these cases
    //     <theStart> is equivalent to <OTC_Range::lower()> and <theLength>
    //     to <OTC_Range::length()>.

    OTC_String&		replace(
			 u_int theStart,
			 u_int theLength,
			 char theChar,
			 u_int theNum=1
			)
				{
				  _replace(theStart,theLength,
				   theChar,theNum);
				  return *this;
				}
				// Replaces <theLength> characters from
				// <theStart> with <theNum> instances of
				// <theChar>.

    OTC_String&		replace(
			 OTC_Range const& theRange,
			 char theChar,
			 u_int theNum=1
			)
				{
				  _replace(theRange.lower(),
				   theRange.length(),theChar,theNum);
				  return *this;
				}
				// Replaces <theRange> of characters with
				// <theNum> instances of <theChar>.

    OTC_String&		replace(
			 u_int theStart,
			 u_int theLength,
			 char const* theString
			)
				{
				  _replace(theStart,theLength,theString,
				   (theString ? ::strlen(theString) : 0));
				  return *this;
				}
				// Replaces <theLength> characters from
				// <theStart> with <theString>.

    OTC_String&		replace(
			 OTC_Range const& theRange,
			 char const* theString
			)
				{
				  _replace(theRange.lower(),
				   theRange.length(),theString,
				   (theString ? ::strlen(theString) : 0));
				  return *this;
				}
				// Replaces <theRange> of characters with
				// <theString>.

    OTC_String&		replace(
			 u_int theStart,
			 u_int theLength,
			 char const* theString,
			 u_int theNum
			)
				{
				  _replace(theStart,theLength,
				   theString,theNum);
				  return *this;
				}
				// Replaces <theLength> characters from
				// <theStart> with the first <theNum>
				// characters of <theString>.

    OTC_String&		replace(
			 OTC_Range const& theRange,
			 char const* theString,
			 u_int theNum
			)
				{
				  _replace(theRange.lower(),
				   theRange.length(),theString,theNum);
				  return *this;
				}
				// Replaces <theRange> of characters with the
				// first <theNum> characters of <theString>.

    OTC_String&		replace(
			 u_int theStart,
			 u_int theLength,
			 OTC_String const& theString
			)
				{
				  _replace(theStart,theLength,
				   theString.string(),theString.length());
				  return *this;
				}
				// Replaces <theLength> characters from
				// <theStart> with <theString>.

    OTC_String&		replace(
			 OTC_Range const& theRange,
			 OTC_String const& theString
			)
				{
				  _replace(theRange.lower(),
				   theRange.length(),theString.string(),
				   theString.length());
				  return *this;
				}
				// Replaces <theRange> of characters with
				// <theString>.

    OTC_String&		replace(
			 u_int theStart,
			 u_int theLength,
			 OTC_String const& theString,
			 u_int theNum
			)
				{
				  _replace(theStart,theLength,
				   theString,theNum);
				  return *this;
				}
				// Replaces <theLength> characters from
				// <theStart> with the first <theNum>
				// characters of <theString>. If the length
				// of <theString> is less than <theNum> an
				// exception is raised.

    OTC_String&		replace(
			 OTC_Range const& theRange,
			 OTC_String const& theString,
			 u_int theNum
			)
				{
				  _replace(theRange.lower(),
				   theRange.length(),theString,theNum);
				  return *this;
				}
				// Replaces <theRange> of characters with the
				// first <theNum> characters of <theString>.
				// If the length of <theString> is less than
				// <theNum> an exception is raised.

    // = ASSIGNMENT

    OTC_String&		assign(char theChar, u_int theNum=1)
				{
				  _replace(0,length(),theChar,theNum);
				  return *this;
				}
				// Sets this string to be a sequence of
				// <theNum> instances of the character
				// <theChar>. If <theNum> is <0> the result
				// is an empty string.

    OTC_String&		assign(char const* theString)
				{
				  _replace(0,length(),theString,
				   (theString ? ::strlen(theString) : 0));
				  return *this;
				}
				// Sets this string to be <theString>. If
				// <theString> is <0> the result is an empty
				// string.

    OTC_String&		assign(char const* theString, u_int theNum)
				{
				  _replace(0,length(),theString,theNum);
				  return *this;
				}
				// Sets this string to be the first <theNum>
				// characters of <theString>. If <theString>
				// is <0> the result is an empty string.

    OTC_String&		assign(OTC_String const& theString);
				// Sets this string to reference the same
				// character string as <theString>. A copy
				// is only made of the string when an attempt
				// is made to modify it. See description of
				// delayed copy mechanism above.

    OTC_String&		assign(OTC_String const& theString, u_int theNum);
				// Sets this string to be the first <theNum>
				// characters of <theString>. If the length
				// of <theString> is less than <theNum> an
				// exception is raised.

    // Following return a reference to this string so that assignments may be
    // chained. This makes the behaviour consistent with the standard
    // assignment operator.

    OTC_String&		operator=(char theChar)
				{
				  _replace(0,length(),theChar,1);
				  return *this;
				}
				// Sets this string to be the single
				// character <theChar>.

    OTC_String&		operator=(char const* theString)
				{
				  _replace(0,length(),theString,
				   (theString ? ::strlen(theString) : 0));
				  return *this;
				}
				// Sets this string to be <theString>. If
				// <theString> is <0> the result is an empty
				// string.

    OTC_String&		operator=(OTC_String const& theString)
				{ assign(theString); return *this; }
				// Sets this string to reference the same
				// character string as <theString>. A copy
				// is only made of the string when an attempt
				// is made to modify it. See description of
				// delayed copy mechanism above.

    // = INSERTION
    //     Valid indexes for insertion are <0> through <length()>. An
    //     index of <length()> will result in concatenation to this string.
    //     An index outside of this range will result in an exception.

    OTC_String&		insert(u_int theIndex, char theChar, u_int theNum=1)
				{
				  _replace(theIndex,0,theChar,theNum);
				  return *this;
				}
				// Inserts <theChar> before <theIndex>
				// position of this string <theNum> times.

    OTC_String&		insert(u_int theIndex, char const* theString)
				{
				  _replace(theIndex,0,theString,
				   (theString ? ::strlen(theString) : 0));
				  return *this;
				}
				// Inserts <theString> before <theIndex>
				// position of this string.

    OTC_String&		insert(
			 u_int theIndex,
			 char const* theString,
			 u_int theNum
			)
				{
				  _replace(theIndex,0,theString,theNum);
				  return *this;
				}
				// Inserts the first <theNum> characters of
				// <theString> before <theIndex> position of
				// this string.

    OTC_String&		insert(u_int theIndex, OTC_String const& theString)
				{
				  _replace(theIndex,0,theString.string(),
				   theString.length());
				  return *this;
				}
				// Inserts <theString> before <theIndex>
				// position of this string.

    OTC_String&		insert(
			 u_int theIndex,
			 OTC_String const& theString,
			 u_int theNum
			)
				{
				  _replace(theIndex,0,theString,theNum);
				  return *this;
				}
				// Inserts the first <theNum> characters of
				// <theString> before <theIndex> position of
				// this string. If the length of <theString>
				// is less than <theNum> an exception is
				// raised.

    // = EXTENSION

    OTC_String&		append(char theChar, u_int theNum=1)
				{
				  _replace(length(),0,theChar,theNum);
				  return *this;
				}
				// Appends <theChar> to the end of this
				// string <theNum> times.

    OTC_String&		append(char const* theString)
				{
				  _replace(length(),0,theString,
				   (theString ? ::strlen(theString) : 0));
				  return *this;
				}
				// Appends <theString> to the end of this
				// string.

    OTC_String&		append(char const* theString, u_int theNum)
				{
				  _replace(length(),0,theString,theNum);
				  return *this;
				}
				// Appends the first <theNum> characters
				// of <theString> to the end of this string.

    OTC_String&		append(OTC_String const& theString)
				{
				  _replace(length(),0,theString.string(),
				   theString.length());
				  return *this;
				}
				// Appends <theString> to the end of this
				// string.

    OTC_String&		append(OTC_String const& theString, u_int theNum)
				{
				  _replace(length(),0,theString,theNum);
				  return *this;
				}
				// Appends the first <theNum> characters
				// of <theString> to the end of this string.
				// If the length of <theString> is less
				// than <theNum> an exception is raised.

    OTC_String&		operator+=(char theChar)
				{
				  _replace(length(),0,theChar,1);
				  return *this;
				}
				// Appends <theChar> to the end of this
				// string.

    OTC_String&		operator+=(char const* theString)
				{
				  _replace(length(),0,theString,
				   (theString ? ::strlen(theString) : 0));
				  return *this;
				}
				// Appends <theString> to the end of this
				// string.

    OTC_String&		operator+=(OTC_String const& theString)
				{
				  _replace(length(),0,theString.string(),
				   theString.length());
				  return *this;
				}
				// Appends <theString> to the end of this
				// string.

    OTC_String&		prepend(char theChar, u_int theNum=1)
				{
				  _replace(0,0,theChar,theNum);
				  return *this;
				}
				// Prepends <theChar> to the start of this
				// string <theNum> times.

    OTC_String&		prepend(char const* theString)
				{
				  _replace(0,0,theString,
				   (theString ? ::strlen(theString) : 0));
				  return *this;
				}
				// Prepends <theString> to the start of this
				// string.

    OTC_String&		prepend(char const* theString, u_int theNum)
				{
				  _replace(0,0,theString,theNum);
				  return *this;
				}
				// Prepends the first <theNum> characters
				// of <theString> to the start of this string.

    OTC_String&		prepend(OTC_String const& theString)
				{
				  _replace(0,0,theString.string(),
				   theString.length());
				  return *this;
				}
				// Prepends <theString> to the start of this
				// string.

    OTC_String&		prepend(OTC_String const& theString, u_int theNum)
				{
				  _replace(0,0,theString,theNum);
				  return *this;
				}
				// Prepends the first <theNum> characters
				// of <theString> to the start of this string.
				// If the length of <theString> is less
				// than <theNum> an exception is raised.

    // = CONCATENATION

    friend OTC_TString	operator+(OTC_String const& s1, OTC_String const& s2);
				// Returns a string which is the concatenation
				// of <s1> and <s2>.

    friend OTC_TString	operator+(OTC_String const& s1, char const* s2);
				// Returns a string which is the concatenation
				// of <s1> and <s2>.

    friend OTC_TString	operator+(char const* s1, OTC_String const& s2);
				// Returns a string which is the concatenation
				// of <s1> and <s2>.

    friend OTC_TString	operator+(OTC_SObject const& s1, OTC_String const& s2);
				// Returns a string which is the concatenation
				// of <s1> and <s2>.

    friend OTC_TString	operator+(OTC_SObject const& s1, char const* s2);
				// Returns a string which is the concatenation
				// of <s1> and <s2>.

    friend OTC_TString	operator+(OTC_String const& s1, OTC_SObject const& s2);
				// Returns a string which is the concatenation
				// of <s1> and <s2>.

    friend OTC_TString	operator+(OTC_String const& s1, char c1);
				// Returns a string which is the concatenation
				// of <s1> and <c1>.

    friend OTC_TString	operator+(char c1, OTC_String const& s1);
				// Returns a string which is the concatenation
				// of <c1> and <s1>.

    // = REMOVAL

    OTC_String&		remove(u_int theStart, u_int theLength)
				{
				  _replace(theStart,theLength,EOS,0);
				  return *this;
				}
				// Removes from this string <theLength>
				// characters starting at <theStart>.

    OTC_String&		remove(OTC_Range const& theRange)
				{
				  _replace(theRange.lower(),
				   theRange.length(),EOS,0);
				  return *this;
				}
				// Removes from this string <theRange>
				// of characters.

    // = TRUNCATION

    OTC_String&		truncate()
				{ length(0); return *this; }
				// Sets the size of the string back to <0>.

    OTC_String&		truncate(u_int theIndex);
				// Removes those characters from <theIndex>
				// to the end of this string. If <theIndex>
				// is greater than the length of the string
				// an exception is raised.

    // = SUBSTRINGS

    OTC_String		section(u_int theStart, u_int theLength) const
				{ return _section(theStart,theLength); }
				// Returns a new string containing
				// <theLength> characters starting at
				// <theStart>.

    OTC_String		section(OTC_Range const& theRange) const
				{
				  return _section(theRange.lower(),
				   theRange.length());
				}
				// Returns a new string containing
				// those characters in <theRange>.

    OTC_String		between(u_int theStart, u_int theEnd) const
				{
				  return _section(theStart+1,
				   theEnd-(theStart+1));
				}
				// Returns a new string containing the
				// characters between <theStart> and <theEnd>.

    OTC_String		after(u_int theIndex) const
				{
				  return _section(theIndex+1,
				   length()-(theIndex+1));
				}
				// Returns a new string containing characters
				// appearing after <theIndex>.

    OTC_String		from(u_int theIndex) const
				{
				  return _section(theIndex,
				   length()-theIndex);
				}
				// Returns a new string containing characters
				// appearing from <theIndex> through to
				// the end of the string.

    OTC_String		before(u_int theIndex) const
				{ return _section(0,theIndex); }
				// Returns a new string containing characters
				// appearing before <theIndex>.

    OTC_String		through(u_int theIndex) const
				{ return _section(0,theIndex+1); }
				// Returns a new string containing characters
				// appearing from the start of the string
				// through <theIndex>.

    OTC_String		except(u_int theStart, u_int theLength) const
				{ return except_(theStart,theLength); }
				// Returns a new string containing the
				// characters not appearing in the part
				// of the string from <theStart> and
				// with <theLength>.

    OTC_String		except(OTC_Range const& theRange) const
				{
				  return except_(theRange.lower(),
				   theRange.length());
				}
				// Returns a new string containing the
				// characters not appearing in <theRange>
				// of characters.

    // = MODIFIERS

    OTC_String&		upper(u_int theStart, u_int theLength)
				{
				  _upper(theStart,theLength);
				  return *this;
				}
				// Converts any lower case characters in the
				// range defined by <theStart> and
				// <theLength> to upper case.

    OTC_String&		upper(OTC_Range const& theRange)
				{
				  _upper(theRange.lower(),
				   theRange.length());
				  return *this;
				}
				// Converts any lower case characters in
				// <theRange> of characters to upper case.

    OTC_String&		upper(u_int theLength)
				{ _upper(0,theLength); return *this; }
				// Converts any lower case characters in the
				// first <theLength> characters of this string
				// string to upper case.

    OTC_String&		upper()
				{ _upper(0,length()); return *this; }
				// Converts any lower case characters in this
				// string to upper case.

    OTC_String&		lower(u_int theStart, u_int theLength)
				{ _lower(theStart,theLength); return *this; }
				// Converts any upper case characters in the
				// range defined by <theStart> and <theLength>
				// to lower case.

    OTC_String&		lower(OTC_Range const& theRange)
				{
				  _lower(theRange.lower(),theRange.length());
				  return *this;
				}
				// Converts any upper case characters in
				// <theRange> of characters to lower case.

    OTC_String&		lower(u_int theLength)
				{ _lower(0,theLength); return *this; }
				// Converts any upper case characters in the
				// first <theLength> characters of this string
				// string to lower case.

    OTC_String&		lower()
				{ _lower(0,length()); return *this; }
				// Converts any upper case characters in this
				// string to lower case.

    OTC_String&		lchop(u_int theNum=1)
				{
				  _replace(0,theNum,EOS,0);
				  return *this;
				}
				// Removes the first <theNum> characters
				// from this string. If that number of
				// characters does not exist, an exception
				// is raised.

    OTC_String&		rchop(u_int theNum=1)
				{
				  _replace(myData.length()-1,theNum,EOS,0);
				  return *this;
				}
				// Removes the last <theNum> characters
				// from this string. If that number of
				// characters does not exist, an exception
				// is raised.

    OTC_String&		ljustify(u_int theWidth, char theChar=' ');
				// If there are less than <theWidth>
				// characters in the string, instances
				// of <theChar> will be added to the
				// string to fill it out to a width
				// of <theWidth>. The original string
				// will be left justified.

    OTC_String&		rjustify(u_int theWidth, char theChar=' ');
				// If there are less than <theWidth>
				// characters in the string, instances
				// of <theChar> will be added to the
				// string to fill it out to a width
				// of <theWidth>. The original string
				// will be right justified.

    OTC_String&		reverse();
				// Reverses the order of characters in
				// this string.

    // For the following functions, white space is any of SPACE, TAB,
    // RETURN, NEWLINE, FORMFEED, or the vertical tab character.

    OTC_String&		rtrim();
				// Removes trailing white space from
				// this string.

    OTC_String&		rtrim(char theChar);
				// Removes trailing instances of <theChar>
				// from the end of this string.

    OTC_String&		ltrim();
				// Removes leading white space from
				// this string.

    OTC_String&		ltrim(char theChar);
				// Removes leading instances of <theChar>
				// from the end of this string.

    OTC_String&		trim()
				{ return rtrim().ltrim(); }
				// Removes trailing and leading white space
				// from this string.

    // = SEARCHING

    int			index(
			 u_int theIndex,
			 char theChar,
			 u_int theNum=1
			) const
				{ return _index(theIndex,theChar,theNum); }
				// Returns the index of <theNum>'th instance
				// of <theChar> in this string appearing from
				// <theIndex> forward, or <-1> if there were
				// not that many instances of <theChar> in
				// the string.

    int			index(char theChar, u_int theNum=1) const
				{ return _index(0,theChar,theNum); }
				// Returns the index of <theNum>'th instance
				// of <theChar> nearest to the start of this
				// string or <-1> if there were not that
				// many instances of <theChar> in the string.

    int			rindex(char theChar, u_int theNum=1) const
				{ return _rindex(theChar,theNum); }
				// Returns the index of <theNum>'th instance
				// of <theChar> nearest to the end of this
				// string or <-1> if there were not that
				// many instances of <theChar> in the string.

    int			index(
			 u_int theIndex,
			 OTC_BitSetC const& theCharSet,
			 u_int theNum=1
			) const
				{ return _index(theIndex,theCharSet,theNum); }
				// Returns the index of <theNum>'th instance
				// of any character from <theCharSet> in this
				// string appearing from <theIndex> forward,
				// or <-1> if there weren't that many
				// instances of characters from <theCharSet>
				// in the string.

    int			index(
			 OTC_BitSetC const& theCharSet,
			 u_int theNum=1
			) const
				{ return _index(0,theCharSet,theNum); }
				// Returns the index of <theNum>'th instance
				// of any character from <theCharSet> nearest
				// to the start of this string or <-1> if
				// there were not that many instances of
				// characters from <theCharSet> in the string.

    int			rindex(
			 OTC_BitSetC const& theCharSet,
			 u_int theNum=1
			) const
				{ return _rindex(theCharSet,theNum); }
				// Returns the index of <theNum>'th instance
				// of any character from <theCharSet> nearest
				// to the end of this string or <-1> if there
				// were not that many instances of characters
				// from <theCharSet> in the string.

    int			index(
			 u_int theIndex,
			 char const* theString
			) const
				{
				  return _index(theIndex,theString,
				   (theString ? ::strlen(theString) : 0));
				}
				// Returns the index of the first instance
				// of <theString> in this string, appearing
				// beyond <theIndex>. If <theString> doesn't
				// match against this string then <-1> is
				// returned.

    int			index(char const* theString) const
				{
				  return _index(0,theString,
				   (theString ? ::strlen(theString) : 0));
				}
				// Returns the index of the first instance
				// of <theString>, in this string. If
				// <theString> doesn't match against
				// this string then <-1> is returned.

    int			index(
			 u_int theIndex,
			 char const* theString,
			 u_int theLength
			) const
				{ return _index(theIndex,theString,theLength); }
				// Returns the index of the first instance
				// of <theString> in this string, appearing
				// beyond <theIndex>. If <theString> doesn't
				// match against this string then <-1> is
				// returned. <theLength> is the length of
				// <theString>.

    int			index(
			 char const* theString,
			 u_int theLength
			) const
				{ return _index(0,theString,theLength); }
				// Returns the index of the first instance
				// of <theString>, in this string. If
				// <theString> doesn't match against
				// this string then <-1> is returned.
				// <theLength> is the length of <theString>.

    int			index(
			 u_int theIndex,
			 OTC_String const& theString
			) const
				{
				  return _index(theIndex,theString,
				   theString.length());
				}
				// Returns the index of the first instance
				// of <theString> in this string appearing
				// beyond <theIndex>. If <theString> doesn't
				// match against this string then <-1> is
				// returned.

    int			index(OTC_String const& theString) const
				{
				  return _index(0,theString,
				   theString.length());
				}
				// Returns the index of the first instance
				// of <theString> in this string. If
				// <theString> doesn't match against
				// this string then <-1> is returned.

    int			rindex(char const* theString) const
				{
				  return _rindex(theString,
				   (theString ? ::strlen(theString) : 0));
				}
				// Returns the index of the first instance
				// of <theString> nearest to the end of this
				// string or <-1> if there is no match.

    int			rindex(char const* theString, u_int theLength) const
				{ return _rindex(theString,theLength); }
				// Returns the index of the first instance
				// of <theString> nearest to the end of this
				// string or <-1> if there is no match.
				// <theLength> is the length of <theString>.

    int			rindex(OTC_String const& theString) const
				{
				  return _rindex(theString.string(),
				   theString.length());
				}
				// Returns the index of the first instance
				// of <theString> nearest to the end of this
				// string or <-1> if there is no match.

  public:

#if !defined(strchr) && !defined(strrchr)

    // Following exist to get around problems which can arise when the X
    // Windows header files start to play games by defining a macro for
    // "index" and "rindex". Things are okay if the X Windows header files
    // are included before this file as "index" and "rindex" are undefined at
    // the start of this file, however, if the X Windows header files are
    // included after this file, "index" can be redefined to "strchr" and
    // "rindex" to "strrchr" with the above functions becoming inaccessable.
    // We cheat by defining equivalents of the above functions using the
    // names that "index" and "rindex" are defined to. The following
    // functions should never be used explicitly, use those named with
    // "index" and "rindex" instead.

    int			strchr(
			 u_int theIndex,
			 char theChar,
			 u_int theNum=1
			) const
				{ return _index(theIndex,theChar,theNum); }

    int			strchr(char theChar, u_int theNum=1) const
				{ return _index(0,theChar,theNum); }

    int			strrchr(char theChar, u_int theNum=1) const
				{ return _rindex(theChar,theNum); }

    int			strchr(
			 u_int theIndex,
			 OTC_BitSetC const& theCharSet,
			 u_int theNum=1
			) const
				{ return _index(theIndex,theCharSet,theNum); }

    int			strchr(
			 OTC_BitSetC const& theCharSet,
			 u_int theNum=1
			) const
				{ return _index(0,theCharSet,theNum); }

    int			strrchr(
			 OTC_BitSetC const& theCharSet,
			 u_int theNum=1
			) const
				{ return _rindex(theCharSet,theNum); }

    int			strchr(
			 u_int theIndex,
			 char const* theString
			) const
				{
				  return _index(theIndex,theString,
				   (theString ? ::strlen(theString) : 0));
				}

    int			strchr(char const* theString) const
				{
				  return _index(0,theString,
				   (theString ? ::strlen(theString) : 0));
				}

    int			strchr(
			 u_int theIndex,
			 char const* theString,
			 u_int theLength
			) const
				{ return _index(theIndex,theString,theLength); }

    int			strchr(
			 char const* theString,
			 u_int theLength
			) const
				{ return _index(0,theString,theLength); }

    int			strchr(
			 u_int theIndex,
			 OTC_String const& theString
			) const
				{
				  return _index(theIndex,theString,
				   theString.length());
				}

    int			strchr(OTC_String const& theString) const
				{
				  return _index(0,theString,
				   theString.length());
				}

    int			strrchr(char const* theString) const
				{
				  return _rindex(theString,
				   (theString ? ::strlen(theString) : 0));
				}

    int			strrchr(char const* theString, u_int theLength) const
				{ return _rindex(theString,theLength); }

    int			strrchr(OTC_String const& theString) const
				{
				  return _rindex(theString.string(),
				   theString.length());
				}

#endif

    // = RANKING
    //     In the following rank functions, a return value of <-1> indicates
    //     that this string ranks less than that being compared against,
    //     using a lexographical comparison. A value of <0> indicates that
    //     the strings are equivalent, and a value of <1> indicates that
    //     this string ranks greater than that being compared against.

    int			rank(OTC_String const& theString) const;
				// Performs a comparison between this string
				// and <theString> and return a value
				// corresponding to how the two strings
				// rank.

    int			rank(char const* theString) const
				{
				  return rank(myData.string(),
				   myData.length(),theString,
				   (theString ? ::strlen(theString) : 0));
				}
				// Performs a comparison between this string
				// and <theString> and return a value
				// corresponding to how the two strings
				// rank.

    int			rank(char const* theString, u_int theLength) const
				{
				  return rank(myData.string(),myData.length(),
				   theString,theLength);
				}
				// Performs a comparison between this string
				// and the first <theLength> characters of
				// <theString> and return a value
				// corresponding to how the two strings rank.

    int			rank(
			 OTC_String const& theString,
			 u_int theLength
			) const;
				// Performs a comparison between this string
				// and the first <theLength> characters of
				// <theString> and return a value
				// corresponding to how the two strings rank.
				// If <theString> is less than <theLength>
				// characters in length, an exception is
				// raised.

    // = HASHING

    int			hash() const
				{
				  return otclib_hash(myData.string(),
				   myData.length());
				}
				// Returns a hash value for this string.

    // = COMPARISION
    //     Note that for the <compare()> functions, a value of <0> for
    //     <theString> is not interpreted as being the same as a null
    //     terminated string. As a result, if <theString> is <0> then
    //     <OTCLIB_FALSE> will always be returned, even if the length of this
    //     string is <0>. Also, each of the <compare()> functions accept an
    //     optional argument to indicate whether an exact match is expected
    //     or whether case can be ignored. The two corresponding values to
    //     indicate this are <OTCLIB_EXACTMATCH> and <OTCLIB_IGNORECASE> with
    //     the default being an exact match.

    OTC_Boolean		compare(
			 u_int theIndex,
			 char const* theString,
			 OTC_CmpType theType=OTCLIB_EXACTMATCH
			) const;
				// Returns <OTCLIB_TRUE> if <theString> is
				// equivalent to the portion of this string
				// starting at <theIndex> through to the end
				// of this string. If <theIndex> is greater
				// than the length of this string an
				// exception is raised.

    OTC_Boolean		compare(
			 char const* theString,
			 OTC_CmpType theType=OTCLIB_EXACTMATCH
			) const
				{ return compare(0,theString,theType); }
				// Returns <OTCLIB_TRUE> if <theString> is
				// equivalent to this string.

    OTC_Boolean		compare(
			 u_int theIndex,
			 char const* theString,
			 u_int theNum,
			 OTC_CmpType theType=OTCLIB_EXACTMATCH
			) const;
				// Returns <OTCLIB_TRUE> if <theNum>
				// characters of this string starting at
				// <theIndex>, are the same as the first
				// <theNum> characters of <theString>. If
				// <theIndex> is greater than the length of
				// this string an exception is raised. If
				// <theIndex> plus <theNum> is greater than
				// the length of this string, <OTCLIB_FALSE>
				// will always be returned.

    OTC_Boolean		compare(
			 u_int theIndex,
			 OTC_String const& theString,
			 OTC_CmpType theType=OTCLIB_EXACTMATCH
			) const;
				// Returns <OTCLIB_TRUE> if <theString> is
				// equivalent to the portion of this string
				// starting at <theIndex> through to the
				// end of this string. If <theIndex> is
				// greater than the length of this string
				// an exception is raised.

    OTC_Boolean		compare(
			 OTC_String const& theString,
			 OTC_CmpType theType=OTCLIB_EXACTMATCH
			) const
				{ return compare(0,theString,theType); }
				// Returns <OTCLIB_TRUE> if <theString> is
				// equivalent to this string.

    OTC_Boolean		compare(
			 u_int theIndex,
			 OTC_String const& theString,
			 u_int theNum,
			 OTC_CmpType theType=OTCLIB_EXACTMATCH
			) const;
				// Returns <OTCLIB_TRUE> if <theNum>
				// characters of this string starting at
				// <theIndex> are the same as the first
				// <theNum> characters of <theString>. If
				// <theIndex> is greater than the length of
				// this string, or <theNum> is greater than
				// the length of <theString>, an exception is
				// raised. If <theIndex> plus <theNum> is
				// greater than the length of this string,
				// <OTCLIB_FALSE> will always be returned.

    // Following all return <OTCLIB_TRUE> if the particular comparison is
    // satisfied.

    OTC_Boolean		operator==(char c1) const
				{
				  return rank(myData.string(),myData.length(),
				   &c1,1) == 0;
				}

    OTC_Boolean		operator==(char const* s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2,(s2 ? ::strlen(s2) : 0)) == 0;
				}

    OTC_Boolean		operator==(OTC_String const& s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2.string(),s2.length()) == 0;
				}

    OTC_Boolean		operator!=(char c1) const
				{
				  return rank(myData.string(),myData.length(),
				   &c1,1) != 0;
				}

    OTC_Boolean		operator!=(char const* s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2,(s2 ? ::strlen(s2) : 0)) != 0;
				}

    OTC_Boolean		operator!=(OTC_String const& s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2.string(),s2.length()) != 0;
				}

    OTC_Boolean		operator<=(char c1) const
				{
				  return rank(myData.string(),myData.length(),
				   &c1,1) <= 0;
				}

    OTC_Boolean		operator<=(char const* s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2,(s2 ? ::strlen(s2) : 0)) <= 0;
				}

    OTC_Boolean		operator<=(OTC_String const& s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2.string(),s2.length()) <= 0;
				}

    OTC_Boolean		operator>=(char c1) const
				{
				  return rank(myData.string(),myData.length(),
				   &c1,1) >= 0;
				}

    OTC_Boolean		operator>=(char const* s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2,(s2 ? ::strlen(s2) : 0)) >= 0;
				}

    OTC_Boolean		operator>=(OTC_String const& s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2.string(),s2.length()) >= 0;
				}

    OTC_Boolean		operator<(char c1) const
				{
				  return rank(myData.string(),myData.length(),
				   &c1,1) < 0;
				}

    OTC_Boolean		operator<(char const* s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2,(s2 ? ::strlen(s2) : 0)) < 0;
				}

    OTC_Boolean		operator<(OTC_String const& s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2.string(),s2.length()) < 0;
				}

    OTC_Boolean		operator>(char c1) const
				{
				  return rank(myData.string(),myData.length(),
				   &c1,1) > 0;
				}

    OTC_Boolean		operator>(char const* s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2,(s2 ? ::strlen(s2) : 0)) > 0;
				}

    OTC_Boolean		operator>(OTC_String const& s2) const
				{
				  return rank(myData.string(),myData.length(),
				   s2.string(),s2.length()) > 0;
				}

    friend OTC_Boolean	operator==(char c1, OTC_String const& s2)
				{
				  return OTC_String::rank(&c1,1,
				   s2.string(),s2.length()) == 0;
				}

    friend OTC_Boolean	operator==(char const* s1, OTC_String const& s2)
				{
				  return OTC_String::rank(s1,
				   (s1 ? ::strlen(s1) : 0),
				   s2.string(),s2.length()) == 0;
				}

    friend OTC_Boolean	operator!=(char c1, OTC_String const& s2)
				{
				  return OTC_String::rank(&c1,1,
				   s2.string(),s2.length()) != 0;
				}

    friend OTC_Boolean	operator!=(char const* s1, OTC_String const& s2)
				{
				  return OTC_String::rank(s1,
				   (s1 ? ::strlen(s1) : 0),
				   s2.string(),s2.length()) != 0;
				}

    friend OTC_Boolean	operator<=(char c1, OTC_String const& s2)
				{
				  return OTC_String::rank(&c1,1,
				   s2.string(),s2.length()) <= 0;
				}

    friend OTC_Boolean	operator<=(char const* s1, OTC_String const& s2)
				{
				  return OTC_String::rank(s1,
				   (s1 ? ::strlen(s1) : 0),
				   s2.string(),s2.length()) <= 0;
				}

    friend OTC_Boolean	operator>=(char c1, OTC_String const& s2)
				{
				  return OTC_String::rank(&c1,1,
				   s2.string(),s2.length()) >= 0;
				}

    friend OTC_Boolean	operator>=(char const* s1, OTC_String const& s2)
				{
				  return OTC_String::rank(s1,
				   (s1 ? ::strlen(s1) : 0),
				   s2.string(),s2.length()) >= 0;
				}

    friend OTC_Boolean	operator>(char c1, OTC_String const& s2)
				{
				  return OTC_String::rank(&c1,1,
				   s2.string(),s2.length()) > 0;
				}

    friend OTC_Boolean	operator>(char const* s1, OTC_String const& s2)
				{
				  return OTC_String::rank(s1,
				   (s1 ? ::strlen(s1) : 0),
				   s2.string(),s2.length()) > 0;
				}

    friend OTC_Boolean	operator<(char c1, OTC_String const& s2)
				{
				  return OTC_String::rank(&c1,1,
				   s2.string(),s2.length()) < 0;
				}

    friend OTC_Boolean	operator<(char const* s1, OTC_String const& s2)
				{
				  return OTC_String::rank(s1,
				   (s1 ? ::strlen(s1) : 0),
				   s2.string(),s2.length()) < 0;
				}

  private:

			OTC_String(u_int theCapacity);
				// Creates an empty string with enough
				// underlying capacity to hold a string of
				// size <theCapacity>. The string will
				// initially have zero length but will not
				// register as being undefined. Made private
				// as no longer provided. Being private
				// should result in compiler error.

    // Privatise the following functions so that we do not get wierd things
    // happening in code unexpectedly. Cannot block implicit use of char*
    // converison operator though.

    void		operator*() const {}

    void		operator~() const {}

    void		operator!() const {}

    // Implementation functions. What they do is easily derived by looking
    // at public equivalents.

    void		_replace(
			 int theStart,
			 u_int theLength,
			 char theChar,
			 u_int theNum=1
			);

    void		_replace(
			 int theStart,
			 u_int theLength,
			 char const* theString,
			 u_int theNum
			);

    void		_replace(
			 int theStart,
			 u_int theLength,
			 OTC_String const& theString,
			 u_int theNum
			);

    OTC_String		_section(int theStart, u_int theLength) const;

    OTC_String		except_(int theStart, u_int theLength) const;

    void		_upper(int theStart, u_int theLength);

    void		_lower(int theStart, u_int theLength);

    int			_index(
			 u_int theIndex,
			 char theChar,
			 u_int theNum=1
			) const;

    int			_rindex(
			 char theChar,
			 u_int theNum=1
			) const;

    int			_index(
			 u_int theIndex,
			 OTC_BitSetC const& theCharSet,
			 u_int theNum=1
			) const;

    int			_rindex(
			 OTC_BitSetC const& theCharSet,
			 u_int theNum=1
			) const;

    int			_index(
			 u_int theIndex,
			 char const* theString,
			 u_int theLength
			) const;

    int			_rindex(
			 char const* theString,
			 u_int theLength
			) const;

    static OTC_NRMutex	_mutex;
				// Lock for threads.

    static OTC_String const*	globNullString;
				// String returned by <nullString()>.

    static OTC_String const*	globUndefinedString;
				// String returned by <undefinedString()>.

    static char const	globCharMap[];
				// Table used for doing case insensitive
				// string comparisons.

    OTC_RString		myData;
				// The raw string data.
};

/* ========================================================================= */

#ifdef __OSE_TEMPLATES__
OSE_MARK_TEMPLATE OTC_RankActions<OTC_String>;
OSE_MARK_TEMPLATE OTC_HashActions<OTC_String>;
#endif

#if defined(CXX_OS)
typedef OTC_RankActions<OTC_String> otc_text_string_hh_typedef1;
#pragma ObjectStore exclude_instantiations OTC_RankActions<OTC_String>;
typedef OTC_HashActions<OTC_String> otc_text_string_hh_typedef2;
#pragma ObjectStore exclude_instantiations OTC_HashActions<OTC_String>;
#endif

#if !defined(CXX_OS)
class OTC_RankActions<OTC_String>
{
  public:
    static int		rank(OTC_String const& s1, OTC_String const& s2)
				{ return s1.rank(s2); }
};

class OTC_HashActions<OTC_String>
{
  public:
    static int		hash(OTC_String const& s)
				{ return s.hash(); }
};
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_TEXT_STRING_HH */
