#ifndef OTC_TEXT_SYMBOL_HH
#define OTC_TEXT_SYMBOL_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     text/symbol.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 TELSTRA CORPORATION LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/text/string.hh>
#include <OTC/collctn/rankactn.hh>
#include <OTC/collctn/hashactn.hh>
#include <OTC/collctn/avlnode.hh>
#include <OTC/memory/cmmnpool.hh>
#include <OTC/thread/nrmutex.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/text/symbol.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Arena;
class OTC_AVLTree;

class OTC_SymbolData : public OTC_AVLNode, public OTC_StringData
    // = TITLE
    //	   AVL tree node for holding symbol length and value.
    //
    // = CLASS TYPE
    //	   Concrete
{
  public:

     friend class	OTC_Symbol;

			~OTC_SymbolData();

#if !defined(ENV_NOCOMMONPOOL)
    void*		operator new(size_t theSize)
				{ return OTC_CommonPool::allocate(theSize); }

    void		operator delete(void* theMem, size_t theSize)
				{ OTC_CommonPool::release(theMem,theSize); }
#endif

    // = INITIALISATION

			OTC_SymbolData(char* theString,u_int theLength);
				// Creates a node to hold the symbol
				// <theString> with length <theLength>.
				// This class is not responsible for
				// deleting <theString>.

    char const*		string() const
				{ return OTC_StringData::data; }
				// Returns the string held.

    u_int		length() const
				{ return OTC_StringData::length; }
				// Returns the length of the symbol
				// data.

    OTC_StringData*	data()
				{ return this; }
				// Returns a pointer to the underlying
				// data structure containing the string.

  private:

			OTC_SymbolData(OTC_SymbolData const&);
				// Do not define an implementation for this.

    OTC_SymbolData&	operator=(OTC_SymbolData const&);
				// Do not define an implementation for this.
};

class OTC_Symbol : public OTC_SObject
    // = TITLE
    //	   Symbol class for a string identifier.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   The <OTC_Symbol> class is an alternative to using the <OTC_String>
    //	   class as an identifier. The benefits of this class are that it
    //	   doesn't actually keep a copy of the string in the class. Instead,
    //	   all strings are kept in a central symbol table with this class
    //	   holding a pointer to the central string. This allows equality tests
    //	   to be very quick as it requires a pointer comparison only. Also,
    //	   as there is only one copy of the string in a central symbol table
    //	   excess memory will not be consumed due to duplicate copies of the
    //	   string.
    //	   
    // = NOTES
    //	   Instances of <OTC_Symbol> cannot be stored in an ObjectStore
    //	   database.
    //	   
    // = SEE ALSO
    //	   <OTC_String>
{
  public:

			~OTC_Symbol();

    // = CONSTRUCTION

			OTC_Symbol();
				// Creates an instance of a symbol class
				// to represent the null string.

			OTC_Symbol(OTC_Symbol const& theSymbol)
			  : myInfo(theSymbol.myInfo) {}
				// Creates an instance of the symbol
				// class which is the same as <theSymbol>.

			OTC_Symbol(char const* theString);
				// Creates an instance of the symbol
				// class to represent <theString>.

			OTC_Symbol(char const* theString, u_int theLength);
				// Creates an instance of the symbol
				// class to represent the first <theLength>
				// characters of <theString>.

			OTC_Symbol(char theChar, u_int theNum=1);
				// Creates an instance of the symbol
				// class to represent a string containing
				// <theNum> instances of <theString>.

			OTC_Symbol(OTC_String const& theString);
				// Creates an instance of the symbol class to
				// represent <theString>. If <theString>
				// represents a symbol, ie., it was created
				// through a conversion of a symbol to a
				// string, the conversion back to a symbol is
				// trivial provided that <theString> hadn't
				// been used in such a way that the delayed
				// copy linkage was broken.

			OTC_Symbol(
			 OTC_String const& theString,
			 u_int theLength
			);
				// Creates an instance of the symbol
				// class to represent the first <theLength>
				// characters of <theString>.

			OTC_Symbol(OTC_TString const& theString);
				// Creates an instance of the symbol class to
				// represent <theString>.

    OTC_Symbol&		operator=(OTC_Symbol const& theSymbol)
				{ myInfo = theSymbol.myInfo; return *this; }
				// Sets this symbol to be the same as
				// <theSymbol>.

    // = QUERY

    OTC_Boolean		isEmpty() const
				{ return myInfo->length() == 0; }
				// Returns <OTCLIB_TRUE> if this symbol
				// is that of an empty string.

    char		operator[](u_int theIndex) const;
				// Returns the character at the location in
				// the string for this symbol, given by
				// <theIndex>. If <theIndex> is greater than
				// or equal to the length of the string for
				// this symbol, an exception is raised.

    char const*		string() const
				{ return myInfo->string(); }
				// Returns a pointer to the character
				// string for the symbol.

    u_int		length() const
				{ return myInfo->length(); }
				// Returns the length of the character
				// string for the symbol.

    static OTC_Boolean	exists(char const* theString)
				{
				  return exists(theString,
				   (theString ? ::strlen(theString) : 0));
				}
				// Returns <OTCLIB_TRUE> if <theString>
				// already exists as a symbol. The null
				// symbol always exists.

    static OTC_Boolean	exists(char const* theString, u_int theLength);
				// Returns <OTCLIB_TRUE> if the symbol given
				// by the first <theLength> characters of
				// <theString> already exists as a symbol.
				// The null symbol always exists.

    // = INEQUALITY

    OTC_Boolean		operator==(OTC_Symbol const& theSymbol) const
				{ return myInfo == theSymbol.myInfo; }
				// Returns <OTCLIB_TRUE> if this symbol
				// is the same as <theSymbol>.

    OTC_Boolean		operator!=(OTC_Symbol const& theSymbol) const
				{ return myInfo != theSymbol.myInfo; }
				// Returns <OTCLIB_TRUE> if this symbol
				// is not the same as <theSymbol>.

    // Following versions are provided so that comparisons against
    // instances of <OTC_String> and <char const*> will not result in
    // an entry being added to the symbol database.

    OTC_Boolean		operator==(OTC_String const& theString) const
				{
				  return OTC_String::rank(myInfo->string(),
				   myInfo->length(),theString.string(),
				   theString.length()) == 0;
				}
				// Returns <OTCLIB_TRUE> if this symbol
				// is the same as <theString>.

    OTC_Boolean		operator!=(OTC_String const& theString) const
				{
				  return OTC_String::rank(myInfo->string(),
				   myInfo->length(),theString.string(),
				   theString.length()) != 0;
				}
				// Returns <OTCLIB_TRUE> if this symbol
				// is not the same as <theString>.

    friend OTC_Boolean	operator==(
			 OTC_String const& theString,
			 OTC_Symbol const& theSymbol
			)
				{ return theSymbol == theString; }
				// Returns <OTCLIB_TRUE> if <theSymbol>
				// is the same as <theString>.

    friend OTC_Boolean	operator!=(
			 OTC_String const& theString,
			 OTC_Symbol const& theSymbol
			)
				{ return theSymbol != theString; }
				// Returns <OTCLIB_TRUE> if <theSymbol>
				// is not the same as <theString>.

    OTC_Boolean		operator==(char const* theString) const
				{
				  return OTC_String::rank(myInfo->string(),
				   myInfo->length(),theString,
				   (theString?strlen(theString):0)) == 0;
				}
				// Returns <OTCLIB_TRUE> if this symbol
				// is the same as <theString>.

    OTC_Boolean		operator!=(char const* theString) const
				{
				  return OTC_String::rank(myInfo->string(),
				   myInfo->length(),theString,
				   (theString?strlen(theString):0)) != 0;
				}
				// Returns <OTCLIB_TRUE> if this symbol
				// is not the same as <theString>.

    friend OTC_Boolean	operator==(
			 char const* theString,
			 OTC_Symbol const& theSymbol
			)
				{ return theSymbol == theString; }
				// Returns <OTCLIB_TRUE> if <theSymbol>
				// is the same as <theString>.

    friend OTC_Boolean	operator!=(
			 char const* theString,
			 OTC_Symbol const& theSymbol
			)
				{ return theSymbol != theString; }
				// Returns <OTCLIB_TRUE> if <theSymbol>
				// is not the same as <theString>.

    // = RANKING

    int			rank(OTC_Symbol const& theSymbol) const
				{ return int(myInfo) - int(theSymbol.myInfo); }
				// Ranks this symbol against <theSymbol>.

    // = HASHING

    int			hash() const
				{ return int(myInfo); }
				// Returns a hash value for the symbol.

    // = NULL SYMBOL

    static OTC_Symbol const&	nullSymbol();
				// Returns a reference to the null symbol.

    // = DISPLAY

    friend ostream&	operator<<(ostream& outs, OTC_Symbol const& theSymbol);
				// Prints <theSymbol> on the stream <outs>.

  protected:

    // = CONVERSION

    OTC_RString		rawString() const;
				// Returns the string which the symbol
				// represents.

  private:

    OTC_SymbolData*	install(char const* theString, u_int theLength);
				// Looks in the symbol table for <theString>
				// and returns the symbol data if it
				// exists. If <theString> doesn't exist a
				// copy is made of it, placed in the table
				// and a pointer to the newly create symbol
				// record returned.

    OTC_SymbolData*	myInfo;
				// Pointer to symbol data.

    static OTC_NRMutex	_mutex;
				// Lock for threads.

    static OTC_Arena*	globArena;
				// Arena from which memory is allocated.

    static OTC_AVLTree* globTree;
				// Look up tree for string representation
				// of symbols.

    static OTC_SymbolData*	globNullInfo;
				// Symbol data for null string.

    static OTC_Symbol*	globNullSymbol;
				// Symbol for null string.
};

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
OSE_MARK_TEMPLATE OTC_RankActions<OTC_Symbol>;
OSE_MARK_TEMPLATE OTC_HashActions<OTC_Symbol>;
#endif

#if defined(CXX_OS)
typedef OTC_RankActions<OTC_Symbol> otc_text_symbol_hh_typedef1;
#pragma ObjectStore exclude_instantiations OTC_RankActions<OTC_Symbol>;
typedef OTC_HashActions<OTC_Symbol> otc_text_symbol_hh_typedef2;
#pragma ObjectStore exclude_instantiations OTC_HashActions<OTC_Symbol>;
#endif

#if !defined(CXX_OS)
class OTC_RankActions<OTC_Symbol>
{
  public:
    static int		rank(OTC_Symbol const& s1, OTC_Symbol const& s2)
				{ return s1.rank(s2); }
};

class OTC_HashActions<OTC_Symbol>
{
  public:
    static int		hash(OTC_Symbol const& s)
				{ return s.hash(); }
};
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_TEXT_SYMBOL_HH */
