#ifndef OTC_TYPES_DURATION_HH
#define OTC_TYPES_DURATION_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     TYPES/duration.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/hashactn.hh>
#include <OTC/collctn/rankactn.hh>
#include <OTC/text/string.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/types/duration.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Duration
    // = TITLE
    //	   Maintains information about the duration between two times.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   Maintains information about the duration between two times.
    //	   Resolution is one second. Durations can be specified in terms
    //	   of days, hours, minutes and seconds.
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    // = INITIALISATION

			OTC_Duration();
				// Creates a duration of zero length.

			OTC_Duration(
			 int theDays,
			 int theHours=0,
			 int theMinutes=0,
			 int theSeconds=0
			);
				// Creates a duration of length corresponding
				// to the arguments <theDays>, <theHours>,
				// <theMinutes> and <theSeconds>. The last
				// three arguments are optional. The actual
				// duration is calculated by adding the
				// values according to their respective
				// weights. For example <1> day and <-1> hour
				// will result in <23> hours.

			OTC_Duration(OTC_Duration const& theDuration);
				// Creates a duration of length
				// <theDuration>.

    OTC_Duration&	operator=(OTC_Duration const& theDuration)
				{ setDuration(theDuration); return *this; }
				// Sets the duration to <theDuration>.

    // = QUERY

    OTC_Boolean		isZeroLength() const;
				// Returns <OTCLIB_TRUE> if the duration
				// is of zero length.

    OTC_Boolean		isPositiveDuration() const
				{ return mySign > 0; }
				// Returns <OTCLIB_TRUE> if the duration
				// is positive in direction. A zero length
				// duration is take to be a duration in the
				// positive direction.

    OTC_Boolean		isNegativeDuration() const
				{ return mySign < 0; }
				// Returns <OTCLIB_TRUE> if the duration
				// is negative in direction.

    int			sign() const
				{ return mySign; }
				// Returns the direction of the duration as
				// an integer. Value of <1> is returned if
				// duration is positive in direction and a
				// value of <-1> if the duration is negative
				// in duration.

    //	   Component parts of duration will always be positive values.
    //	   To determine if the duration is in the negative direction,
    //	   use the <sign()> member function.

    int			days() const
				{ return myDays; }
				// Returns the day component of the duration.

    int			hours() const
				{ return myHours; }
				// Returns the hour component of the duration.

    int			minutes() const
				{ return myMinutes; }
				// Returns the minute component of the
				// duration.

    int			seconds() const
				{ return mySeconds; }
				// Returns the second component of the
				// duration.

    OTC_Duration	plusSeconds(int theNum) const
				{
				  OTC_Duration theDuration(0,0,0,theNum);
				  return plusDuration(theDuration);
				}
				// Returns this duration with <theNum>
				// seconds added.

    OTC_Duration	minusSeconds(int theNum) const
				{
				  OTC_Duration theDuration(0,0,0,theNum);
				  return minusDuration(theDuration);
				}
				// Returns this duration with <theNum>
				// seconds subtracted.

    OTC_Duration	plusMinutes(int theNum) const
				{
				  OTC_Duration theDuration(0,0,theNum,0);
				  return plusDuration(theDuration);
				}
				// Returns this duration with <theNum>
				// minutes added.

    OTC_Duration	minusMinutes(int theNum) const
				{
				  OTC_Duration theDuration(0,0,theNum,0);
				  return minusDuration(theDuration);
				}
				// Returns this duration with <theNum>
				// minutes subtracted.

    OTC_Duration	plusHours(int theNum) const
				{
				  OTC_Duration theDuration(0,theNum,0,0);
				  return plusDuration(theDuration);
				}
				// Returns this duration with <theNum> hours
				// added.

    OTC_Duration	minusHours(int theNum) const
				{
				  OTC_Duration theDuration(0,theNum,0,0);
				  return minusDuration(theDuration);
				}
				// Returns this duration with <theNum> hours
				// subtracted.

    OTC_Duration	plusDays(int theNum) const
				{
				  OTC_Duration theDuration(theNum,0,0,0);
				  return plusDuration(theDuration);
				}
				// Returns this duration with <theNum> days
				// added.

    OTC_Duration	minusDays(int theNum) const
				{
				  OTC_Duration theDuration(theNum,0,0,0);
				  return minusDuration(theDuration);
				}
				// Returns this duration with <theNum> days
				// subtracted.

    OTC_Duration	plusWeeks(int theNum) const
				{
				  OTC_Duration theDuration(7*theNum,0,0,0);
				  return plusDuration(theDuration);
				}
				// Returns this duration with <theNum> weks
				// added.

    OTC_Duration	minusWeeks(int theNum) const
				{
				  OTC_Duration theDuration(7*theNum,0,0,0);
				  return minusDuration(theDuration);
				}
				// Returns this duration with <theNum> weeks
				// subtracted.

    OTC_Duration	plusDuration(OTC_Duration const& theDuration) const;
				// Returns this duration with <theDuration>
				// added.

    OTC_Duration	minusDuration(OTC_Duration const& theDuration) const;
				// Returns this duration with <theDuration>
				// subtracted.

    OTC_Duration	plusDuration(
			 int theDays,
			 int theHours,
			 int theMinutes,
			 int theSeconds
			) const;
				// Returns this duration with the duration
				// corresponding to that defined by
				// <theDays>, <theHours>, <theMinutes> and
				// <theSeconds> added.

    OTC_Duration	minusDuration(
			 int theDays,
			 int theHours,
			 int theMinutes,
			 int theSeconds
			) const;
				// Returns this duration with the duration
				// corresponding to that defined by
				// <theDays>, <theHours>, <theMinutes> and
				// <theSeconds> subtracted.

    // = TEXTUAL REPRESENTATION
    //    The sign is not considered when converting duration to a string
    //    representation.

    OTC_String          asString() const;
				// Returns the duration as a string. The
				// format used for the duration when
				// converting it to a string, is that defined
				// by ISO 8601:1988.

    friend ostream&     operator<<(
				 ostream& theStream,
				 OTC_Duration const& theDuration
				);
				// Outputs the duration onto <theStream>. The
				// format used for the duration when
				// displaying it, is that defined by ISO
				// 8601:1988.

    // = MODIFIERS

    void		setDuration(OTC_Duration const& theDuration);
				// Sets this duration to <theDuration>.

    void		setDuration(
			 int theDays,
			 int theHours=0,
			 int theMinutes=0,
			 int theSeconds=0
			);
				// Sets the duration to a length
				// corresponding to the arguments <theDays>,
				// <theHours>, <theMinutes> and <theSeconds>.
				// The last three arguments are optional. The
				// actual duration is calculated by adding
				// the values according to their respective
				// weights. For example <1> day and <-1> hour
				// will result in <23> hours.

    OTC_Duration&	addSeconds(int theNum)
				{
				  OTC_Duration theDuration(0,0,0,theNum);
				  return addDuration(theDuration);
				}
				// Adds <theNum> seconds to this duration.

    OTC_Duration&	subtractSeconds(int theNum)
				{
				  OTC_Duration theDuration(0,0,0,theNum);
				  return subtractDuration(theDuration);
				}
				// Subtracts <theNum> seconds from this
				// duration.

    OTC_Duration&	addMinutes(int theNum)
				{
				  OTC_Duration theDuration(0,0,theNum,0);
				  return addDuration(theDuration);
				}
				// Adds <theNum> minutes to this duration.

    OTC_Duration&	subtractMinutes(int theNum)
				{
				  OTC_Duration theDuration(0,0,theNum,0);
				  return subtractDuration(theDuration);
				}
				// Subtracts <theNum> minutes from this
				// duration.

    OTC_Duration&	addHours(int theNum)
				{
				  OTC_Duration theDuration(0,theNum,0,0);
				  return addDuration(theDuration);
				}
				// Adds <theNum> hours to this duration.

    OTC_Duration&	subtractHours(int theNum)
				{
				  OTC_Duration theDuration(0,theNum,0,0);
				  return subtractDuration(theDuration);
				}
				// Subtracts <theNum> hours from this
				// duration.

    OTC_Duration&	addDays(int theNum)
				{
				  OTC_Duration theDuration(theNum,0,0,0);
				  return addDuration(theDuration);
				}
				// Adds <theNum> days to this duration.

    OTC_Duration&	subtractDays(int theNum)
				{
				  OTC_Duration theDuration(theNum,0,0,0);
				  return subtractDuration(theDuration);
				}
				// Subtracts <theNum> days from this
				// duration.

    OTC_Duration&	addWeeks(int theNum)
				{
				  OTC_Duration theDuration(7*theNum,0,0,0);
				  return addDuration(theDuration);
				}
				// Adds <theNum> weeks to this duration.

    OTC_Duration&	subtractWeeks(int theNum)
				{
				  OTC_Duration theDuration(7*theNum,0,0,0);
				  return subtractDuration(theDuration);
				}
				// Subtracts <theNum> weeks from this
				// duration.

    OTC_Duration&	addDuration(OTC_Duration const& theDuration);
				// Adds <theDuration> to this duration.

    OTC_Duration&	subtractDuration(OTC_Duration const& theDuration);
				// Subtracts <theDuration> from this duration.

    OTC_Duration&	addDuration(
			 int theDays,
			 int theHours,
			 int theMinutes,
			 int theSeconds
			);
				// Adds the duration corresponding to that
				// defined by <theDays>, <theHours>,
				// <theMinutes> and <theSeconds> to this
				// duration.

    OTC_Duration&	subtractDuration(
			 int theDays,
			 int theHours,
			 int theMinutes,
			 int theSeconds
			);
				// Subtracts the duration corresponding to
				// that defined by <theDays>, <theHours>,
				// <theMinutes> and <theSeconds> from this
				// duration.

    // = COMPARISONS
    //	   The following do not take into consideration the sign of the
    //	   duration. They compare the period of the duration only.

    OTC_Boolean		operator==(OTC_Duration const& theDuration) const;
				// Returns <OTCLIB_TRUE> if the period of
				// this duration is the same as that of
				// <theDuration>.

    OTC_Boolean		operator!=(OTC_Duration const& theDuration) const;
				// Returns <OTCLIB_TRUE> if the period of
				// this duration is not the same as that of
				// <theDuration>.

    OTC_Boolean		operator<=(OTC_Duration const& theDuration) const;
				// Returns <OTCLIB_TRUE> if the period of
				// this duration is less than or equal to
				// that of <theDuration>.

    OTC_Boolean		operator<(OTC_Duration const& theDuration) const;
				// Returns <OTCLIB_TRUE> if the period of
				// this duration is less than to that of
				// <theDuration>.

    OTC_Boolean		operator>=(OTC_Duration const& theDuration) const;
				// Returns <OTCLIB_TRUE> if the period of
				// this duration is greater than or equal to
				// that of <theDuration>.

    OTC_Boolean		operator>(OTC_Duration const& theDuration) const;
				// Returns <OTCLIB_TRUE> if the period of
				// this duration is greater than that of
				// <theDuration>.

    // = ARITHMETIC

    OTC_Duration	operator+(OTC_Duration const& theDuration) const
				{ return plusDuration(theDuration); }
				// Adds <theDuration> to this duration,
				// returning the result. This duration is not
				// modified.

    OTC_Duration&	operator+=(OTC_Duration const& theDuration)
				{ return addDuration(theDuration); }
				// Adds <theDuration> to this duration. The
				// modified value of this duration is
				// returned.

    OTC_Duration	operator-() const;
				// Returns a duration which is of the
				// opposite sign to this duration. This
				// duration is not modified.

    OTC_Duration	operator-(OTC_Duration const& theDuration) const
				{ return minusDuration(theDuration); }
				// Subtracts <theDuration> from this duration,
				// returning the result. This duration is
				// not modified.

    OTC_Duration&	operator-=(OTC_Duration const& theDuration)
				{ return subtractDuration(theDuration); }
				// Subtracts <theDuration> from this
				// duration. The modified value of this
				// duration is returned.

    OTC_Duration	operator*(int theNum) const;
				// Returns a duration which is <theNum>
				// times longer than this duration. This
				// duration is not modified.

    friend OTC_Duration operator*(int theNum, OTC_Duration const& theDuration)
				{ return theDuration * theNum; }
				// Returns a duration which is <theNum>
				// times longer than <theDuration>.
				// <theDuration> is not modified.

    OTC_Duration&	operator*=(int theNum);
				// Increases the length of this duration by
				// <theNum> times. The modified value of this
				// duration is returned.

    // The following return approximate results only.

    OTC_Duration	operator/(u_int theNum) const;
				// Divides this duration by the factor
				// <theNum>, returning the result. This
				// duration is not modified. Raises an
				// exception if <theNum> is <0>. Note that
				// <theNum> must be a positive value.

    OTC_Duration&	operator/=(u_int theNum);
				// Divides this duration by the factor
				// <theNum>. The modified value of this
				// duration is returned. Raises an exception
				// if <theNum> is <0>. Note that <theNum>
				// must be a positive value.

    int			operator/(OTC_Duration const& theDuration) const;
				// Returns how many times <theDuration> will
				// fit into this duration. The sign of the
				// duration is not considered. Raises an
				// exception if <theDuration> is of
				// zero length.

    // = RANKING

    int			rank(OTC_Duration const& theDuration) const;
				// Returns a value less than, equal to,
				// or greater than zero corresponding to
				// how this date compares to <theDuration>.

    // = HASHING

    int			hash() const;
				// Returns a hash value for this date.

  private:

    void		normalise();
				// Normalises the various components of the
				// duration such that they are all positive
				// and the sign bit set correspondingly.

    int			mySign;
				// The sign of the duration. Has value <1> if
				// duration is in positive direction. Has
				// value <-1> if duration is in negative
				// direction.

    int			myDays;
				// The day component of the duration.

    int			myHours;
				// The hour component of the duration.

    int			myMinutes;
				// The minute component of the duration.

    int			mySeconds;
				// The second component of the duration.
};

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
OSE_MARK_TEMPLATE OTC_RankActions<OTC_Duration>;
OSE_MARK_TEMPLATE OTC_HashActions<OTC_Duration>;
#endif

#if defined(CXX_OS)
typedef OTC_RankActions<OTC_Duration> otc_types_duration_hh_typedef1;
#pragma ObjectStore exclude_instantiations OTC_RankActions<OTC_Duration>;
typedef OTC_HashActions<OTC_Duration> otc_types_duration_hh_typedef2;
#pragma ObjectStore exclude_instantiations OTC_HashActions<OTC_Duration>;
#endif

#if !defined(CXX_OS)
class OTC_RankActions<OTC_Duration>
{
  public:
    static int		rank(
			 OTC_Duration const& theDuration1,
			 OTC_Duration const& theDuration2
			)
				{ return theDuration1.rank(theDuration2); }
};

class OTC_HashActions<OTC_Duration>
{
  public:
    static int		hash(OTC_Duration const& theDuration)
				{ return theDuration.hash(); }
};
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_TYPES_DURATION_HH */
