#ifndef OTC_TYPES_TIME_HH
#define OTC_TYPES_TIME_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     types/time.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/types/date.hh>

/* ------------------------------------------------------------------------- */

class OTC_Time : public OTC_Date
    // = TITLE
    //	   Class maintaining information about a day and the time of day.
    //	   
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   Maintains time information down to a resolution of one second.
    //	   The class does not track anything related to timezones or whether
    //	   daylight savings time is in operation. It exists purely to record
    //	   a time value. Times can be from 00:00:00, 1 Jan 4713 BC thru to
    //	   some time in the distant future.
    //
    //     The format described for the string representation of time in ISO
    //     8601:1988 is Y-M-DTH:M:S. For example 12 noon, 25th December 1995,
    //     woudl be represented as "1995-25-12T12:00:00".
    //     
    // = SEE ALSO
    //	   <OTC_Date>, <OTC_Duration>
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    // = INITIALISATION

			OTC_Time();
				// Initialises the time to the current time
				// as used by the machine.

			OTC_Time(OTC_Time const& theTime);
				// Initialises the time to that specified by
				// <theTime>.

			OTC_Time(int theHour, int theMin, int theSec);
				// Initialises the time to today and with
				// the hour set to <theHour>, the minute
				// set to <theMin> and seconds to <theSec>.

			OTC_Time(
			 OTC_Date const& theDate,
			 int theHour=0,
			 int theMin=0,
			 int theSec=0
			);
				// Initialises the time to that specified by
				// <theDate> and with the hour set to
				// <theHour>, the minute set to <theMin> and
				// seconds to <theSec>.

			OTC_Time(OTC_String const& theTimeString);
				// Initialises the time to that described by
				// <theTimeString>. The only format for
				// <theTimeString> which is accepted is that
				// described for the string representation of
				// times in ISO 8601:1988. An exception will
				// be raised if <theTimeString> is in an
				// incorrect format.

    // = CLASS OPERATIONS

    static OTC_Time	currentTime()
				{ OTC_Time theTime; return theTime; }
				// Returns the current time.

    static OTC_Boolean	isValidTime(int theHour, int theMin, int theSec);
				// Returns <OTCLIB_TRUE> if the arguments
				// are within valid ranges.

    static OTC_Boolean	isValidTime(OTC_String const& theTimeString);
				// Returns <OTCLIB_TRUE> if <theTimeString>
				// describes a valid time.

    static void		convertToTime(
			 OTC_String const& theTimeString,
			 int& theDay,
			 int& theMonth,
			 int& theYear,
			 int& theHour,
			 int& theMin,
			 int& theSec
			);
				// Sets <theDay>, <theMonth>, <theYear>,
				// <theHour>, <theMin> and <theSec>
				// corresponding to the time described by the
				// contents of <theTimeString>. The only
				// format for <theTimeString> which is
				// currently understood is that described for
				// times in ISO 8601:1988. If the value <0>
				// is returned in each of the variables the
				// string was not in the format expected.

    // = QUERY

    int			hour() const;
				// Returns the number of hours using 24 hour
				// time, <0-23>.

    int			minute() const;
				// Returns the number of minutes, <0-59>.

    int			second() const;
				// Returns the number of seconds, <0-59>.

    OTC_Time		plusSeconds(int theNum) const;
				// Returns this time with <theNum> seconds
				// added.

    OTC_Time		minusSeconds(int theNum) const
				{ return plusSeconds(-theNum); }
				// Returns this time with <theNum> seconds
				// subtracted.

    OTC_Time		plusMinutes(int theNum) const
				{ return plusSeconds(60*theNum); }
				// Returns this time with <theNum> minutes
				// added.

    OTC_Time		minusMinutes(int theNum) const
				{ return minusSeconds(60*theNum); }
				// Returns this time with <theNum> minutes
				// subtracted.

    OTC_Time		plusHours(int theNum) const
				{ return plusSeconds(60*60*theNum); }
				// Returns this time with <theNum> hours
				// added.

    OTC_Time		minusHours(int theNum) const
				{ return minusSeconds(60*60*theNum); }
				// Returns this time with <theNum> hours
				// subtracted.

    OTC_Time		plusDuration(OTC_Duration const& theDuration) const;
				// Returns this time with <theDuration>
				// added.

    OTC_Time		minusDuration(OTC_Duration const& theDuration) const
				{ return plusDuration(-theDuration); }
				// Returns this time with <theDuration>
				// subtracted.

    // = TEXTUAL REPRESENTATION

    OTC_String		asString() const;
				// Returns the time as a string. The format
				// used for the time when converting it to a
				// string, is that defined by ISO 8601:1988.

    friend ostream&	operator<<(ostream& theStream, OTC_Time const& theTime);
				// Outputs the time onto <theStream>. The
				// format used for the time when displaying
				// it, is that defined by ISO 8601:1988.

    // = MODIFIERS

    void		setTime();
				// Sets the time to the current time.

    void		setTime(OTC_Time const& theTime);
				// Sets the time to that defined by
				// <theTime>.

    void		setTime(int theHour, int theMin, int theSec);
				// Sets the time to that defined
				// by <theHour>, <theMin> and <theSec>
				// within the current day. Values of
				// zero indicate midnight.

    void		setTime(
			 OTC_Date const& theDate,
			 int theHour,
			 int theMin,
			 int theSec
			);
				// Sets the time to that defined
				// by <theHour>, <theMin> and <theSec>
				// within the <theDate>.

    OTC_Time&		operator=(OTC_Time const& theTime)
				{ setTime(theTime); return *this; }
				// Sets the time to that defined by
				// <theTime>.

    OTC_Time&		addSeconds(int theNum);
				// Adds <theNum> seconds to the time.

    OTC_Time&		subtractSeconds(int theNum)
				{ return addSeconds(-theNum); }
				// Subtracts <theNum> seconds from the time.

    OTC_Time&		addMinutes(int theNum);
				// Adds <theNum> minutes to the time.

    OTC_Time&		subtractMinutes(int theNum)
				{ return addMinutes(-theNum); }
				// Subtracts <theNum> minutes from the time.

    OTC_Time&		addHours(int theNum);
				// Adds <theNum> hours to the time.

    OTC_Time&		subtractHours(int theNum)
				{ return addHours(-theNum); }
				// Subtracts <theNum> hours from the time.

    OTC_Time&		addDuration(OTC_Duration const& theDuration);
				// Adds <theDuration> to the time.

    OTC_Time&		subtractDuration(OTC_Duration const& theDuration);
				// Subtracts <theDuration> from the time.

    // = COMPARISONS

    OTC_Boolean		operator==(OTC_Time const& theTime) const;
				// Returns <OTCLIB_TRUE> if this time
				// is the same as <theTime>.

    OTC_Boolean		operator!=(OTC_Time const& theTime) const;
				// Returns <OTCLIB_TRUE> if this time
				// is not the same as <theTime>.

    OTC_Boolean		operator<(OTC_Time const& theTime) const;
				// Returns <OTCLIB_TRUE> if this time
				// comes before <theTime>.

    OTC_Boolean		operator<=(OTC_Time const& theTime) const;
				// Returns <OTCLIB_TRUE> if this time
				// comes before or is the same as <theTime>.

    OTC_Boolean		operator>(OTC_Time const& theTime) const;
				// Returns <OTCLIB_TRUE> if this time
				// comes after <theTime>.

    OTC_Boolean		operator>=(OTC_Time const& theTime) const;
				// Returns <OTCLIB_TRUE> if this time
				// comes after or is the same as <theTime>.

    // = ARITHMETIC

    OTC_Time&		operator+=(OTC_Duration const& theDuration)
				{ return addDuration(theDuration); }
				// Adds <theDuration> to the time.

    OTC_Time&		operator-=(OTC_Duration const& theDuration)
				{ return addDuration(-theDuration); }
				// Subtracts <theDuration> from the time.

    OTC_Time		operator+(OTC_Duration const& theDuration) const
				{ return plusDuration(theDuration); }
				// Returns this time with <theDuration>
				// added.

    OTC_Time		operator-(OTC_Duration const& theDuration) const
				{ return plusDuration(-theDuration); }
				// Returns this time with <theDuration>
				// subtracted.

    OTC_Duration	operator-(OTC_Time const& theTime) const;
				// Returns the duration between this
				// time and <theTime>. Note that a negative
				// duration indicates that <theTime> is
				// in the future with respect to this time.

    OTC_Duration	operator-(OTC_Date const& theDate) const;
				// Returns the duration between this
				// time and <theDate>. Note that a negative
				// duration indicates that <theDate> is
				// in the future with respect to this time.

    // = RANKING

    int			rank(OTC_Time const& theTime) const;
				// Returns a value less than, equal to,
				// or greater than zero corresponding to
				// how this time compares to <theTime>.

    // = HASHING

    int			hash() const
				{ return OTC_Date::hash() ^ int(mySeconds); }
				// Returns a hash value for this time.

  private:

    int			mySeconds;
				// Number of seconds since midnight.
};

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
OSE_MARK_TEMPLATE OTC_RankActions<OTC_Time>;
OSE_MARK_TEMPLATE OTC_HashActions<OTC_Time>;
#endif

#if defined(CXX_OS)
typedef OTC_RankActions<OTC_Time> otc_types_time_hh_typedef1;
#pragma ObjectStore exclude_instantiations OTC_RankActions<OTC_Time>;
typedef OTC_HashActions<OTC_Time> otc_types_time_hh_typedef2;
#pragma ObjectStore exclude_instantiations OTC_HashActions<OTC_Time>;
#endif

#if !defined(CXX_OS)
class OTC_RankActions<OTC_Time>
{
  public:
    static int		rank(OTC_Time const& theTime1, OTC_Time const& theTime2)
				{ return theTime1.rank(theTime2); }
};

class OTC_HashActions<OTC_Time>
{
  public:
    static int		hash(OTC_Time const& theTime)
				{ return theTime.hash(); }
};
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_TYPES_TIME_HH */
