#ifndef OTK_DISPATCH_JOBQUEUE_HH
#define OTK_DISPATCH_JOBQUEUE_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTK
// 
// = FILENAME
//     dispatch/jobqueue.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 TELSTRA CORPORATION LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OUX/dispatch/jobqueue.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTK/dispatch/jobqueue.hh"
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTK_DummyJob;

class OTK_JobQueue : public OTC_JobQueue
    // = TITLE
    //	   Job queue which understands TK, UNIX signals, time and I/O events.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   <OTK_JobQueue> is a derived version of <OTC_JobQueue> which
    //	   understands how to manage signals, alarms, timers and I/O events,
    //	   within the UNIX environment. This queue adds an extra internal
    //     queue to that in the base class. The queue in the base class is
    //     used to hold idle jobs, the new queue priority jobs.
    //
    //	   When the <next()> member function is invoked to return the
    //	   next job to execute, the class first checks for any signals
    //	   which may have occurred. If no signals have occurred, a check
    //	   is made for any alarms which may have expired. Checks are then
    //	   made for any alarms which may have expired. Checks are then
    //     made for priority jobs, timers, io events and finally idle jobs.
    //	   
    //	   When a request is made for a specific type of job, the valid
    //	   flags which can OR'ed together are:
    //
    // = BEGIN<INDENT>
    // = BEGIN<NOFILL>
    //	   - <OUXLIB_SIGNAL_JOBS>
    //	   - <OUXLIB_ALARM_JOBS>
    //	   - <OUXLIB_TIMEOUT_JOBS>
    //	   - <OUXLIB_IO_JOBS>
    //	   - <OUXLIB_IDLE_JOBS>
    //	   - <OUXLIB_PRIORITY_JOBS>
    // = END<NOFILL>
    // = END<INDENT>
    //
    //	   If <OUXLIB_DONT_WAIT> is included in the set of flags, the
    //	   function will return without executing a job if the function would
    //	   have needed to block in order to get a job. The value <0> can be
    //	   used to select all event sources, or <OUXLIB_DONT_WAIT> by itself
    //	   if all event sources should be select but it shouldn't block.
    //
    // = SEE ALSO
    //	   <OTC_JobQueue>, <OTC_Job>, <OUXEV_Signal>, <OTCEV_Alarm>,
    //	   <OTCEV_Timeout>, <OTCEV_IOEvent>, <OUX_Dispatcher>,
    //	   <OTK_JobQueue>
{
  public:

			OTK_JobQueue();

			~OTK_JobQueue();

    // = QUEUEING

    void                add(OTC_Job* theJob, int theOptions);
                                // Adds <theJob> to the end of an appropriate
                                // queue. Generates an exception if <theJob>
                                // is <0>. If <theOptions> is
                                // <OUXLIB_IDLE_JOB>, the job is taken to be
                                // an idle job. If <theOptions>, is
                                // <OUXLIB_PRIORITY_JOB> is is taken as being
                                // a priority job.

    OTC_Job*		next(int theActions, int theOptions);
				// Returns a job to be executed. The type of
				// job and whether the function blocks is
				// determined by <theActions>. If <theActions>
				// is <0>, all event sources will be checked
				// and the function will block if necessary.
				// If non zero, <theActions> should be the
				// OR'ed combination of the flags defined
				// by the enum <OUX_JobActions>.
				// <theOptions> is used internally where
				// the dispatcher is effectively calling
				// itself because of a "tkwait" command.
				// <theOptions> should be set to <0> for
				// normal use.

    // = BAILING OUT

    static void		setExit(void (*theFunc)())
				{ globExit = theFunc; }
				// If the dispatcher is stopped while inside
				// the "tkwait" command, we have no choice
				// but to exit the application. This function
				// provides a means of specifying a function
				// to perform any necessary cleanup for the
				// application and nicely exit the
				// application. If the supplied function
				// returns, <exit()> will still be called.

  private:

    static int		nfds();
				// Returns the highest numbered file
				// descriptor the operating system can
				// manage.

    OTK_DummyJob*	myDummyJob;
				// Dummy job. Does no real work.

    OTC_JobQueue*	myPriorityJobs;
				// List of priority jobs.

    OTC_JobQueue*	myIdleJobs;
				// List of idle jobs being processed.

    int*		myPendingFdEvents;
				// Array of fd events still to be processed.

    int			myNextFd;
				// Next fd to check in set of pending fd
				// events.

    int			myMaxFd;
				// Maximum numbered file descriptor which
				// may have events still to be processed.

    static void		(*globExit)();
				// User provided exit function to be called
				// in the event that the dispatcher is stopped
				// while in a "tkwait" command.

    static OTK_JobQueue*	globQueue;
				// Pointer to single instance of job queue.

    static int		tkDispatch(int theFlags);
				// Dispatch routine called by TK.
};

/* ------------------------------------------------------------------------- */

#endif /* OTK_DISPATCH_JOBQUEUE_HH */
