#ifndef OUX_DISPATCH_SIGNAL_HH
#define OUX_DISPATCH_SIGNAL_HH
/*
// ============================================================================
//
// = LIBRARY
//     OUX
// 
// = FILENAME
//     dispatch/signal.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/dispatch/event.hh>
#include <OTC/thread/nrmutex.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OUX/dispatch/signal.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Job;
class OTC_JobQueue;

class OUX_SignalSubscription;
class OUX_SignalInfo;

class OUXEV_Signal : public OTC_Event
    // = TITLE
    //     Event object to notify that signal has occurred.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     <OUXEV_Signal> is a derived version of <OTC_Event> specifically
    //     for notifying agents of the occurence of a UNIX signal. The
    //     class also provides the interface for registration of interest
    //     in signals by agents.
    //
    // = SEE ALSO
    //     <OTC_Event>, <OTC_EVAgent>
{
  public:

			~OUXEV_Signal();

    // = CONSTRUCTION

			OUXEV_Signal(int theSignal)
			  : mySignal(theSignal)
				{}
				// Creates an event object for <theSignal>.

    // = QUERY

    int			signal() const
				{ return mySignal; }
				// Returns the number of the signal that
				// occurred.

    // = IDENTIFICATION

    void*		type() const;
				// Returns a value that uniquely identifies
				// this type of event.

    static void*	typeId()
				{ return &globTypeId; }
				// Returns a value that uniquely identifies
				// this type of event.

    // = DEBUGGING

    void		dump(ostream& outs) const;
				// Dumps info about the event onto <outs>.

    // = SUBSCRIPTION

    static void		subscribe(int theAgentId, int theSignal);
				// Registers that the agent with ID number
				// <theAgentId> should be notified when
				// <theSignal> occurs.

    static void		unsubscribe(int theAgentId, int theSignal);
				// Forget that the agent with ID number
				// <theAgentId> is interested in being
				// notified about occurences of <theSignal>.

    static void		unsubscribeAgent(int theAgentId);
				// Forget that the agent identified by
				// <theAgentId> is interested in any signals.

    // = INITIALISATION

    static void		capacity(u_int theCapacity);
				// Sets the capacity of the array used to
				// hold information about pending signals.
				// Must be called before any signal
				// subscriptions are made. If called after
				// signal subscriptions have been made,
				// the call is ignored. If not called at all,
				// the default size of <64> is used. If
				// <theCapacity> is <0> an exception is
				// raised.

    // = SCHEDULING

    static OTC_Job*	pending();
				// Returns a job for the next pending
				// signal or <0> if no pending signals
				// exist.

    static int		fd()
				{ return globSigCount ? globSignalFds[0] : -1; }
				// Returns a file descriptor which can be
				// waited upon to determine if a signal has
				// occurred. Returns <-1> if no signals have
				// yet been subscribed to.

  protected:

    // = NON DELIVERY

    void		cancelSource(int theAgentId);
				// Unsubscribes the agent with ID
				// <theAgentId> to which this event was to be
				// delivered, from all further signals. This
				// function will be called when the agent to
				// which this event was to be delivred does
				// not exist.

  private:

			OUXEV_Signal(OUXEV_Signal const&);
				// Do not define an implementation for this.

    OUXEV_Signal&	operator=(OUXEV_Signal const&);
				// Do not define an implementation for this.

    static void		fill(OTC_JobQueue* theJobQueue);
				// Adds jobs for any pending signals to
				// <theJobQueue>.

    static void		sighandler(int theSignal);
				// Signal handler.

    static OTC_NRMutex	_mutex;
				// Lock for threads.

    static OUX_SignalSubscription**	globSubscriptions;
				// List of agents interested in signals.

    static OUX_SignalInfo*	globPendingSignals;
				// Table of signals which have occurred
				// but which haven't been processed yet.

    static u_int	globPendingSize;
				// Maximum number of queue signals which
				// can be kept. Actually, it will be more
				// than this as we compress signals and
				// keep a count of the number of times
				// they occur when the signal number
				// matches for sequential signals.

    static u_int	globNextAvailable;
				// Next available slot in table of pending
				// signals.

    static OTC_Boolean	globSignalsLost;
				// Set to <OTCLIB_TRUE> by signal handler
				// if there was no space in the table to
				// record information about a signal.

    static u_int	globSigCount;
				// Number of different signals to which
				// agents are subscribed.

    static OTC_JobQueue*	globSignalJobs;
				// Pending signal jobs.

    static int		globSignalFds[2];
				// File descriptor for a pipe

    static int		globTypeId;
				// Anchor for use a type identifier.

    int			mySignal;
				// The signal which occurred.
};

/* ------------------------------------------------------------------------- */

#endif /* OUX_DISPATCH_SIGNAL_HH */
