#ifndef OUX_SYSTEM_GROUP_HH
#define OUX_SYSTEM_GROUP_HH
/*
// ============================================================================
//
// = LIBRARY
//     OUX
//
// = FILENAME
//     system/group.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 1992 OTC LIMITED
//     Copyright 1994 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifndef ENV_VXWORKS

#include <OTC/collctn/set.hh>
#include <OTC/text/string.hh>

#include <sys/types.h>
#include <stdio.h>
#include <grp.h>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OUX/system/group.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
OSE_TEMPLATE OTC_Set<OTC_String>;
#endif

typedef OTC_Set<OTC_String> oux_system_group_hh_typedef1;

class OUX_Group
    // = TITLE
    //     Class to encapsulate UNIX group information.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     The <OUX_Group> class is a wrapper around the group entry
    //     structure.
    //     
    // = NOTES
    //     If the data used to get information about the group, doesn't
    //     identify a valid group on the host then the information will be
    //     filled out with default values. The <isValid()> function should
    //     always be invoked to actually determine if the information
    //     gathered is correct.
    //
    //     Note that successive calls to <getgrent()> to get all entries in
    //     the group file may not be able to be used while creating instances
    //     of this class using any constructor other than that taking a
    //     pointer to the group structure, or the copy constructor, as
    //     instantiation of this class may have the result of reseting the
    //     pointer used by <getgrent()> back to the start. Whether this
    //     occurs will depend on your variant of UNIX.
    //     
    //     An <OTC_BaseActions> and <OTC_HashActions> are defined for
    //     <OUX_Group>. For <OTC_HashActions> the key on which hashing
    //     and comparison is performed is the group id.
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    // = INITIALISATION

			OUX_Group();
				// Create a class for the group corresponding
				// to the current group id. If the current
				// group id doesn't describe a valid group on
				// this host then the group id will still be
				// used, and group name will be set to
				// the group id. The set of members for the
				// group will be empty.

			OUX_Group(gid_t theGID);
				// Creates a class for the group with group
				// id <theGID>. If <theGID> doesn't describe
				// a valid group on this host then the group
				// id will still be used, and group name will
				// be set to the group id. The set of
				// members for the group will be empty.

			OUX_Group(OTC_String const& theName);
				// Creates a class for the group with group
				// name <theName>. If <theName> doesn't
				// describe a valid group on this host then
				// the group name will still be used, however
				// the group id will be set to <MAXUID>,
				// the id generally associated with the user
				// <nobody>. The set of members for the group
				// will be empty.

			OUX_Group(group const* theGrent);
				// Create a class for the group corresponding
				// to that described by the group structure
				// <theGrent>. If <theGrent> is <0> then the
				// group id and group name are set to
				// <MAXUID>, the id generally associated with
				// the user <nobody>. The set of members for
				// the group will be empty.

			OUX_Group(OUX_Group const& theGroup);
				// Creates a copy of <theGroup>.

    OUX_Group&		operator=(OUX_Group const& theGroup);
				// Initialises this class to <theGroup>.

    OUX_Group&		operator=(group const* theGrent);
				// Initialises this class to that group
				// described by the group structure
				// <theGrent>. If <theGrent> is <0> then the
				// group id and group name are set to
				// <MAXUID>, the id generally associated with
				// the user <nobody>. The set of members for
				// the group will be empty.

    // = ENQUIRY

    OTC_Boolean		isValid() const
				{ return myValid; }
				// Return <OTCLIB_TRUE> if this class
				// describes a valid group on this host.

    OTC_String const&	name() const
				{ return myName; }
				// Returns the name of the group.

    gid_t		gid() const
				{ return myGID; }
				// Returns the group id of the group.

			operator gid_t() const
				{ return gid(); }
				// Returns the group id of the group.

    OTC_Iterator<OTC_String>	members() const
				{ return myMembers.items(); }
				// Returns an iterator over the list of
				// members in the group.

    OTC_Boolean		contains(OTC_String const& theName) const
				{ return myMembers.contains(theName); }
				// Returns <OTCLIB_TRUE> if <theName> is a
				// member of this group.

    OTC_Boolean		isMember(OTC_String const& theName) const
				{ return myMembers.contains(theName); }
				// Returns <OTCLIB_TRUE> if <theName> is a
				// member of this group. This function is
				// provided for backward compatability
				// only, use <contains()> instead.

  private:

    void		init(group const* theGrent);
				// Initialises <myName>, <myId> and
				// <myMembers> from <theGrent>.

    OTC_Boolean		myValid;
				// <OTCLIB_TRUE> if the group exists on this
				// host.

    OTC_String		myName;
				// The name of the group.

    gid_t		myGID;
				// The group id of the group.

    OTC_Set<OTC_String>	myMembers;
				// The members of the group.
};

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
OSE_MARK_TEMPLATE OTC_RankActions<OUX_Group>;
OSE_MARK_TEMPLATE OTC_HashActions<OUX_Group>;
#endif

#if defined(CXX_OS)
typedef OTC_RankActions<OUX_Group> oux_system_group_hh_typedef2;
#pragma ObjectStore exclude_instantiations OTC_RankActions<OUX_Group>;
typedef OTC_HashActions<OUX_Group> oux_system_group_hh_typedef3;
#pragma ObjectStore exclude_instantiations OTC_HashActions<OUX_Group>;
#endif

#if !defined(CXX_OS)
class OTC_RankActions<OUX_Group>
{
  public:
    static int		rank(
			 OUX_Group const& theGrp1,
			 OUX_Group const& theGrp2
			)
				{ return theGrp1.gid() - theGrp2.gid(); }
};

class OTC_HashActions<OUX_Group>
{
  public:
    static int		hash(OUX_Group const& theGrp)
				{ return theGrp.gid(); }
};
#endif

/* ------------------------------------------------------------------------- */

#endif

#endif /* OUX_SYSTEM_GROUP_HH */
