#ifndef OUX_SYSTEM_USER_HH
#define OUX_SYSTEM_USER_HH
/*
// ============================================================================
//
// = LIBRARY
//     OUX
//
// = FILENAME
//     system/user.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 1992 OTC LIMITED
//     Copyright 1994 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifndef ENV_VXWORKS

#include <OTC/text/string.hh>
#include <OTC/text/record.hh>

#include <sys/types.h>
#include <pwd.h>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OUX/system/user.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OUX_User
    // = TITLE
    //     Class to encapsulate UNIX user information.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     The <OUX_User> class is a wrapper around the password entry
    //     for a user.
    //
    // = NOTES
    //     If the data used to get information about the user, doesn't
    //     identify a valid user on the host then the information will be
    //     filled out with default values. The <isValid()> function should
    //     always be invoked to actually determine if the information
    //     gathered is correct.
    //     
    //     Note that successive calls to <getpwent()> to get all entries in
    //     the passwd file may not be able to be used while creating
    //     instances of this class using any constructor besides the copy
    //     constructor and that accepting a pointer to a passwd structure, as
    //     instantiation of the class may have the result of reseting the
    //     pointer used by <getpwent()> back to the start. Whether this
    //     occurs will depend on your variant of UNIX.
    //     
    //     An <OTC_BaseActions> and <OTC_HashActions> are defined for
    //     <OUX_User>. For <OTC_HashActions> the key on which hashing
    //     and comparison is based is the user id.
{
  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    // = INITIALISATION

			OUX_User();
				// Creates a class for the user corresponding
				// to the current user id. If the current
				// user id doesn't describe a valid user on
				// this host then the user id will still be
				// used and the group id, and user name will
				// also be set to the user id. Also, the full
				// name of the user will be empty.

			OUX_User(uid_t theUID);
				// Creates a class for the user with user id
				// <theUID>. If <theUID> doesn't describe a
				// valid user on this host then the user id
				// will still be used and the group id, and
				// user name will also be set to the user id.
				// Also, the full name of the user will be
				// empty.

			OUX_User(OTC_String const& theName);
				// Creates a class for the user with user
				// name <theName>. If <theName> doesn't
				// describe a valid user on this host then
				// the user name will still be used, however
				// the user id and group id will be set to
				// <MAXUID>, the id generally assigned to the
				// user <nobody>. Also, the full name of the
				// user will be empty.

			OUX_User(passwd const* thePwent);
				// Creates a class for the user corresponding
				// to the password structure <thePwent>. If
				// <thePwent> is <0> the user id, group id,
				// and user name will be set to <MAXUID>, the
				// id generally assigned to the user
				// <nobody>. Also, the full name of the user
				// will be empty.

			OUX_User(OUX_User const& theUser);
				// Creates a copy of <theUser>.

    OUX_User&		operator=(OUX_User const& theUser);
				// Initialises this user to be the same
				// as <theUser>.

    OUX_User&		operator=(passwd const* thePwent);
				// Initialises this user to be that defined
				// by the passwd structure <thePwent>. If
				// <thePwent> is <0> the user id, group id,
				// and user name will be set to <MAXUID>, the
				// id generally assigned to the user
				// <nobody>. Also, the full name of the user
				// will be empty.

    // = ENQUIRY

    OTC_Boolean		isValid() const
				{ return myValid; }
				// Returns <OTCLIB_TRUE> if this class
				// describes a valid user on this host.

    OTC_String const&	name() const
				{ return myName; }
				// Returns the user name of the user.

    OTC_String const&	fullname() const
				{ return myFullname; }
				// Returns the real name of the user.

    uid_t		uid() const
				{ return myUID; }
				// Returns the user id of the user.

			operator uid_t() const
				{ return uid(); }
				// Returns the user id of the user.

    gid_t		gid() const
				{ return myGID; }
				// Returns the group id corresponding to the
				// user.

  private:

    void		init(passwd const* thePwent);
				// Initialises the attributes of the user
				// from <thePwent>.

    OTC_Boolean		myValid;
				// <OTCLIB_TRUE> if this class describes a
				// valid user on this host.

    OTC_String		myName;
				// The user name of the user.

    OTC_String		myFullname;
				// The real name of the user.

    uid_t		myUID;
				// The user if of the user.

    gid_t		myGID;
				// The group id corresponding to the user.
};

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
OSE_MARK_TEMPLATE OTC_RankActions<OUX_User>;
OSE_MARK_TEMPLATE OTC_HashActions<OUX_User>;
#endif

#if defined(CXX_OS)
typedef OTC_RankActions<OUX_User> oux_system_user_hh_typedef1;
#pragma ObjectStore exclude_instantiations OTC_RankActions<OUX_User>;
typedef OTC_HashActions<OUX_User> oux_system_user_hh_typedef2;
#pragma ObjectStore exclude_instantiations OTC_HashActions<OUX_User>;
#endif

#if !defined(CXX_OS)
class OTC_RankActions<OUX_User>
{
  public:
    static int		rank(
			 OUX_User const& theUser1,
			 OUX_User const& theUser2
			)
				{ return theUser1.uid() - theUser2.uid(); }
};

class OTC_HashActions<OUX_User>
{
  public:
    static int		hash(OUX_User const& theUser)
				{ return theUser.uid(); }
};
#endif

/* ------------------------------------------------------------------------- */

#endif

#endif /* OUX_SYSTEM_USER_HH */
