/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/otcavlnode.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 OTC LIMITED
//     Copyright 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/collctn/avlnode.hh"
#endif

#include <OTC/collctn/avlnode.hh>
#include <OTC/collctn/avltree.hh>

#ifdef index
#undef index
#endif

/* ------------------------------------------------------------------------- */
enum
{
  OTCLIB_LEFTHEAVY = -2,
  OTCLIB_LEFTHIGH = -1,
  OTCLIB_EQUALHEIGHT = 0,
  OTCLIB_RIGHTHIGH = 1,
  OTCLIB_RIGHTHEAVY = 2
};

/* ------------------------------------------------------------------------- */
OTC_AVLNode::OTC_AVLNode()
  : myTree(0), myFather(0), myLeft(0), myRight(0),
    myBalance(OTCLIB_EQUALHEIGHT), myCount(1)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_AVLNode::~OTC_AVLNode()
{
  OTCLIB_ENSURE((isRoot()),
   "OTC_AVLNode::~OTC_AVLNode() - Only root node can be deleted");

  if (myLeft != 0)
  {
    myLeft->_setFather(0);
    myLeft->_setTree(0);
    delete myLeft;
  }
  if (myRight != 0)
  {
    myRight->_setFather(0);
    myRight->_setTree(0);
    delete myRight;
  }

  if (myTree != 0)
  {
    myTree->_setRoot(0);
    myTree->_setPopulation(0);
  }
}

/* ------------------------------------------------------------------------- */
OTC_AVLNode* OTC_AVLNode::_brother() const
{
  if (isRoot())
    return 0;

  if (isLeft())
    return (OTC_AVLNode*)father()->right();

  if (isRight())
    return (OTC_AVLNode*)father()->left();

  return 0;
}

/* ------------------------------------------------------------------------- */
u_int OTC_AVLNode::level() const
{
  u_int theLevel = 0;
  OTC_AVLNode const* theNode = father();
  while (theNode != 0)
  {
    theNode = theNode->father();
    theLevel++;
  }
  return theLevel;
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::addBefore(OTC_AVLNode* theNode)
{
  OTCLIB_ENSURE((tree() != 0),
   "OTC_AVLNode::addBefore() - This node must be in a tree");
  OTCLIB_ENSURE((theNode != 0),
   "OTC_AVLNode::addBefore() - Must have valid object to insert");
  OTCLIB_ENSURE((theNode->tree() == 0),
   "OTC_AVLNode::addBefore() - Node is already in a tree");

  if (left() != 0)
  {
    OTC_AVLNode* theParent = left();
    OTC_AVLNode* tmpNode = theParent->right();
    while (tmpNode != 0)
    {
      theParent = tmpNode;
      tmpNode = tmpNode->right();
    }
    theParent->addAfter(theNode);
  }
  else
  {
    theNode->_setBalance(OTCLIB_EQUALHEIGHT);
    theNode->_setTree(tree());
    theNode->_setFather(this);
    theNode->_setCount(1);
    _setLeft(theNode);
    _setBalance(_balance()-1);

    _setCount(count()+1);
    OTC_AVLNode* tmpNode = father();
    OTC_Boolean childIsLeft = isLeft();
    while (tmpNode != 0)
    {
      if (childIsLeft != OTCLIB_FALSE)
	tmpNode->_setCount(tmpNode->count()+1);
      childIsLeft = tmpNode->isLeft();
      tmpNode = tmpNode->father();
    }

    _addBalance();
    tree()->myPopulation++;
  }
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::addAfter(OTC_AVLNode* theNode)
{
  OTCLIB_ENSURE((tree() != 0),
   "OTC_AVLNode::addAfter() - This node must be in a tree");
  OTCLIB_ENSURE((theNode != 0),
   "OTC_AVLNode::addAfter() - Must have valid object to insert");
  OTCLIB_ENSURE((theNode->tree() == 0),
   "OTC_AVLNode::addAfter() - Node is already in a tree");

  if (right() != 0)
  {
    OTC_AVLNode* theParent = right();
    OTC_AVLNode* tmpNode = theParent->left();
    while (tmpNode != 0)
    {
      theParent = tmpNode;
      tmpNode = tmpNode->left();
    }
    theParent->addBefore(theNode);
  }
  else
  {
    theNode->_setBalance(OTCLIB_EQUALHEIGHT);
    theNode->_setTree(tree());
    theNode->_setFather(this);
    theNode->_setCount(1);
    _setRight(theNode);
    _setBalance(_balance()+1);

    OTC_AVLNode* tmpNode = father();
    OTC_Boolean childIsLeft = isLeft();
    while (tmpNode != 0)
    {
      if (childIsLeft != OTCLIB_FALSE)
	tmpNode->_setCount(tmpNode->count()+1);
      childIsLeft = tmpNode->isLeft();
      tmpNode = tmpNode->father();
    }

    _addBalance();
    tree()->myPopulation++;
  }
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::swap(OTC_AVLNode* theNode)
{
  OTCLIB_ENSURE((tree() != 0),
   "OTC_AVLNode::swap() - This node must be in a tree");
  OTCLIB_ENSURE((theNode != 0),
   "OTC_AVLNode::swap() - Must have a valid object to insert");

  if (father() == 0)
    tree()->_setRoot(theNode);

  if (theNode->father() == 0 && theNode->tree() != 0)
    theNode->tree()->_setRoot(this);

  OTC_AVLTree* theTree = theNode->tree();
  theNode->_setTree(tree());
  _setTree(theTree);

  int theBalance = theNode->_balance();
  theNode->_setBalance(_balance());
  _setBalance(theBalance);

  u_int theCount = theNode->count();
  theNode->_setCount(count());
  _setCount(theCount);

  if (father() == theNode || theNode->father() == this)
  {
    // One is the direct child of the other.

    OTC_AVLNode* topNode = 0;
    OTC_AVLNode* bottomNode = 0;

    if (father() == theNode)
    {
      topNode = theNode;
      bottomNode = this;
    }
    else
    {
      topNode = this;
      bottomNode = theNode;
    }

    OTC_AVLNode* theFather = topNode->father();

    topNode->_setFather(bottomNode);
    bottomNode->_setFather(theFather);

    if (theFather != 0)
    {
      if (theFather->right() == topNode)
	theFather->_setRight(bottomNode);
      else if (theFather->left() == topNode)
	theFather->_setLeft(bottomNode);
    }

    OTC_AVLNode* leftNode = bottomNode->left();
    OTC_AVLNode* rightNode = bottomNode->right();

    if (topNode->left() == bottomNode)
    {
      bottomNode->_setLeft(topNode);
      bottomNode->_setRight(topNode->right());
      if (topNode->right() != 0)
	topNode->right()->_setFather(bottomNode);
    }
    else if (topNode->right() == bottomNode)
    {
      bottomNode->_setRight(topNode);
      bottomNode->_setLeft(topNode->left());
      if (topNode->left() != 0)
	topNode->left()->_setFather(bottomNode);
    }

    topNode->_setLeft(leftNode);
    if (leftNode != 0)
      leftNode->_setFather(topNode);
    topNode->_setRight(rightNode);
    if (rightNode != 0)
      rightNode->_setFather(topNode);
  }
  else
  {
    // No father child relationship.

    OTC_AVLNode* theFather = theNode->father();

    if (myFather == 0)
    {
      theNode->_setFather(0);
    }
    else
    {
      if (myFather->left() == this)
	myFather->_setLeft(theNode);
      else if (myFather->right() == this)
	myFather->_setRight(theNode);
      if (theNode != 0)
	theNode->_setFather(myFather);
    }

    if (theFather == 0)
    {
      _setFather(0);
    }
    else
    {
      if (theFather->left() == theNode)
	theFather->_setLeft(this);
      else if (theFather->right() == theNode)
	theFather->_setRight(this);
      _setFather(theFather);
    }

    OTC_AVLNode* theLeft = theNode->left();
    OTC_AVLNode* theRight = theNode->right();

    theNode->_setLeft(left());
    if (left() != 0)
      left()->_setFather(theNode);
    theNode->_setRight(right());
    if (right() != 0)
      right()->_setFather(theNode);

    _setLeft(theLeft);
    if (theLeft != 0)
      theLeft->_setFather(this);
    _setRight(theRight);
    if (theRight != 0)
      theRight->_setFather(this);
  }
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::unlink()
{
  OTCLIB_ENSURE((tree() != 0),
   "OTC_AVLNode::unlink() - This node must be in a tree");

  if (left() != 0 && right() != 0)
  {
    OTC_AVLNode* theParent = left();
    OTC_AVLNode* theNode = theParent->right();
    while (theNode != 0)
    {
      theParent = theNode;
      theNode = theNode->right();
    }
    theParent->swap(this);
  }

  OTC_AVLNode* tmpNode = father();
  OTC_Boolean childIsLeft = isLeft();
  while (tmpNode != 0)
  {
    if (childIsLeft != OTCLIB_FALSE)
      tmpNode->_setCount(tmpNode->count() - 1);
    childIsLeft = tmpNode->isLeft();
    tmpNode = tmpNode->father();
  }

  OTCLIB_ASSERT(left() == 0 || right() == 0);

  OTC_AVLNode* childNode = 0;

  if (left() != 0)
    childNode = left();
  else if (right() != 0)
    childNode = right();

  OTC_AVLNode* fatherNode = father();

  if (fatherNode == 0)
  {
    if (childNode != 0)
    {
      tree()->_setRoot(childNode);
      childNode->_setFather(0);
    }
    else
      tree()->_setRoot(0);
  }
  else
  {
    if (fatherNode->left() == this)
    {
      fatherNode->_setLeft(childNode);
      fatherNode->_setBalance(fatherNode->_balance()+1);
    }
    else if (fatherNode->right() == this)
    {
      fatherNode->_setRight(childNode);
      fatherNode->_setBalance(fatherNode->_balance()-1);
    }
    if (childNode != 0)
      childNode->_setFather(fatherNode);
  }

  if (fatherNode != 0)
    fatherNode->_unlinkBalance();

  tree()->myPopulation--;

  _setTree(0);
  _setFather(0);
  _setBalance(OTCLIB_EQUALHEIGHT);
  _setCount(1);
  _setRight(0);
  _setLeft(0);
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::_addBalance()
{
  OTCLIB_ASSERT(_balance() >= OTCLIB_LEFTHEAVY);
  OTCLIB_ASSERT(_balance() <= OTCLIB_RIGHTHEAVY);

  switch (_balance())
  {
    case OTCLIB_LEFTHEAVY:
    {
      _leftBalance();
      break;
    }

    case OTCLIB_LEFTHIGH:
    case OTCLIB_RIGHTHIGH:
    {
      if (!isRoot())
      {
	if (isLeft())
	{
	  father()->_setBalance(father()->_balance()-1);
	  father()->_addBalance();
	}
	else if (isRight())
	{
	  father()->_setBalance(father()->_balance()+1);
	  father()->_addBalance();
	}
      }
      break;
    }

    case OTCLIB_EQUALHEIGHT:
    {
      break;
    }

    case OTCLIB_RIGHTHEAVY:
    {
      _rightBalance();
      break;
    }
  }
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::_unlinkBalance()
{
  OTCLIB_ASSERT(_balance() >= OTCLIB_LEFTHEAVY);
  OTCLIB_ASSERT(_balance() <= OTCLIB_RIGHTHEAVY);

  switch (_balance())
  {
    case OTCLIB_LEFTHEAVY:
    {
      OTC_AVLNode* fatherNode = father();
      OTC_AVLNode* leftNode = left();

      switch (leftNode->_balance())
      {
	case OTCLIB_EQUALHEIGHT:
	{
	  _rotateRight();
	  _setBalance(OTCLIB_LEFTHIGH);
	  leftNode->_setBalance(OTCLIB_RIGHTHIGH);
	  break;
	}

	case OTCLIB_LEFTHIGH:
	{
	  _rotateRight();
	  _setBalance(OTCLIB_EQUALHEIGHT);
	  leftNode->_setBalance(OTCLIB_EQUALHEIGHT);
	  if (fatherNode != 0)
	  {
	    if (leftNode->isLeft())
	      fatherNode->_setBalance(fatherNode->_balance()+1);
	    else if (leftNode->isRight())
	      fatherNode->_setBalance(fatherNode->_balance()-1);
	    fatherNode->_unlinkBalance();
	  }
	  break;
	}

	case OTCLIB_RIGHTHIGH:
	{
	  OTC_AVLNode* leftRightNode = leftNode->right();
	  _leftBalance();
	  if (fatherNode != 0)
	  {
	    if (leftRightNode->isLeft())
	      fatherNode->_setBalance(fatherNode->_balance()+1);
	    else if (leftRightNode->isRight())
	      fatherNode->_setBalance(fatherNode->_balance()-1);
	    fatherNode->_unlinkBalance();
	  }
	  break;
	}
      }
      break;
    }

    case OTCLIB_LEFTHIGH:
    case OTCLIB_RIGHTHIGH:
    {
      break;
    }

    case OTCLIB_EQUALHEIGHT:
    {
      if (father() != 0)
      {
	if (isLeft())
	{
	  father()->_setBalance(father()->_balance()+1);
	  father()->_unlinkBalance();
	}
	else if (isRight())
	{
	  father()->_setBalance(father()->_balance()-1);
	  father()->_unlinkBalance();
	}
      }
      break;
    }

    case OTCLIB_RIGHTHEAVY:
    {
      OTC_AVLNode* fatherNode = father();
      OTC_AVLNode* rightNode = right();

      switch (rightNode->_balance())
      {
	case OTCLIB_EQUALHEIGHT:
	{
	  _rotateLeft();
	  _setBalance(OTCLIB_RIGHTHIGH);
	  rightNode->_setBalance(OTCLIB_LEFTHIGH);
	  break;
	}

	case OTCLIB_LEFTHIGH:
	{
	  OTC_AVLNode* rightLeftNode = rightNode->left();
	  _rightBalance();
	  if (fatherNode != 0)
	  {
	    if (rightLeftNode->isLeft())
	      fatherNode->_setBalance(fatherNode->_balance()+1);
	    else if (rightLeftNode->isRight())
	      fatherNode->_setBalance(fatherNode->_balance()-1);
	    fatherNode->_unlinkBalance();
	  }
	  break;
	}

	case OTCLIB_RIGHTHIGH:
	{
	  _rotateLeft();
	  _setBalance(OTCLIB_EQUALHEIGHT);
	  rightNode->_setBalance(OTCLIB_EQUALHEIGHT);
	  if (fatherNode != 0)
	  {
	    if (rightNode->isLeft())
	      fatherNode->_setBalance(fatherNode->_balance()+1);
	    else if (rightNode->isRight())
	      fatherNode->_setBalance(fatherNode->_balance()-1);
	    fatherNode->_unlinkBalance();
	  }
	  break;
	}
      }
      break;
    }
  }
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::_rotateLeft()
{
  OTC_AVLNode* fatherNode = father();
  OTC_AVLNode* rightNode = right();
  OTC_AVLNode* rightLeftNode = right()->left();
  _setRight(rightLeftNode);
  if (rightLeftNode != 0)
    rightLeftNode->_setFather(this);
  rightNode->_setLeft(this);
  _setFather(rightNode);
  rightNode->_setFather(fatherNode);

  if (fatherNode == 0)
    tree()->_setRoot(rightNode);
  else if (fatherNode->left() == this)
    fatherNode->_setLeft(rightNode);
  else if (fatherNode->right() == this)
    fatherNode->_setRight(rightNode);

  rightNode->_setCount(rightNode->count()+count());
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::_rotateRight()
{
  OTC_AVLNode* fatherNode = father();
  OTC_AVLNode* leftNode = left();
  OTC_AVLNode* leftRightNode = left()->right();
  _setLeft(leftRightNode);
  if (leftRightNode != 0)
    leftRightNode->_setFather(this);
  leftNode->_setRight(this);
  _setFather(leftNode);
  leftNode->_setFather(fatherNode);

  if (fatherNode == 0)
    tree()->_setRoot(leftNode);
  else if (fatherNode->left() == this)
    fatherNode->_setLeft(leftNode);
  else if (fatherNode->right() == this)
    fatherNode->_setRight(leftNode);

  _setCount(count() - leftNode->count());
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::_leftBalance()
{
  OTC_AVLNode* leftNode = left();

  OTCLIB_ASSERT(leftNode != 0);
  OTCLIB_ASSERT(leftNode->_balance() != OTCLIB_EQUALHEIGHT);

  switch (leftNode->_balance())
  {
    case OTCLIB_LEFTHIGH:
    {
      _setBalance(OTCLIB_EQUALHEIGHT);
      leftNode->_setBalance(OTCLIB_EQUALHEIGHT);
      _rotateRight();
      break;
    }

    case OTCLIB_RIGHTHIGH:
    {
      OTC_AVLNode* leftRightNode = leftNode->right();

      OTCLIB_ASSERT(leftRightNode != 0);

      switch (leftRightNode->_balance())
      {
        case OTCLIB_EQUALHEIGHT:
        {
          _setBalance(OTCLIB_EQUALHEIGHT);
          leftNode->_setBalance(OTCLIB_EQUALHEIGHT);
          break;
        }

        case OTCLIB_LEFTHIGH:
        {
          _setBalance(OTCLIB_RIGHTHIGH);
          leftNode->_setBalance(OTCLIB_EQUALHEIGHT);
          break;
        }

        case OTCLIB_RIGHTHIGH:
        {
          _setBalance(OTCLIB_EQUALHEIGHT);
          leftNode->_setBalance(OTCLIB_LEFTHIGH);
          break;
        }
      }
      leftRightNode->_setBalance(OTCLIB_EQUALHEIGHT);
      leftNode->_rotateLeft();
      _rotateRight();
    }
  }
}

/* ------------------------------------------------------------------------- */
void OTC_AVLNode::_rightBalance()
{
  OTC_AVLNode* rightNode = right();

  OTCLIB_ASSERT(rightNode != 0);
  OTCLIB_ASSERT(rightNode->_balance() != OTCLIB_EQUALHEIGHT);

  switch (rightNode->_balance())
  {
    case OTCLIB_RIGHTHIGH:
    {
      _setBalance(OTCLIB_EQUALHEIGHT);
      rightNode->_setBalance(OTCLIB_EQUALHEIGHT);
      _rotateLeft();
      break;
    }

    case OTCLIB_LEFTHIGH:
    {
      OTC_AVLNode* rightLeftNode = rightNode->left();

      OTCLIB_ASSERT(rightLeftNode != 0);

      switch (rightLeftNode->_balance())
      {
        case OTCLIB_EQUALHEIGHT:
        {
          _setBalance(OTCLIB_EQUALHEIGHT);
          rightNode->_setBalance(OTCLIB_EQUALHEIGHT);
          break;
        }

        case OTCLIB_LEFTHIGH:
        {
          _setBalance(OTCLIB_EQUALHEIGHT);
          rightNode->_setBalance(OTCLIB_RIGHTHIGH);
          break;
        }

        case OTCLIB_RIGHTHIGH:
        {
          _setBalance(OTCLIB_LEFTHIGH);
          rightNode->_setBalance(OTCLIB_EQUALHEIGHT);
          break;
        }
      }
      rightLeftNode->_setBalance(OTCLIB_EQUALHEIGHT);
      rightNode->_rotateRight();
      _rotateLeft();
    }
  }
}

/* ------------------------------------------------------------------------- */
u_int OTC_AVLNode::index() const
{
  u_int theIndex = 0;
  u_int tmpCount = count();
  OTC_Boolean childIsLeft = isLeft();
  OTC_AVLNode const* theNode = father();
  while (theNode != 0)
  {
    if (theNode->isLeft())
    {
      if (childIsLeft == OTCLIB_FALSE)
      {
	theIndex += tmpCount;
	tmpCount = theNode->count();
	childIsLeft = OTCLIB_TRUE;
      }
    }
    else
    {
      theIndex += tmpCount;
      if (childIsLeft != OTCLIB_FALSE)
      {
	childIsLeft = OTCLIB_FALSE;
	tmpCount = 0;
      }
      else
	tmpCount = theNode->count();
    }
    theNode = theNode->father();
  }
  theIndex += tmpCount;
  return theIndex-1;
}

/* ------------------------------------------------------------------------- */
u_int OTC_AVLNode::height() const
{
  u_int theHeight = 0;
  OTC_AVLNode const* theNode = this;
  while (theNode != 0)
  {
    theHeight++;

    OTCLIB_ASSERT(theNode->_balance() >= OTCLIB_LEFTHIGH);
    OTCLIB_ASSERT(theNode->_balance() <= OTCLIB_RIGHTHIGH);

    switch (theNode->_balance())
    {
      case OTCLIB_LEFTHIGH:
      {
        theNode = theNode->left();
        break;
      }
 
      case OTCLIB_EQUALHEIGHT:
      {
        theNode = theNode->left();
        break;
      }
 
      case OTCLIB_RIGHTHIGH:
      {
        theNode = theNode->right();
        break;
      }
    }
  }
  return theHeight;
}

/* ------------------------------------------------------------------------- */
OTC_AVLNode* OTC_AVLNode::_first() const
{
  OTC_AVLNode const* theNode = this;
  OTC_AVLNode const* theLeft = left();

  while (theLeft != 0)
  {
    theNode = theLeft;
    theLeft = theLeft->left();
  }

  return (OTC_AVLNode*)theNode;
}

/* ------------------------------------------------------------------------- */
OTC_AVLNode* OTC_AVLNode::_last() const
{
  OTC_AVLNode const* theNode = this;
  OTC_AVLNode const* theRight = right();

  while (theRight != 0)
  {
    theNode = theRight;
    theRight = theRight->right();
  }

  return (OTC_AVLNode*)theNode;
}

/* ------------------------------------------------------------------------- */
OTC_AVLNode* OTC_AVLNode::_prev() const
{
  if (left() != 0)
    return (OTC_AVLNode*)left()->last();

  if (isRight())
    return (OTC_AVLNode*)father();

  OTC_AVLNode const* theNode = this;
  while (theNode->isLeft())
    theNode = theNode->father();
  theNode = theNode->father();

  return (OTC_AVLNode*)theNode;
}

/* ------------------------------------------------------------------------- */
OTC_AVLNode* OTC_AVLNode::_next() const
{
  if (right() != 0)
    return (OTC_AVLNode*)right()->first();

  if (isLeft())
    return (OTC_AVLNode*)father();

  OTC_AVLNode const* theNode = this;
  while (theNode->isRight())
    theNode = theNode->father();
  theNode = theNode->father();

  return (OTC_AVLNode*)theNode;
}

/* ------------------------------------------------------------------------- */
