/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     debug/otclogger.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// = NOTES
//     DEC C++ has prototypes for <fopen()> and <fwrite()> which take <char*>
//     instead of <char const*> for the name of the file and the buffer to be
//     output respectively. Thus need to forcibly cast to <char*> when calling
//     those functions. 
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/debug/logger.hh"
#endif

#include <OTC/debug/logger.hh>

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#if defined(SYS_UNIX)
#include <unistd.h>
#if defined(CXX_ATT2_1) || defined(CXX_CL1_1)
#include <osfcn.h>
#endif
#endif

#ifdef ENV_VXWORKS
#include <hostLib.h>
#endif

#if defined(SYS_UNIX)
#if defined(HAVE_GETHOSTNAME)
#if !defined(__linux__) && !defined(hpux)
extern "C" int gethostname(char*, int);
#endif
#else
#include <sys/utsname.h>
extern "C" int uname(struct utsname*);
#endif
#endif

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 80
#endif

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Logger::myInitialised = OTCLIB_FALSE;
OTC_Boolean OTC_Logger::myUseStderr = OTCLIB_FALSE;
char const* OTC_Logger::myLogFile = 0;
OTC_Logger* OTC_Logger::myHead = 0;
FILE* OTC_Logger::myStream = 0;
OTC_Boolean OTC_Logger::myLongFormat = OTCLIB_FALSE;
char OTC_Logger::myHostName[MAXHOSTNAMELEN+1] = { 0 };
OTC_Mutex OTC_Logger::_mutex;

/* ------------------------------------------------------------------------- */
char const* const OTCLIB_LOGLEVELMESSAGES[] =
{
  "EMERGENCY",
  "ALERT",
  "CRITICAL",
  "ERROR",
  "WARNING",
  "NOTICE",
  "INFO",
  "DEBUG"
};

/* ------------------------------------------------------------------------- */
static char const* const OTCLIB_SHORTMONTHS[] =
{
  "Jan",
  "Feb",
  "Mar",
  "Apr",
  "May",
  "Jun",
  "Jul",
  "Aug",
  "Sep",
  "Oct",
  "Nov",
  "Dec"
};

/* ------------------------------------------------------------------------- */
OTC_Logger::~OTC_Logger()
{
  _mutex.lock();

  unlink();

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTC_Logger::initialise()
{
  // Assumed that this function is called from inside locked region.

  // If not already initialised, do it.

  if (myInitialised == OTCLIB_FALSE)
  {
#if defined(SYS_UNIX)
#if defined(HAVE_GETHOSTNAME)
    gethostname(myHostName,MAXHOSTNAMELEN+1);
#else
    static struct utsname hostinfo;
    uname(&hostinfo);
    int theLength = strlen(hostinfo.nodename);
    if (theLength < MAXHOSTNAMELEN)
      strcpy(myHostName,hostinfo.nodename);
    else
      strncpy(myHostName,hostinfo.nodename,MAXHOSTNAMELEN);
#endif
#endif

    myLogFile = getenv("OTCLIB_LOGFILE");

    if (myLogFile != 0)
    {
      char const* appendenv = getenv("OTCLIB_APPENDLOGFILE");

      FILE* fp = 0;

      char const* theFile = 0;
#if defined(SYS_UNIX)
      // Make assumption that we will have enough space. Bad !!!!

      char buf[MAXPATHLEN+1];
      char const* p = myLogFile;
      char* q = buf;
      while (*p)
      {
	if (*p != '%')
	{
	  *q++ = *p++;
	}
	else
	{
	  p++;
	  if (*p != EOS)
	  {
	    if (*p == 'p')
	    {
	      sprintf(q,"%d",(int)getpid());
	      while (*q != EOS)
		q++;
	    }
	    else if (*p == 'h')
	    {
	      strcpy(q,myHostName);
	      while (*q != EOS)
		q++;
	    }
	    else if (*p == '%')
	    {
	      *q++ = '%';
	    }

	    p++;
	  }
	}
      }
      *q = EOS;

      theFile = buf;
#else
      theFile = myLogFile;
#endif

      if (appendenv != 0)
	fp = fopen((char*)theFile,"a+");
      else
	fp = fopen((char*)theFile,"w");

      if (fp == 0)
      {
	myLogFile = 0;
	errno = 0;
      }
      else
	fclose(fp);
    }

#ifdef NDEBUG
    myUseStderr = OTCLIB_FALSE;
    char const* stderrenv = getenv("OTCLIB_LOGSTDERR");
    if (stderrenv)
      myUseStderr = OTCLIB_TRUE;
#else
    myUseStderr = OTCLIB_TRUE;
    char const* stderrenv = getenv("OTCLIB_NOLOGSTDERR");
    if (stderrenv)
      myUseStderr = OTCLIB_FALSE;
#endif

#ifndef _MSC_VER
    char const* fdenv = getenv("OTCLIB_LOGFD");
    int logfd = 0;
    if (fdenv)
    {
      logfd = atoi(fdenv);
      myStream = fdopen(logfd,"w+");
    }
#endif

    if (myStream == 0)
    {
      myStream = stderr;
      errno = 0;
    }

    char const* formatenv = getenv("OTCLIB_LOGLONGFORMAT");
    if (formatenv)
      myLongFormat = OTCLIB_TRUE;

    myInitialised = OTCLIB_TRUE;
  }
}

/* ------------------------------------------------------------------------- */
void OTC_Logger::notify(
 char const* theTarget,
 OTC_LogLevel theLevel,
 char const* theMessage,
 int theLength
)
{
  _mutex.lock();

  initialise();

  if (theMessage == 0)
    theMessage = "";

  if (theLength < 0)
    theLength = strlen(theMessage);

  char buffer[64];

  if (myLongFormat != OTCLIB_FALSE)
  {
    time_t secs = 0;
    secs = time(0);
    struct tm* timetm = localtime(&secs);

    int theProcessId = 0;
#if defined(SYS_UNIX) && defined(HAVE_GETPID)
    theProcessId = getpid();
#endif

    int theThreadId = 0;
#if defined(HAVE_SOLARIS_THREADS)
     theThreadId = (int)thr_self();
#else
#if defined(HAVE_POSIX_THREADS)
    pthread_t theThread;
    theThread = pthread_self();
    theThreadId = pthread_getunique_np(&theThread);
#else
#if defined(HAVE_VXWORKS_THREADS)
    theThreadId = (int)taskIdSelf();
#endif
#endif
#endif

    sprintf(
     buffer,
     "%s %2d %.2d:%.2d:%.2d [%d/%d] %s: ",
     OTCLIB_SHORTMONTHS[timetm->tm_mon],
     timetm->tm_mday,
     timetm->tm_hour,
     timetm->tm_min,
     timetm->tm_sec,
     theProcessId,
     theThreadId,
     OTCLIB_LOGLEVELMESSAGES[theLevel]
    );
  }
  else
    sprintf(buffer,"%s: ",OTCLIB_LOGLEVELMESSAGES[theLevel]);

  // Log message to standard error.

  if (myStream != stderr || myUseStderr != OTCLIB_FALSE)
  {
    char const* theMatch;
    char const* theText;

    theText = theMessage;

    theMatch = strchr(theText,EOL);
    while (theMatch != 0)
    {
      fwrite(buffer,1,strlen(buffer),myStream);
      fwrite((char*)theText,1,theMatch-theText+1,myStream);
      theText = theMatch + 1;
      theMatch = strchr(theText,EOL);
    }

    fwrite(buffer,1,strlen(buffer),myStream);
    fwrite((char*)theText,1,theMessage+theLength-theText,myStream);
    fwrite("\n",1,1,myStream);
    fflush(myStream);
  }

  // Log message to log file.

  if (myLogFile != 0)
  {
    char const* theFile = 0;
#if defined(SYS_UNIX)
    // Make assumption that we will have enough space. Bad !!!!

    char buf[MAXPATHLEN+1];
    char const* p = myLogFile;
    char* q = buf;
    while (*p)
    {
      if (*p != '%')
      {
	*q++ = *p++;
      }
      else
      {
	p++;
	if (*p != EOS)
	{
	  if (*p == 'p')
	  {
	    sprintf(q,"%d",(int)getpid());
	    while (*q != EOS)
	      q++;
	  }
	  else if (*p == 'h')
	  {
	    strcpy(q,myHostName);
	    while (*q != EOS)
	      q++;
	  }
	  else if (*p == '%')
	  {
	    *q++ = '%';
	  }

	  p++;
	}
      }
    }
    *q = EOS;

    theFile = buf;
#else
    theFile = myLogFile;
#endif

    FILE* fp = fopen((char*)theFile,"a+");
    if (fp != 0)
    {
      char const* theMatch;
      char const* theText;

      theText = theMessage;

      theMatch = strchr(theText,EOL);
      while (theMatch != 0)
      {
	fwrite(buffer,1,strlen(buffer),fp);
	fwrite((char*)theText,1,theMatch-theText+1,fp);
	theText = theMatch + 1;
	theMatch = strchr(theText,EOL);
      }

      fwrite(buffer,1,strlen(buffer),fp);
      fwrite((char*)theText,1,theMessage+theLength-theText,fp);
      fwrite("\n",1,1,fp);
      fclose(fp);
    }
  }

  // Now send to all user defined loggers. Make sure we are not in a loop.
  // If we are in a loop then don't call user loggers again.

  static int loop = 0;
  if (loop == 0)
  {
    loop++;
    OTC_Logger* theLogger = myHead;
    while (theLogger)
    {
      theLogger->log(theTarget,theLevel,theMessage,theLength);
      theLogger = theLogger->myNext;
    }
    loop--;
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Logger::stderrOutputEnabled()
{
  OTC_Boolean theState;

  _mutex.lock();

  initialise();

  theState = myUseStderr;

  _mutex.unlock();

  return theState;
}

/* ------------------------------------------------------------------------- */
void OTC_Logger::enableStderrOutput()
{
  _mutex.lock();

  initialise();

  myUseStderr = OTCLIB_TRUE;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTC_Logger::disableStderrOutput()
{
  _mutex.lock();

  initialise();

  myUseStderr = OTCLIB_FALSE;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
char const* OTC_Logger::logFile()
{
  char const* theFile;

  _mutex.lock();

  initialise();

  theFile = myLogFile;

  _mutex.unlock();

  return theFile;
}

/* ------------------------------------------------------------------------- */
void OTC_Logger::setLogFile(char const* theFile)
{
  _mutex.lock();

  initialise();

  myLogFile = theFile;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_Logger::OTC_Logger()
 : myNext(0), myPrev(0)
{
  _mutex.lock();

  initialise();

  link();

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTC_Logger::link()
{
  // Assumed that this function is called from inside lock.

  if (!myHead)
  {
    myHead = this;
  }
  else
  {
    myNext = myHead;
    myNext->myPrev = this;
    myHead = this;
  }
}

/* ------------------------------------------------------------------------- */
void OTC_Logger::unlink()
{
  // Assumed that this function is called from inside lock.

  if (myHead == this)
  {
    myHead = myNext;
  }
  else
  {
    myPrev->myNext = myNext;
    if (myNext)
      myNext->myPrev = myPrev;
  }
}

/* ------------------------------------------------------------------------- */
