/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     debug/otctracer.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/debug/tracer.hh"
#endif

#include <OTC/debug/tracer.hh>
#include <OTC/debug/logstrm.hh>

#include <fstream.h>
#include <stdlib.h>

/* ------------------------------------------------------------------------- */
OTC_Mutex OTC_Tracer::_mutex;
int OTC_Tracer::globInit = 0;
int OTC_Tracer::globLevel = 0;
OTC_Boolean OTC_Tracer::globTraceAll = OTCLIB_FALSE;
OTC_Boolean OTC_Tracer::globInfo = OTCLIB_FALSE;
ostream* OTC_Tracer::globStream = 0;
OTC_Boolean OTC_Tracer::globLogTrace = OTCLIB_FALSE;
OTC_LogStream* OTC_Tracer::globLogStream = 0;
OTC_Boolean OTC_Tracer::globIndent = OTCLIB_FALSE;
OTC_TraceStream* OTC_Tracer::globTraceStream = 0;
char* OTC_Tracer::globLogBuf = 0;
OTC_Tracer* OTC_Tracer::globLast = 0;

/* ------------------------------------------------------------------------- */
OTC_Tracer::OTC_Tracer(
 char const* thePrototype,
 char const* theFile,
 u_int theLine,
 int theEnable
)
  : myPrototype(thePrototype), myEnable(OTCLIB_TRUE), myPrev(0)
{
  _mutex.lock();

  initialise();

#if !defined(ENV_THREADS)
  myPrev = globLast;
  globLast = this;
#endif

  if (theEnable < 0 || theEnable > globLevel)
    myEnable = OTCLIB_FALSE;

  if (globTraceAll || myEnable)
  {
    if (myPrototype != 0 && globInfo && theFile != 0)
    {
      *globTraceStream << "@location - \"" <<
       theFile << "\", line " << theLine << endl;
    }

    if (myPrototype != 0)
      *globTraceStream << "@enter - " << prototype() << endl;
  }

  *globTraceStream << OTCLIB_TRACEIN;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_Tracer::OTC_Tracer(char const* thePrototype)
  : myPrototype(thePrototype), myEnable(OTCLIB_TRUE), myPrev(0)
{
  _mutex.lock();

  initialise();

#if !defined(ENV_THREADS)
  myPrev = globLast;
  globLast = this;
#endif

  if (globTraceAll || myEnable)
  {
    if (myPrototype != 0)
      *globTraceStream << "@enter - " << prototype() << endl;
  }

  *globTraceStream << OTCLIB_TRACEIN;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTC_Tracer::initialise()
{
  // This function does not lock the mutex as it is expected that the
  // calling function has locked it.

  if (globInit == 0)
  {
    globInit = 1;

    char* s;

    globTraceStream = new OTC_TraceStream;
    OTCLIB_ASSERT(globTraceStream != 0);

    globLogBuf = new char[2048];
    OTCLIB_ASSERT(globLogBuf != 0);
    globLogStream = new OTC_LogStream(globLogBuf,2048);
    OTCLIB_ASSERT(globLogStream != 0);
    globLogStream->setTarget("trace");
    globLogStream->setLevel(OTCLIB_LOG_DEBUG);

    globIndent = OTCLIB_TRUE;
    s = getenv("OTCLIB_NOTRACEINDENT");
    if (s != 0)
      globIndent = OTCLIB_FALSE;

    if (globStream == 0)
    {
      s = getenv("OTCLIB_TRACEFD");
      if (s != 0)
      {
	int fd = atoi(s);
	ofstream* outs = new ofstream(fd);
	OTCLIB_ASSERT(outs != 0);
	if (outs->fail())
	  delete outs;
	else
	  globStream = outs;
      }
    }

    if (globStream == 0)
    {
      s = getenv("OTCLIB_LOGTRACE");
      if (s != 0)
	globLogTrace = OTCLIB_TRUE;
    }

    if (globStream == 0)
    {
      s = getenv("OTCLIB_TRACEFILE");
      if (s != 0)
      {
	ofstream* outs = new ofstream(s,ios::trunc,0666);
	OTCLIB_ASSERT(outs != 0);
	if (outs->fail())
	  delete outs;
	else
	  globStream = outs;
      }
    }

    if (globStream == 0)
      globStream = &clog;

    if (globLevel == 0)
    {
      s = getenv("OTCLIB_TRACELEVEL");
      if (s == 0)
	globLevel = 0;
      else
	globLevel = atoi(s);
    }

    s = getenv("OTCLIB_TRACEINFO");
    if (s != 0)
      globInfo = OTCLIB_TRUE;
    else
      globInfo = OTCLIB_FALSE;

    s = getenv("OTCLIB_GLOBALTRACE");
    if (s != 0)
      globTraceAll = OTCLIB_TRUE;
    else
      globTraceAll = OTCLIB_FALSE;
  }
}

/* ------------------------------------------------------------------------- */
OTC_Tracer::~OTC_Tracer()
{
  _mutex.lock();

  *globTraceStream << OTCLIB_TRACEOUT;

#if !defined(ENV_THREADS)
  globLast = myPrev;
#endif

  if (globTraceAll || myEnable)
  {
    if (myPrototype != 0)
      *globTraceStream << "@exit - " << prototype() << endl;
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Tracer::enabled(int theLevel)
{
  OTC_Boolean theState;
  theState = OTCLIB_TRUE;

  _mutex.lock();

  if (((theLevel < 0) || ((theLevel >= 0) && (theLevel > globLevel)))
   && (globTraceAll == OTCLIB_FALSE))
  {
    theState = OTCLIB_FALSE;
  }

  _mutex.unlock();

  return theState;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Tracer::globalTrace()
{
  OTC_Boolean theState;

  _mutex.lock();

  initialise();

  theState = globTraceAll;

  _mutex.unlock();

  return theState;
}

/* ------------------------------------------------------------------------- */
void OTC_Tracer::enableGlobalTrace()
{
  _mutex.lock();

  initialise();

  globTraceAll = OTCLIB_TRUE;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTC_Tracer::disableGlobalTrace()
{
  _mutex.lock();

  initialise();

  globTraceAll = OTCLIB_FALSE;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
int OTC_Tracer::level()
{
  int theLevel;

  _mutex.lock();

  initialise();

  theLevel = globLevel;

  _mutex.unlock();

  return theLevel;
}

/* ------------------------------------------------------------------------- */
void OTC_Tracer::setLevel(int theLevel)
{
  _mutex.lock();

  initialise();

  globLevel = theLevel;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Tracer::traceIndent()
{
  OTC_Boolean theState;

  _mutex.lock();

  initialise();

  theState = globIndent;

  _mutex.unlock();

  return theState;
}

/* ------------------------------------------------------------------------- */
void OTC_Tracer::enableTraceIndent()
{
  _mutex.lock();

  initialise();

  globIndent = OTCLIB_TRUE;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTC_Tracer::disableTraceIndent()
{
  _mutex.lock();

  initialise();

  globIndent = OTCLIB_FALSE;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_Tracer const* OTC_Tracer::last()
{
#if !defined(ENV_THREADS)
  return globLast;
#else
  return 0;
#endif
}

/* ------------------------------------------------------------------------- */
ostream* OTC_Tracer::stream()
{
  ostream* theStream;

  _mutex.lock();

  if (globLogTrace == OTCLIB_FALSE)
    theStream = globStream;
  else
    theStream = globLogStream;

  _mutex.unlock();

  return theStream;
}

/* ------------------------------------------------------------------------- */
void OTC_Tracer::setStream(ostream* theStream)
{
  _mutex.lock();

  globStream = theStream;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Tracer::logTrace()
{
  OTC_Boolean theState;

  _mutex.lock();

  initialise();

  theState = globLogTrace;

  _mutex.unlock();

  return theState;
}

/* ------------------------------------------------------------------------- */
void OTC_Tracer::enableLogTrace()
{
  _mutex.lock();

  initialise();

  if (globLogTrace == OTCLIB_FALSE)
  {
    globLogTrace = OTCLIB_TRUE;
    globStream->flush();
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTC_Tracer::disableLogTrace()
{
  _mutex.lock();

  initialise();

  if (globLogTrace != OTCLIB_FALSE)
  {
    globLogTrace = OTCLIB_FALSE;
    globLogStream->flush();
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
ostream& otclib_tracer()
{
  ostream* theStream;

  OTC_Tracer::_mutex.lock();

  OTC_Tracer::initialise();

  theStream = OTC_Tracer::globTraceStream;

  OTC_Tracer::_mutex.unlock();

  return *theStream;
}

/* ------------------------------------------------------------------------- */
