/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     dispatch/otcalarm.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/dispatch/alarm.hh"
#endif

#include <OTC/dispatch/alarm.hh>
#include <OTC/dispatch/eventjob.hh>

#include <iostream.h>
#include <time.h>

/* ------------------------------------------------------------------------- */
class OTC_AlarmSubscription
{
  public:

    void*		operator new(size_t theSize)
				{ return OTC_CommonPool::allocate(theSize); }

    void		operator delete(void* theMem, size_t theSize)
				{ OTC_CommonPool::release(theMem,theSize); }

    int			alarmId;

    int			agentId;

    OTC_Boolean		expired;

    long		time;

    OTC_AlarmSubscription*	next;
};

/* ------------------------------------------------------------------------- */
OTC_NRMutex OTCEV_Alarm::_mutex;
int OTCEV_Alarm::globAlarmIdCount = 0;
OTC_AlarmSubscription* OTCEV_Alarm::globSubscriptions = 0;
int OTCEV_Alarm::globTypeId = 0;

/* ------------------------------------------------------------------------- */
OTCEV_Alarm::~OTCEV_Alarm()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
void* OTCEV_Alarm::type() const
{
  return typeId();
}

/* ------------------------------------------------------------------------- */
void OTCEV_Alarm::dump(ostream& outs) const
{
  outs << "<OTC> ALARM - alarm = " << alarm() << ", time = " << time();
}

/* ------------------------------------------------------------------------- */
int OTCEV_Alarm::set(int theAgentId, long theTime)
{
  OTCLIB_ENSURE((theTime >= 0),
   "OTCEV_Alarm::set() - invalid time");
  OTCLIB_ENSURE((theAgentId != 0),
   "OTCEV_Alarm::set() - invalid agent ID");

  OTC_AlarmSubscription* aSubscription;
  aSubscription = new OTC_AlarmSubscription;
  OTCLIB_ASSERT(aSubscription != 0);

  _mutex.lock();

  if (globAlarmIdCount == 0)
    globAlarmIdCount++;

  aSubscription->alarmId = globAlarmIdCount++;
  aSubscription->agentId = theAgentId;
  aSubscription->expired = (theTime == 0) ? OTCLIB_TRUE : OTCLIB_FALSE;
  aSubscription->time = theTime;
  aSubscription->next = 0;

  if (globSubscriptions == 0)
  {
    globSubscriptions = aSubscription;
  }
  else if (theTime < globSubscriptions->time)
  {
    aSubscription->expired = globSubscriptions->expired;
    aSubscription->next = globSubscriptions;
    globSubscriptions = aSubscription;
  }
  else
  {
    OTC_AlarmSubscription* tmpSubscription1;
    OTC_AlarmSubscription* tmpSubscription2;
    tmpSubscription1 = globSubscriptions->next;
    tmpSubscription2 = globSubscriptions;

    while (tmpSubscription1 != 0 && theTime >= tmpSubscription1->time)
    {
      tmpSubscription2 = tmpSubscription1;
      tmpSubscription1 = tmpSubscription1->next;
    }

    aSubscription->next = tmpSubscription1;
    tmpSubscription2->next = aSubscription;

    if (theTime != 0)
    {
      if (tmpSubscription2->time == theTime)
	aSubscription->expired = tmpSubscription2->expired;
      else if (tmpSubscription1 != 0)
	aSubscription->expired = tmpSubscription1->expired;
    }
  }

  _mutex.unlock();

  return aSubscription->alarmId;
}

/* ------------------------------------------------------------------------- */
void OTCEV_Alarm::cancel(int theAlarmId)
{
  if (theAlarmId == 0)
    return;

  _mutex.lock();

  if (globSubscriptions != 0)
  {
    OTC_AlarmSubscription* tmpSubscription1;
    OTC_AlarmSubscription* tmpSubscription2;

    if (globSubscriptions->alarmId == theAlarmId)
    {
      tmpSubscription1 = globSubscriptions;
      globSubscriptions = globSubscriptions->next;
      delete tmpSubscription1;
    }
    else
    {
      tmpSubscription1 = globSubscriptions->next;
      tmpSubscription2 = globSubscriptions;

      while (tmpSubscription1 != 0)
      {
	if (tmpSubscription1->alarmId == theAlarmId)
	{
	  tmpSubscription2->next = tmpSubscription1->next;
	  delete tmpSubscription1;
	  break;
	}

	tmpSubscription2 = tmpSubscription1;
	tmpSubscription1 = tmpSubscription1->next;
      }
    }
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTCEV_Alarm::cancelAgent(int theAgentId)
{
  if (theAgentId == 0)
    return;

  _mutex.lock();

  if (globSubscriptions != 0)
  {
    OTC_AlarmSubscription* tmpSubscription1;
    OTC_AlarmSubscription* tmpSubscription2;

    tmpSubscription1 = globSubscriptions;
    tmpSubscription2 = 0;

    while (tmpSubscription1 != 0)
    {
      if (tmpSubscription1->agentId == theAgentId)
      {
	if (tmpSubscription2 != 0)
	{
	  tmpSubscription2->next = tmpSubscription1->next;
	  delete tmpSubscription1;
	  tmpSubscription1 = tmpSubscription2->next;
	}
	else
	{
	  globSubscriptions = tmpSubscription1->next;
	  delete tmpSubscription1;
	  tmpSubscription1 = globSubscriptions;
	}
      }
      else
      {
	tmpSubscription2 = tmpSubscription1;
	tmpSubscription1 = tmpSubscription1->next;
      }
    }
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTCEV_Alarm::active(int theAlarmId)
{
  if (theAlarmId == 0)
    return OTCLIB_FALSE;

  OTC_Boolean theStatus;
  theStatus = OTCLIB_FALSE;

  _mutex.lock();

  if (globSubscriptions != 0)
  {
    OTC_AlarmSubscription* aSubscription;
    aSubscription = globSubscriptions;

    while (aSubscription != 0)
    {
      if (aSubscription->alarmId == theAlarmId)
      {
	theStatus = OTCLIB_TRUE;
	break;
      }

      aSubscription = aSubscription->next;
    }
  }

  _mutex.unlock();

  return theStatus;
}

/* ------------------------------------------------------------------------- */
long OTCEV_Alarm::period()
{
  long theResult;

  _mutex.lock();

  if (globSubscriptions == 0)
  {
    theResult = -1;
  }
  else
  {
    long theTime = ::time(0);

    if (globSubscriptions->time < theTime)
      theResult = 0;
    else
      theResult = globSubscriptions->time - theTime;
  }

  _mutex.unlock();

  return theResult;
}

/* ------------------------------------------------------------------------- */
OTC_Job* OTCEV_Alarm::pending()
{
  OTC_Job* theJob;
  theJob = 0;

  _mutex.lock();

  if (globSubscriptions != 0)
  {
    if (globSubscriptions->expired == OTCLIB_FALSE)
    {
      long theTime = ::time(0);

      OTC_AlarmSubscription* tmpSubscription1;
      tmpSubscription1 = globSubscriptions;

      while (tmpSubscription1 != 0 && theTime >= tmpSubscription1->time)
      {
	tmpSubscription1->expired = OTCLIB_TRUE;
	tmpSubscription1 = tmpSubscription1->next;
      }
    }

    if (globSubscriptions->expired != OTCLIB_FALSE)
    {
      OTC_AlarmSubscription* tmpSubscription1;
      tmpSubscription1 = globSubscriptions->next;

      OTCEV_Alarm* theEvent;
      theEvent = new OTCEV_Alarm(globSubscriptions->alarmId,
       globSubscriptions->time);
      OTCLIB_ASSERT(theEvent != 0);

      theJob = new OTC_EventJob(globSubscriptions->agentId,theEvent);
      OTCLIB_ASSERT(theJob != 0);

      delete globSubscriptions;
      globSubscriptions = tmpSubscription1;
    }
  }

  _mutex.unlock();

  return theJob;
}

/* ------------------------------------------------------------------------- */
