/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     dispatch/otcioevent.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/dispatch/ioevent.hh"
#endif

#include <OTC/dispatch/ioevent.hh>
#include <OTC/dispatch/eventjob.hh>
#include <OTC/dispatch/jobqueue.hh>
#include <OTC/dispatch/evagent.hh>
#include <OTC/debug/logstrm.hh>

#include <iostream.h>
#include <stdio.h>

#if defined(SYS_UNIX)
#include <sys/param.h>
#endif

#ifndef NOFILE
#ifdef OPEN_MAX
#define NOFILE OPEN_MAX
#else
#ifdef FOPEN_MAX
#define NOFILE FOPEN_MAX
#else
#ifdef NFDS
#define NOFILE NFDS
#else
#ifdef _NFILE
#define NOFILE _NFILE
#else
#ifdef _NFILE_
#define NOFILE _NFILE_
#endif
#endif
#endif
#endif
#endif
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYSCONF
extern "C" long sysconf(int);
#endif

// If we do not have sysconf() and NOFILE still isn't set then tough. :-)

/* ------------------------------------------------------------------------- */
class OTC_IOEventSubscription
{
  public:

    int			agentId;

    int			events;
};

/* ------------------------------------------------------------------------- */
OTC_NRMutex OTCEV_IOEvent::_mutex;
int OTCEV_IOEvent::globLimit = 0;
OTC_IOEventSubscription* OTCEV_IOEvent::globSubscriptions = 0;
int OTCEV_IOEvent::globTypeId = 0;

/* ------------------------------------------------------------------------- */
OTCEV_IOEvent::~OTCEV_IOEvent()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
void* OTCEV_IOEvent::type() const
{
  return &globTypeId;
}

/* ------------------------------------------------------------------------- */
void OTCEV_IOEvent::dump(ostream& outs) const
{
  outs << "<OTC> IOEVENT - fd = " << fd() << ", events = " << events();
}

/* ------------------------------------------------------------------------- */
int OTCEV_IOEvent::agent(int theFd)
{
  int theResult;
  theResult = 0;

  _mutex.lock();

  if (theFd >= 0 && theFd < nfds() && globSubscriptions != 0)
    theResult = globSubscriptions[theFd].agentId;

  _mutex.unlock();

  return theResult;
}

/* ------------------------------------------------------------------------- */
void OTCEV_IOEvent::subscribe(int theAgentId, int theFd, int theEvents)
{
  OTCLIB_ENSURE((theFd >= 0 && theFd < nfds()),
   "OTCEV_IOEvent::subscribe() - invalid file descriptor");
  OTCLIB_ENSURE((theAgentId != 0),
   "OTCEV_IOEvent::subscribe() - invalid agent ID");
  
  theEvents &= (OTCLIB_POLLIN | OTCLIB_POLLPRI | OTCLIB_POLLOUT);

  if (theEvents == 0)
    return;

  _mutex.lock();

  if (globSubscriptions == 0)
  {
    globSubscriptions = new OTC_IOEventSubscription[nfds()];
    OTCLIB_ASSERT(globSubscriptions != 0);
    for (int i=nfds(); i>0; i--)
    {
      globSubscriptions[i-1].agentId = 0;
      globSubscriptions[i-1].events = 0;
      globLimit = -1;
    }
  }

  int tmpAgentId = globSubscriptions[theFd].agentId;
  OTCLIB_ENSURE((tmpAgentId == 0 || tmpAgentId == theAgentId),
   "OTCEV_IOEvent::subscribe() - file descriptor subscribed by other agent");

  if (theFd > globLimit)
    globLimit = theFd;

  globSubscriptions[theFd].agentId = theAgentId;
  globSubscriptions[theFd].events |= theEvents;

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTCEV_IOEvent::unsubscribe(int theAgentId, int theFd, int theEvents)
{
  if (theAgentId == 0 || theFd < 0 || theFd >= nfds())
    return;

  theEvents &= (OTCLIB_POLLIN | OTCLIB_POLLPRI | OTCLIB_POLLOUT);

  if (theEvents == 0)
    return;

  _mutex.lock();

  if (globSubscriptions != 0 && globSubscriptions[theFd].events != 0)
  {
    if (globSubscriptions[theFd].agentId == theAgentId)
    {
      int tmpEvents = globSubscriptions[theFd].events & ~theEvents;
      tmpEvents &= (OTCLIB_POLLIN | OTCLIB_POLLPRI | OTCLIB_POLLOUT);
      globSubscriptions[theFd].events = tmpEvents;

      if (globSubscriptions[theFd].events == 0)
      {
	globSubscriptions[theFd].agentId = 0;
	while (globLimit >= 0 && globSubscriptions[globLimit].events == 0)
	  globLimit--;
      }
    }
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTCEV_IOEvent::unsubscribeFd(int theFd)
{
  if (theFd < 0 || theFd >= nfds())
    return;

  _mutex.lock();

  if (globSubscriptions != 0 && globSubscriptions[theFd].events != 0)
  {
    globSubscriptions[theFd].agentId = 0;
    globSubscriptions[theFd].events = 0;

    while (globLimit >= 0 && globSubscriptions[globLimit].events == 0)
      globLimit--;
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OTCEV_IOEvent::unsubscribeAgent(int theAgentId)
{
  if (theAgentId == 0)
    return;

  for (int i=0; i<=maxFd(); i++)
    unsubscribe(theAgentId,i);
}

/* ------------------------------------------------------------------------- */
OTC_Job* OTCEV_IOEvent::job(int theFd, int theEvents)
{
  OTC_EventJob* theJob;
  theJob = 0;

  _mutex.lock();

  if (globSubscriptions != 0 && globSubscriptions[theFd].agentId != 0)
  {
    int tmpEvents = 0;
    tmpEvents |= theEvents & globSubscriptions[theFd].events;
    tmpEvents |= theEvents &
     (OTCLIB_POLLERR | OTCLIB_POLLHUP | OTCLIB_POLLNVAL);

    if (tmpEvents != 0)
    {
      OTCEV_IOEvent* theEvent;
      theEvent = new OTCEV_IOEvent(theFd,tmpEvents);
      OTCLIB_ASSERT(theEvent != 0);

      theJob = new OTC_EventJob(globSubscriptions[theFd].agentId,theEvent);
      OTCLIB_ASSERT(theJob != 0);
    }
  }

  _mutex.unlock();

  return theJob;
}

/* ------------------------------------------------------------------------- */
int OTCEV_IOEvent::events(int theFd)
{
  if (theFd < 0 || theFd >= nfds())
    return 0;

  int theResult;
  theResult = 0;

  _mutex.lock();

  if (globSubscriptions != 0)
    theResult = globSubscriptions[theFd].events;

  _mutex.unlock();

  return theResult;
}

/* ------------------------------------------------------------------------- */
void OTCEV_IOEvent::cancelSource(int theAgentId)
{
  OTCEV_IOEvent::unsubscribeAgent(theAgentId);

  OTCLIB_LOGGER(OTCLIB_LOG_WARNING) <<
   "OTCEV_IOEvent::cancelSource() - ioevent subscription cancelled" << flush;
  OTCLIB_LOGGER(OTCLIB_LOG_DEBUG) <<
   "Agent " << theAgentId << flush;
}

/* ------------------------------------------------------------------------- */
int OTCEV_IOEvent::nfds()
{
  // Assumed that this is always called from inside a lock.
  // Not really supposed to cache these values as they could
  // change if resource limits of a program are changed. But
  // then we base the size of an array on the first call and
  // thus probably do not want this to change, ie., get larger.
  // Can only assume that when first called that limit is at
  // the max it will for the program.

#ifdef HAVE_SYSCONF_SC_OPEN_MAX
  static int _nfds = (int)sysconf(_SC_OPEN_MAX);
#else
#ifdef HAVE_GETDTABLESIZE
  static int _nfds = getdtablesize();
#else
  static int _nfds = NOFILE;
#endif
#endif
  return _nfds;
}

/* ------------------------------------------------------------------------- */
