/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     file/otcpathname.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1992 OTC LIMITED
//     Copyright 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// = NOTES
//     In DEC C++ the <access()> function has prototype where the first
//     argument is of type <char*> instead of <char const*>. Thus need
//     to forcibly cast to <char*> in calls to that function.
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/files/pathname.hh"
#endif

#include <OTC/files/pathname.hh>

#include <stdlib.h>
#include <string.h>

#if defined(SYS_UNIX)

#include <unistd.h>
#if defined(CXX_ATT2_1) || defined(CXX_CL1_1)
#include <osfcn.h>
#include <sys/file.h>
#endif

#else

#ifdef __BORLANDC__
#include <io.h>
#endif

#ifdef __IBMCPP__
#include <io.h>
#endif

#ifdef __WATCOMC__
#include <io.h>
#endif

#ifdef _MSC_VER
#include <io.h>
#include <direct.h>
#ifdef __STDC__
#ifndef access
#define access _access
#endif
#endif
#endif

#endif

#ifndef F_OK
#define F_OK 00
#define X_OK 01
#define W_OK 02
#define R_OK 04
#endif

#if defined(HAVE_GETWD) && (defined(CXX_OS) || defined(CXX_DEC))
extern "C" char* getwd(char*);
#endif

#if defined(ENV_OSTORE)
/* ------------------------------------------------------------------------- */
os_typespec* OTC_Pathname::typespec()
{
  static os_typespec ts("OTC_Pathname");
  return &ts;
}
#endif

/* ------------------------------------------------------------------------- */
OTC_Pathname::~OTC_Pathname()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_Pathname::OTC_Pathname()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_Pathname::OTC_Pathname(char const* thePath)
  : myPath(thePath)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_Pathname::OTC_Pathname(OTC_String const& thePath)
  : myPath(thePath)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_Pathname::OTC_Pathname(OTC_Pathname const& thePath)
  : myPath(thePath.path())
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Pathname::isAccessable() const
{
  return !access((char*)path().string(),F_OK);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Pathname::isWritable() const
{
  return !access((char*)path().string(),W_OK);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Pathname::isReadable() const
{
  return !access((char*)path().string(),R_OK);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Pathname::isExecutable() const
{
  return !access((char*)path().string(),X_OK);
}

/* ------------------------------------------------------------------------- */
OTC_RString OTC_Pathname::rawString() const
{
  return path().rawString();
}

/* ------------------------------------------------------------------------- */
OTC_Pathname OTC_Pathname::absolute() const
{
  if (isAbsolute())
    return *this;

  else if (path().isEmpty())
    return OTC_String::nullString();

  else
  {
    char buf[MAXPATHLEN];
    char const* res;
#ifdef HAVE_GETWD
    res = (char*)getwd((char*)buf);
#else
    res = getcwd(buf,MAXPATHLEN);
#endif
    if (res == 0 || *buf == EOS)
      OTCLIB_EXCEPTION("Can't get name of current directory");

#ifndef SYS_UNIX
    char* p = buf;
    while (*p != EOS)
    {
      if (*p == '\\')
        *p = '/';
      p++;
    }
#endif

    OTC_String aString = buf;

    if (!path().isEmpty() && path() != ".")
    {
      aString.append('/');
      aString.append(path());
    }

    return aString;
  }
}

/* ------------------------------------------------------------------------- */
OTC_Pathname OTC_Pathname::dirname() const
{
  if (path().isEmpty())
    return OTC_String::nullString();

  else
  {
    OTC_String aString = path();

    // Remove trailing slashes.

    if (aString[aString.length()-1] == '/')
    {
      u_int i=0;
      while (i<aString.length() && aString[aString.length()-1-i] == '/')
	i++;
      aString.truncate(aString.length()-i);
    }

    // Check for empty string, meaning that original path represented
    // root directory.

    if (aString.isEmpty())
    {
      aString = "/";
      return aString;
    }

    // Remove last pathname component.

    u_int j=0;
    while (j<aString.length() && aString[aString.length()-1-j] != '/')
      j++;
    if (j > 0)
      aString.truncate(aString.length()-j);

    // Check for empty string, meaning that original path was name of
    // file in current directory.

    if (aString.isEmpty())
    {
      aString = ".";
      return aString;
    }

    // Remove trailing slashes again.

    u_int k=0;
    while (k<aString.length() && aString[aString.length()-1-k] == '/')
      k++;
    if (k > 0)
      aString.truncate(aString.length()-k);

    // Check for empty string, meaning that original path was name of
    // file in root directory.

    if (aString.isEmpty())
    {
      aString = "/";
      return aString;
    }

    // Whats left is the directory portion.

    return aString;
  }
}

/* ------------------------------------------------------------------------- */
OTC_Pathname OTC_Pathname::basename() const
{
  if (path().isEmpty())
    return OTC_String::nullString();

  else
  {
    OTC_Record record(path(),'/');
    return record.field(record.numFields());
  }
}

/* ------------------------------------------------------------------------- */
OTC_String OTC_Pathname::basename(char const* theSuffix) const
{
  if (path().isEmpty())
    return OTC_String::nullString();
  
  if (theSuffix == 0 || *theSuffix == EOS)
    return basename().path();

  else
  {
    OTC_String aString(basename().path());

    if (aString.isEmpty())
      return OTC_String::nullString();

    u_int theLength = strlen(theSuffix);

    if (theLength > aString.length())
      return basename().path();

    u_int theStart = aString.length() - theLength;
    OTC_String aSuffix(aString.section(theStart,theLength));

    if (aSuffix == theSuffix)
      aString.truncate(theStart);

    return aString;
  }
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Pathname::isAbsolute() const
{
  if (path().isEmpty())
    return OTCLIB_FALSE;

  if (path()[(u_int)0] == '/')
    return OTCLIB_TRUE;

#ifndef SYS_UNIX

  if (path().length() >= 3 && path()[(u_int)1] == ':')
  {
    // Assuming ASCII character ordering.

    char theDrive = path()[(u_int)0];
    if (theDrive >= 'a' && theDrive <= 'z')
      return OTCLIB_TRUE;
    if (theDrive >= 'A' && theDrive <= 'Z')
      return OTCLIB_TRUE;
  }

#endif

  return OTCLIB_FALSE;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Pathname::isRelative() const
{
  return path().isEmpty() ? OTCLIB_FALSE : path()[(u_int)0] != '/';
}

#if defined(CXX_OS)
/* ------------------------------------------------------------------------- */
int OTC_RankActions<OTC_Pathname>::rank(
 OTC_Pathname const& p1,
 OTC_Pathname const& p2
)
{
  return OTC_RankActions<OTC_String>::rank(p1.path(),p2.path());
}

/* ------------------------------------------------------------------------- */
int OTC_HashActions<OTC_Pathname>::hash(OTC_Pathname const& p)
{
  return OTC_HashActions<OTC_String>::hash(p.path());
}
#endif

/* ------------------------------------------------------------------------- */
