/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     memory/otcarena.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/memory/arena.hh"
#endif

#include <OTC/memory/arena.hh>

#include <stdlib.h>

#if defined(ENV_OSTORE)
/* ------------------------------------------------------------------------- */
os_typespec* OTC_ArenaBlock::typespec()
{
  static os_typespec ts("OTC_ArenaBlock");
  return &ts;
}
#endif

/* ------------------------------------------------------------------------- */
OTC_ArenaBlock::~OTC_ArenaBlock()
{
  // Nothing to do.
}

#if defined(ENV_OSTORE)
/* ------------------------------------------------------------------------- */
os_typespec* OTC_Arena::typespec()
{
  static os_typespec ts("OTC_Arena");
  return &ts;
}
#endif

/* ------------------------------------------------------------------------- */
OTC_Arena::OTC_Arena(size_t theAlign)
  : myBlockSize(2040), mySlop(16), myAlign(theAlign), myFull(0), myFree(0)
{
  OTCLIB_ENSURE((theAlign != 0),
   "OTC_Arena::OTC_Arena() - invalid memory alignment");

  char const* szenv = 0;
  int tmpBlockSize = myBlockSize;
  szenv = getenv("OTCLIB_ARENABLOCKSIZE");
  if (szenv != 0)
    tmpBlockSize = atoi(szenv);

  int tmpSlop = mySlop;
  szenv = getenv("OTCLIB_ARENASLOPSIZE");
  if (szenv != 0)
    tmpSlop = atoi(szenv);

  OTCLIB_ENSURE((tmpBlockSize >= 0),
   "OTC_Arena::OTC_Arena() - invalid block size");
  OTCLIB_ENSURE((tmpSlop >= 0),
   "OTC_Arena::OTC_Arena() - invalid slop size");

  myBlockSize = tmpBlockSize;
  mySlop = tmpSlop;
}

/* ------------------------------------------------------------------------- */
OTC_Arena::OTC_Arena(size_t theBlockSize, size_t theSlop, size_t theAlign)
  : myBlockSize(theBlockSize), mySlop(theSlop), myAlign(theAlign),
    myFull(0), myFree(0)
{
  OTCLIB_ENSURE((theAlign != 0),
   "OTC_Arena::OTC_Arena() - invalid memory alignment");
}

/* ------------------------------------------------------------------------- */
OTC_Arena::~OTC_Arena()
{
  while (myFull != 0)
  {
    OTC_ArenaBlock* tmpBlock;
    tmpBlock = myFull->next;
    delete [] myFull->block;
    delete myFull;
    myFull = tmpBlock;
  }
  while (myFree != 0)
  {
    OTC_ArenaBlock* tmpBlock;
    tmpBlock = myFree->next;
    delete [] myFree->block;
    delete myFree;
    myFree = tmpBlock;
  }
}

/* ------------------------------------------------------------------------- */
void* OTC_Arena::allocate(size_t theSize)
{
  // Check if request is larger than block size. If it is then
  // don't both checking available memory, just allocate it.

  if (theSize > myBlockSize)
    return allocateNewBlock(theSize);

  // Allow for memory alignment.

  if ((theSize % myAlign) != 0)
    theSize = (theSize/myAlign + 1) * myAlign;

  // Check if we have any blocks.

  if (myFree == 0)
    return allocateNewBlock(theSize);

  // Search all the blocks.

  OTC_ArenaBlock* tmpBlock1;
  OTC_ArenaBlock* tmpBlock2;
  tmpBlock1 = myFree;
  tmpBlock2 = 0;

  while (tmpBlock1 != 0 && tmpBlock1->size < theSize)
  {
    tmpBlock2 = tmpBlock1;
    tmpBlock1 = tmpBlock1->next;
  }

  // Nothing available, allocate new block.

  if (tmpBlock1 == 0)
    return allocateNewBlock(theSize);

  // Allocate from block we found.

  char* tmpString;
  tmpString = tmpBlock1->free;
  tmpBlock1->free += theSize;
  tmpBlock1->size -= theSize;

  // If amount of available space drops below slop factor
  // move it to full block list and don't use it anymore.

  if (tmpBlock1->size <= mySlop)
  {
    if (tmpBlock2 == 0)
    {
      tmpBlock2 = myFull;
      myFull = tmpBlock1;
      myFree = tmpBlock1->next;
      tmpBlock1->next = tmpBlock2;
    }
    else
    {
      OTC_ArenaBlock* tmpBlock3;
      tmpBlock3 = myFull;
      myFull = tmpBlock1;
      tmpBlock2->next = tmpBlock1->next;
      tmpBlock1->next = tmpBlock3;
    }
  }

  return tmpString;
}

/* ------------------------------------------------------------------------- */
void* OTC_Arena::allocateNewBlock(size_t theSize)
{
  OTCLIB_ASSERT(theSize != 0);

  size_t allocSize = theSize;
  if (theSize < myBlockSize)
    allocSize = myBlockSize;

  OTC_ArenaBlock* tmpBlock;

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  tmpBlock = new ((os_segment*)theLocality) OTC_ArenaBlock;
#else
#if defined(ENV_OSTORE)
  tmpBlock = new (theLocality,OTC_ArenaBlock::get_os_typespec()) OTC_ArenaBlock;
#else
  tmpBlock = new OTC_ArenaBlock;
#endif
#endif
  OTCLIB_ASSERT(tmpBlock != 0);

#if defined(ENV_OSTORE_DML)
  tmpBlock->block = new ((os_segment*)theLocality) char[allocSize];
#else
#if defined(ENV_OSTORE)
  tmpBlock->block = new (theLocality,os_typespec::get_char(),allocSize)
   char[allocSize];
#else
  tmpBlock->block = new char[allocSize];
#endif
#endif
  OTCLIB_ASSERT(tmpBlock->block != 0);

  if (theSize > myBlockSize)
  {
    tmpBlock->free = 0;
    tmpBlock->size = 0;
    tmpBlock->next = myFull;
    myFull = tmpBlock;
  }
  else
  {
    tmpBlock->free = tmpBlock->block + theSize;
    tmpBlock->size = myBlockSize - theSize;

    if (tmpBlock->size <= mySlop)
    {
      tmpBlock->next = myFull;
      myFull = tmpBlock;
    }
    else
    {
      tmpBlock->next = myFree;
      myFree = tmpBlock;
    }
  }

  return tmpBlock->block;
}

/* ------------------------------------------------------------------------- */
