/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     text/otcrstring.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1993 1994 TELSTRA CORPORATION LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/text/rstring.hh"
#endif

#include <OTC/text/rstring.hh>
#include <OTC/collctn/copyactn.hh>

#include <iostream.h>
#include <stdlib.h>

/* ------------------------------------------------------------------------- */
OTC_NRMutex OTC_RString::_mutex;
OTC_NRMutex OTC_RString::_sdmutex;
OTC_Boolean OTC_RString::myInitialised = OTCLIB_FALSE;
OTC_BufferingType OTC_RString::myBufType = OTCLIB_16BUFFERING;

#if defined(ENV_OSTORE)
/* ------------------------------------------------------------------------- */
os_typespec* OTC_StringData::typespec()
{
  static os_typespec ts("OTC_StringData");
  return &ts;
}
#endif

/* ------------------------------------------------------------------------- */
OTC_StringData::~OTC_StringData()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_RString::~OTC_RString()
{
  _sdmutex.lock();

  if (myData->count == 1)
  {
    delete [] myData->data;
    delete myData;
  }
  else
    myData->count--;

  _sdmutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_RString::OTC_RString(
 u_int theLength,
 OTC_StringBuffering theType
)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myData = new ((os_segment*)theLocality) OTC_StringData;
#else
#if defined(ENV_OSTORE)
  myData = new (theLocality,OTC_StringData::get_os_typespec()) OTC_StringData;
#else
  myData = new OTC_StringData;
#endif
#endif
  OTCLIB_ASSERT(myData != 0);

  myData->count = 1;
  myData->length = theLength;
  myData->capacity = myData->length+1;
  myData->locked = OTCLIB_FALSE;
  myData->symbol = OTCLIB_FALSE;

  if (theType == OTCLIB_BUFFERED)
  {
    myData->capacity = capacity(myData->length)+1;
    OTCLIB_ASSERT(myData->capacity >= myData->length+1);
  }

#if defined(ENV_OSTORE_DML)
    myData->data = new ((os_segment*)theLocality) char[myData->capacity];
#else
#if defined(ENV_OSTORE)
  myData->data = new (theLocality,os_typespec::get_char(),
   myData->capacity) char[myData->capacity];
#else
  myData->data = new char[myData->capacity];
#endif
#endif
  OTCLIB_ASSERT(myData->data != 0);

  myData->data[myData->length] = EOS;
}

/* ------------------------------------------------------------------------- */
OTC_RString::OTC_RString(
 u_int theLength,
 void const* theObject,
 OTC_StringBuffering theType
)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality;
  if (theObject != 0)
    theLocality = OTC_Locality::of(theObject);
  else
    theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
    myData = new ((os_segment*)theLocality) OTC_StringData;
#else
#if defined(ENV_OSTORE)
  myData = new (theLocality,OTC_StringData::get_os_typespec()) OTC_StringData;
#else
  myData = new OTC_StringData;
#endif
#endif
  OTCLIB_ASSERT(myData != 0);

  myData->count = 1;
  myData->length = theLength;
  myData->capacity = myData->length+1;
  myData->locked = OTCLIB_FALSE;
  myData->symbol = OTCLIB_FALSE;

  if (theType == OTCLIB_BUFFERED)
  {
    myData->capacity = capacity(myData->length)+1;
    OTCLIB_ASSERT(myData->capacity >= myData->length+1);
  }

#if defined(ENV_OSTORE_DML)
  myData->data = new ((os_segment*)theLocality) char[myData->capacity];
#else
#if defined(ENV_OSTORE)
  myData->data = new (theLocality,os_typespec::get_char(),
   myData->capacity) char[myData->capacity];
#else
  myData->data = new char[myData->capacity];
#endif
#endif
  OTCLIB_ASSERT(myData->data != 0);

  myData->data[myData->length] = EOS;
}

/* ------------------------------------------------------------------------- */
OTC_RString::OTC_RString(OTC_RString const& theString)
  : myData(0)
{
  *this = theString;
}

/* ------------------------------------------------------------------------- */
OTC_RString::OTC_RString(OTC_StringData* theData)
  : myData(theData)
{
  OTCLIB_ENSURE((theData != 0),
   "OTC_RString(OTC_StringData*) - invalid data");

  // Following lock is to cover case where this constructor was called at
  // the same time in two or more threads with the same data as argument.
  // This can only happen from within <OTC_Symbol>.

  _sdmutex.lock();

  theData->count++;

  _sdmutex.unlock();
}

/* ------------------------------------------------------------------------- */
char const* OTC_RString::string() const
{
  return myData->data;
}

/* ------------------------------------------------------------------------- */
char* OTC_RString::string()
{
  return myData->data;
}

/* ------------------------------------------------------------------------- */
void OTC_RString::reallocate(u_int theLength)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

  // If memory is shared, always create new memory area. Only need to
  // hold lock while operating on shared data.

  OTC_Boolean theShared;

  _sdmutex.lock();

  theShared = (myData->count != 1);

  if (theShared != OTCLIB_FALSE)
  {
    OTC_StringData* tmpData;
#if defined(ENV_OSTORE_DML)
    tmpData = new ((os_segment*)theLocality) OTC_StringData;
#else
#if defined(ENV_OSTORE)
    tmpData = new (theLocality,OTC_StringData::get_os_typespec())
     OTC_StringData;
#else
    tmpData = new OTC_StringData;
#endif
#endif
    OTCLIB_ASSERT(tmpData != 0);

    tmpData->count = 1;
    tmpData->length = theLength;
    tmpData->locked = OTCLIB_FALSE;
    tmpData->symbol = OTCLIB_FALSE;

    tmpData->capacity = capacity(tmpData->length)+1;
    OTCLIB_ASSERT(tmpData->capacity >= tmpData->length+1);

#if defined(ENV_OSTORE_DML)
    tmpData->data = new ((os_segment*)theLocality) char[tmpData->capacity];
#else
#if defined(ENV_OSTORE)
    tmpData->data = new (theLocality,os_typespec::get_char(),
     tmpData->capacity) char[tmpData->capacity];
#else
    tmpData->data = new char[tmpData->capacity];
#endif
#endif
    OTCLIB_ASSERT(tmpData->data != 0);

    tmpData->data[tmpData->length] = EOS;

    if (tmpData->length >= myData->length)
      OTC_CopyActions<char>::copy(tmpData->data,myData->data,myData->length);
    else
      OTC_CopyActions<char>::copy(tmpData->data,myData->data,tmpData->length);

    myData->count--;
    myData = tmpData;
  }

  _sdmutex.unlock();

  if (theShared != OTCLIB_FALSE)
    return;

  // If not shared and length has not changed, don't bother doing
  // anything.

  if (theLength == myData->length)
    return;

  // If length is reduced, add in a new null terminator and adjust
  // length.

  if (theLength <= myData->length)
  {
    myData->data[theLength] = EOS;
    myData->length = theLength;

    return;
  }

  // If length has increased, but have sufficient capacity, add in a new null
  // terminator and adjust length.

  if (theLength > myData->length && theLength < myData->capacity)
  {
    myData->data[theLength] = EOS;
    myData->length = theLength;

    return;
  }

  // If we get here, it means we have to allocate some new memory.

  myData->capacity = capacity(theLength)+1;
  OTCLIB_ASSERT(myData->capacity >= theLength+1);

  char* tmpString;
#if defined(ENV_OSTORE_DML)
  tmpString = new ((os_segment*)theLocality) char[myData->capacity];
#else
#if defined(ENV_OSTORE)
  tmpString = new (theLocality,os_typespec::get_char(),
   myData->capacity) char[myData->capacity];
#else
  tmpString = new char[myData->capacity];
#endif
#endif
  OTCLIB_ASSERT(tmpString != 0);
  tmpString[theLength] = EOS;

  OTC_CopyActions<char>::copy(tmpString,myData->data,myData->length);

  delete [] myData->data;
  myData->data = tmpString;

  myData->length = theLength;
}

/* ------------------------------------------------------------------------- */
void OTC_RString::lock()
{
  OTCLIB_ENSURE((myData->locked == OTCLIB_FALSE),
   "OTC_RString::lock() - Memory is already locked");

  sync();

  myData->locked = OTCLIB_TRUE;
}

/* ------------------------------------------------------------------------- */
void OTC_RString::unlock()
{
  OTCLIB_ENSURE((myData->locked != OTCLIB_FALSE),
   "OTC_RString::unlock() - Memory is not locked");

  myData->locked = OTCLIB_FALSE;
}

/* ------------------------------------------------------------------------- */
OTC_RString const& OTC_RString::operator=(OTC_RString const& theString)
{
  if (&theString == this)
    return *this;

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  // Make sure segment check is against <theString.myData> as ctor
  // accepting <void*> for locality means that data could be in different
  // class than what is holding it. If localities are different, we will
  // always have to make a copy of the string. Whether the source is locked
  // doesn't matter. Reuse the string data object for this class if we can.

  OTC_Locality theLocality = OTC_Locality::of(this);
  OTC_Locality theRHSLocality = OTC_Locality::of(theString.myData);

  if (theLocality != theRHSLocality)
  {
    _sdmutex.lock();

    if (myData != 0)
    {
      if (myData->count == 1)
      {
	// Reuse string data object.

	delete [] myData->data;
      }
      else
      {
	myData->count--;

#if defined(ENV_OSTORE_DML)
	myData = new ((os_segment*)theLocality) OTC_StringData;
#else
	myData = new (theLocality,OTC_StringData::get_os_typespec())
	 OTC_StringData;
#endif
	OTCLIB_ASSERT(myData != 0);
      }
    }
    else
    {
#if defined(ENV_OSTORE_DML)
      myData = new ((os_segment*)theLocality) OTC_StringData;
#else
      myData = new (theLocality,OTC_StringData::get_os_typespec())
       OTC_StringData;
#endif
      OTCLIB_ASSERT(myData != 0);
    }

    // Fix up string data object and copy source string across.

    myData->count = 1;
    myData->length = theString.myData->length;
    myData->locked = OTCLIB_FALSE;
    myData->symbol = OTCLIB_FALSE;

    myData->capacity = capacity(myData->length)+1;
    OTCLIB_ASSERT(myData->capacity >= myData->length+1);

#if defined(ENV_OSTORE_DML)
    myData->data = new ((os_segment*)theLocality) char[myData->capacity];
#else
    myData->data = new (theLocality,os_typespec::get_char(),
     myData->capacity) char[myData->capacity];
#endif
    OTCLIB_ASSERT(myData->data != 0);

    OTC_CopyActions<char>::copy(myData->data,theString.myData->data,
     myData->length);

    myData->data[myData->length] = EOS;

    _sdmutex.unlock();

    return *this;
  }
#endif

  // If source string is locked, we cannot point at it so must make a copy
  // of it. As a result the existing string data object is kept around
  // until we know that we cannot use it.

  _sdmutex.lock();

  if (myData != 0)
  {
    if (myData->count == 1)
    {
      // Reuse string data object.

      delete [] myData->data;
    }
    else
    {
      myData->count--;
      myData = 0;
    }
  }

  if (theString.myData->locked == OTCLIB_FALSE)
  {
    // No need to reuse string data object after all.

    if (myData != 0)
      delete myData;

    theString.myData->count++;
    myData = theString.myData;
  }
  else
  {
    // No string data object to reuse.

    if (myData == 0)
    {
#if defined(ENV_OSTORE_DML)
      myData = new ((os_segment*)theLocality) OTC_StringData;
#else
#if defined(ENV_OSTORE)
      myData = new (theLocality,OTC_StringData::get_os_typespec())
       OTC_StringData;
#else
      myData = new OTC_StringData;
#endif
#endif
      OTCLIB_ASSERT(myData != 0);
    }

    // Fix up string data object and copy source string across.

    myData->count = 1;
    myData->length = theString.myData->length;
    myData->locked = OTCLIB_FALSE;
    myData->symbol = OTCLIB_FALSE;

    myData->capacity = capacity(myData->length)+1;
    OTCLIB_ASSERT(myData->capacity >= myData->length+1);

#if defined(ENV_OSTORE_DML)
    myData->data = new ((os_segment*)theLocality) char[myData->capacity];
#else
#if defined(ENV_OSTORE)
    myData->data = new (theLocality,os_typespec::get_char(),
     myData->capacity) char[myData->capacity];
#else
    myData->data = new char[myData->capacity];
#endif
#endif
    OTCLIB_ASSERT(myData->data != 0);

    OTC_CopyActions<char>::copy(theString.myData->data,myData->data,
     myData->length);

    myData->data[myData->length] = EOS;
  }

  _sdmutex.unlock();

  return *this;
}

/* ------------------------------------------------------------------------- */
u_int OTC_RString::capacity(u_int theCapacity) const
{
  _mutex.lock();

  if (myInitialised == OTCLIB_FALSE)
  {
    myInitialised = OTCLIB_TRUE;
    myBufType = OTCLIB_16BUFFERING;

    char const* env = 0;

    env = getenv("OTCLIB_NOSTRINGBUFFERING");
    if (env != 0)
    {
      myBufType = OTCLIB_NOBUFFERING;
    }
    else
    {
      env = getenv("OTCLIB_MOWBRAYBUFFERING");
      if (env != 0)
	myBufType = OTCLIB_MMBUFFERING;
    }
  }

  _mutex.unlock();

  switch (myBufType)
  {
    case OTCLIB_NOBUFFERING:
    {
      break;
    }

    case OTCLIB_MMBUFFERING:
    {
      if (theCapacity >= 64)
      {
	if (theCapacity >= 256)
	{
	  if (theCapacity >= 1024)
	  {
	    if (theCapacity >= 4096)
	      theCapacity += (theCapacity >> 2);
	    else
	      theCapacity += (theCapacity >> 1);
	  }
	  else
	    theCapacity = 1024;
	}
	else
	  theCapacity = 256;
      }
      else
      {
	if (theCapacity >= 16)
	  theCapacity = 64;
	else
	  theCapacity = 16;
      }

      break;
    }

    default:
    {
      if ((theCapacity % 16) != 0)
	theCapacity = (theCapacity/16 + 1) * 16;

      break;
    }
  }

  return theCapacity;
}

/* ------------------------------------------------------------------------- */
ostream& operator<<(ostream& outs, OTC_RString const& theString)
{
  // Special streams prefix operations.

  if (!outs.opfx())
    return outs;

  // If number of characters in string greater than width defined in
  // stream then just write it out. If not we must do some formatting.

  u_int theWidth = outs.width();
  u_int theLength = theString.length();

  if (theLength >= theWidth)
  {
    outs.write(theString.string(),theLength);
  }
  else
  {
    if (!(outs.flags() & ios::left))
    {
      for (int i=0; i<int(theWidth-theLength); i++)
        outs.put((char)outs.fill());
    }
    outs.write(theString.string(),theLength);
    if (outs.flags() & ios::left)
    {
      for (int i=0; i<int(theWidth-theLength); i++)
        outs.put((char)outs.fill());
    }
  }

  // Reset the stream width setting.

  outs.width(0);

  // Sepcial streams suffix operations.

  outs.osfx();

  return outs;
}

/* ------------------------------------------------------------------------- */
