/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     text/otcstring.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1993 1994 TELSTRA CORPORATION LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/text/string.hh"
#pragma implementation "OTC/text/tstring.hh"
#pragma implementation "OTC/text/sobject.hh"
#endif

#include <OTC/text/string.hh>
#include <OTC/collctn/copyactn.hh>

#include <ctype.h>

#include <string.h>

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#ifdef NEED_MEMCHR
extern "C" void* memchr(void const*, int, int);
#endif
#ifdef NEED_MEMSET
extern "C" void* memset(void*, int, int);
#endif

/* ------------------------------------------------------------------------- */
OTC_SObject::OTC_SObject()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_SObject::~OTC_SObject()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_NRMutex OTC_String::_mutex;
OTC_String const* OTC_String::globNullString = 0;
OTC_String const* OTC_String::globUndefinedString = 0;

const char OTC_String::globCharMap[] =
{
  '\000', '\001', '\002', '\003', '\004', '\005', '\006', '\007',
  '\010', '\011', '\012', '\013', '\014', '\015', '\016', '\017',
  '\020', '\021', '\022', '\023', '\024', '\025', '\026', '\027',
  '\030', '\031', '\032', '\033', '\034', '\035', '\036', '\037',
  '\040', '\041', '\042', '\043', '\044', '\045', '\046', '\047',
  '\050', '\051', '\052', '\053', '\054', '\055', '\056', '\057',
  '\060', '\061', '\062', '\063', '\064', '\065', '\066', '\067',
  '\070', '\071', '\072', '\073', '\074', '\075', '\076', '\077',
  '\100', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
  '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
  '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
  '\170', '\171', '\172', '\133', '\134', '\135', '\136', '\137',
  '\140', '\141', '\142', '\143', '\144', '\145', '\146', '\147',
  '\150', '\151', '\152', '\153', '\154', '\155', '\156', '\157',
  '\160', '\161', '\162', '\163', '\164', '\165', '\166', '\167',
  '\170', '\171', '\172', '\173', '\174', '\175', '\176', '\177',
  '\200', '\201', '\202', '\203', '\204', '\205', '\206', '\207',
  '\210', '\211', '\212', '\213', '\214', '\215', '\216', '\217',
  '\220', '\221', '\222', '\223', '\224', '\225', '\226', '\227',
  '\230', '\231', '\232', '\233', '\234', '\235', '\236', '\237',
  '\240', '\241', '\242', '\243', '\244', '\245', '\246', '\247',
  '\250', '\251', '\252', '\253', '\254', '\255', '\256', '\257',
  '\260', '\261', '\262', '\263', '\264', '\265', '\266', '\267',
  '\270', '\271', '\272', '\273', '\274', '\275', '\276', '\277',
  '\300', '\301', '\302', '\303', '\304', '\305', '\306', '\307',
  '\310', '\311', '\312', '\313', '\314', '\315', '\316', '\317',
  '\320', '\321', '\322', '\323', '\324', '\325', '\326', '\327',
  '\330', '\331', '\332', '\333', '\334', '\335', '\336', '\337',
  '\340', '\341', '\342', '\343', '\344', '\345', '\346', '\347',
  '\350', '\351', '\352', '\353', '\354', '\355', '\356', '\357',
  '\360', '\361', '\362', '\363', '\364', '\365', '\366', '\367',
  '\370', '\371', '\372', '\373', '\374', '\375', '\376', '\377'
};

#if defined(ENV_OSTORE)
/* ------------------------------------------------------------------------- */
os_typespec* OTC_String::typespec()
{
  static os_typespec ts("OTC_String");
  return &ts;
}
#endif

/* ------------------------------------------------------------------------- */
OTC_String::~OTC_String()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String()
  : myData(undefinedString().myData)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(OTC_Capacity theCapacity)
  : myData(theCapacity.capacity())
{
  myData.reallocate(0);
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(OTC_Length theLength)
  : myData(theLength.length())
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(OTC_Length theLength, OTC_Capacity theCapacity)
  : myData(theCapacity.capacity())
{
  myData.reallocate(theLength.length());
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(u_int theCapacity)
  : myData(theCapacity)
{
  myData.reallocate(0);
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(char const* theString)
  : myData((theString ? ::strlen(theString) : 0))
{
  if (theString != 0)
    OTC_CopyActions<char>::copy(myData.string(),theString,myData.length());
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(char const* theString, u_int theNum)
  : myData(theNum)
{
  OTCLIB_ENSURE((theString != 0 || theNum == 0),
   "OTC_String(char const*, u_int) - Invalid character count");

  if (theString != 0)
    OTC_CopyActions<char>::copy(myData.string(),theString,myData.length());
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(char theChar, u_int theNum)
  : myData(theNum)
{
  memset(myData.string(),theChar,myData.length());
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(OTC_String const& theString)
  : myData(theString.myData)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(OTC_String const& theString, u_int theNum)
  : myData(theNum)
{
  OTCLIB_ENSURE((theNum <= theString.length()),
   "OTC_String(OTC_String const&, u_int) - Invalid character count");

  OTC_CopyActions<char>::copy(myData.string(),theString.string(),
   myData.length());
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(OTC_SObject const& theString)
  : myData(theString.rawString())
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(OTC_CString const& theString)
  : myData(theString.myData)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(OTC_RString const& theString)
  : myData(theString)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_String::OTC_String(OTC_TString const& theString)
  : myData(theString.myData)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
istream& operator>>(istream& ins, OTC_String& theString)
{
  OTCLIB_ENSURE((theString.myData.isLocked() == OTCLIB_FALSE),
   "operator>>(istream&, OTC_String&) - String is locked");

  // Go directly to raw string to make things quick.

  OTC_RString& theData = theString.myData;

  // Set initial string size. Characters will be read directly into the
  // string, growing it further if necessary, and the length readjusted
  // when finished. Use initial size of <BLKSIZE-1> instead of <BLKSIZE> as
  // raw string adds <1> to this to accomodate a null terminator. <BLKSIZE>
  // must be greater than <1>.

  int const BLKSIZE = 16;

  u_int theCapacity = BLKSIZE-1;
  theData.reallocate(theCapacity);

  // Special streams prefix operations. Set string size to zero thus
  // creating an empty string if this fails. Note that <ipfx()> skips
  // leading white space if stream is configured to skip white space.

  if (!ins.ipfx(0))
  {
    theData.reallocate(0);
    return ins;
  }

  // Set the length. This will be incremented as each character is read
  // and will be used to readjust the string length at the end. This
  // will also be used to determine if we have exceeded the capacity of
  // the string and thus it needs to be resized, and whether we have
  // exceeded the width request programmed into the stream.

  u_int theLength = 0;

  // If non zero, the maximum number of characters to read in.

  u_int theWidth = ins.width();

  // Read the characters in.

  int c;
  c = ins.get();

  if (c != EOF && !isspace(c))
  {
    theData[theLength++] = c;
    c = ins.get();

    while (c != EOF && !isspace(c) && (theWidth == 0 || theLength < theWidth))
    {
      if (theLength == theCapacity)
      {
	theCapacity += BLKSIZE;
        theData.reallocate(theCapacity);
      }
      theData[theLength++] = c;
      c = ins.get();
    }
  }

  // Set the length of the string to what it should be.

  theData.reallocate(theLength);

  // Put back the last character into the stream.

  ins.putback(c);

  // Reset the stream width setting.

  ins.width(0);

  //  Special streams suffix operations.

  ins.isfx();

  return ins;
}

/* ------------------------------------------------------------------------- */
ostream& operator<<(ostream& outs, OTC_String const& theString)
{
  return outs << theString.myData;
}

/* ------------------------------------------------------------------------- */
OTC_String OTC_String::get(istream& ins, char theDelim)
{
  OTC_String s;
  return get(s,ins,theDelim);
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::get(
 OTC_String& theString,
 istream& ins,
 char theDelim
)
{
  OTCLIB_ENSURE((theString.myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::get() - String is locked");

  // Characters will be read directly into the string, growing it further if
  // necessary, and the length readjusted when finished.

  // <BLKSIZE> is the amount we increase the capacity of the string.

  int const BLKSIZE = 64;

  OTC_RString& theData = theString.myData;

  // The accumulated length of the string, the capacity, and amount of
  // available space within that capacity.

  u_int theLength = theString.myData.length();
  u_int theCapacity = theString.myData.capacity();
  u_int theSpace = theCapacity - theLength;

  // If current capacity of string is filled up, or string is shared,
  // increase the capacity of the string, otherwise, adjust the length
  // of the string so as to use available space.

  if (theSpace == 0 || theString.myData.isShared())
  {
    theData.reallocate(theLength+BLKSIZE);
    theCapacity = theString.myData.capacity();
    theSpace = theCapacity - theLength;
  }
  else
  {
    theData.reallocate(theLength+theSpace);
  }

  // Flag for when we have finished.

  OTC_Boolean theEnd = OTCLIB_FALSE;

  // Read in the characters. Note that <istream::get()> will write a
  // null character over the raw string null terminator, but this is okay.

  while (theEnd == OTCLIB_FALSE)
  {
    char* tmpString = theData.string() + theLength;
    ins.get(tmpString,theSpace,theDelim);
    theLength += ins.gcount();

    if ((int)ins.gcount() != (int)theSpace-1)
    {
      theEnd = OTCLIB_TRUE;
    }
    else
    {
      theData.reallocate(theLength+BLKSIZE);
      theCapacity = theString.myData.capacity();
      theSpace = theCapacity - theLength;
    }
  }

  // Readjust the length of the string.

  theData.reallocate(theLength);

  return theString;
}

/* ------------------------------------------------------------------------- */
OTC_String OTC_String::getline(istream& ins, char theDelim)
{
  OTC_String s;
  return getline(s,ins,theDelim);
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::getline(
 OTC_String& theString,
 istream& ins,
 char theDelim
)
{
  OTCLIB_ENSURE((theString.myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::getline() - String is locked");

  // Characters will be read directly into the string, growing it further if
  // necessary, and the length readjusted when finished.

  // <BLKSIZE> is the amount we increase the capacity of the string.

  int const BLKSIZE = 64;

  OTC_RString& theData = theString.myData;

  // The accumulated length of the string, the capacity, and amount of
  // available space within that capacity.

  u_int theLength = theString.myData.length();
  u_int theCapacity = theString.myData.capacity();
  u_int theSpace = theCapacity - theLength;

  // If current capacity of string is filled up, or string is shared,
  // increase the capacity of the string, otherwise, adjust the length
  // of the string so as to use available space.

  if (theSpace == 0 || theString.myData.isShared())
  {
    theData.reallocate(theLength+BLKSIZE);
    theCapacity = theString.myData.capacity();
    theSpace = theCapacity - theLength;
  }
  else
  {
    theData.reallocate(theLength+theSpace);
  }

  // Flag for when we have finished.

  OTC_Boolean theEnd = OTCLIB_FALSE;

  // Read in the characters. Note that <istream::get()> will write a
  // null character over the raw string null terminator, but this is okay.
  // Also, we use <istream::get()> instead of <istream::getline()> so
  // we we do not have to adjust <istream::gcount()> when we get the
  // to <EOF> and do not find the delimiter.

  while (theEnd == OTCLIB_FALSE)
  {
    char* tmpString = theData.string() + theLength;
    ins.get(tmpString,theSpace,theDelim);
    theLength += ins.gcount();

    if ((int)ins.gcount() != (int)theSpace-1)
    {
      theEnd = OTCLIB_TRUE;
    }
    else
    {
      theData.reallocate(theLength+BLKSIZE);
      theCapacity = theString.myData.capacity();
      theSpace = theCapacity - theLength;
    }
  }

  // Readjust the length of the string.

  theData.reallocate(theLength);

  // Discard the delimiter.

  if (!ins.eof())
    ins.ignore(1);

  return theString;
}

/* ------------------------------------------------------------------------- */
OTC_String OTC_String::read(istream& ins, u_int theNum)
{
  OTC_String s;
  return read(s,ins,theNum);
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::read(OTC_String& theString, istream& ins, u_int theNum)
{
  if (theNum > 0)
  {
    OTC_RString& theData = theString.myData;

    u_int oldLength = theData.length();

    theData.reallocate(oldLength+theNum);

    ins.read(theData.string()+oldLength,theNum);

    theData.reallocate(oldLength+ins.gcount());
  }

  return theString;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::readFile(istream& theStream)
{
  length(0);
  OTC_String::getline(*this,theStream,EOF);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::readLine(istream& theStream, OTC_Boolean theSkipWhite)
{
  if (theSkipWhite != OTCLIB_FALSE)
    theStream >> ws;

  length(0);
  OTC_String::getline(*this,theStream,EOL);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::readString(istream& theStream)
{
  length(0);
  OTC_String::getline(*this,theStream,EOS);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::readToDelim(istream& theStream, char theDelim)
{
  length(0);
  OTC_String::getline(*this,theStream,theDelim);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::readToken(istream& theStream)
{
  length(0);
  theStream >> ws;
  theStream >> *this;

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::readData(istream& theStream, u_int theNum)
{
  length(0);
  OTC_String::read(*this,theStream,theNum);

  return *this;
}

/* ------------------------------------------------------------------------- */
char const* OTC_String::string() const
{
  return myData.string();
}

/* ------------------------------------------------------------------------- */
OTC_String::operator char const*() const
{
  return myData.string();
}

/* ------------------------------------------------------------------------- */
char const* OTC_String::data() const
{
  return myData.string();
}

/* ------------------------------------------------------------------------- */
char const* OTC_String::buffer() const
{
  return (myData.length() == 0) ? 0 : myData.string();
}

/* ------------------------------------------------------------------------- */
char* OTC_String::buffer()
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::buffer() - String is locked");

  if (myData.length() == 0)
    return 0;

  if (myData.isShared())
    myData.sync();

  return myData.string();
}

/* ------------------------------------------------------------------------- */
char OTC_String::operator[](u_int theIndex) const
{
  OTCLIB_ENSURE((theIndex < myData.length()),
   "OTC_String::operator[](u_int) const - Index out of range");

  return myData[theIndex];
}

/* ------------------------------------------------------------------------- */
char& OTC_String::operator[](u_int theIndex)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::operator[](u_int) - String is locked");

  OTCLIB_ENSURE((theIndex < myData.length()),
   "OTC_String::operator[](u_int) - Index out of range");

  if (myData.isShared())
    myData.sync();

  return myData[theIndex];
}

/* ------------------------------------------------------------------------- */
void OTC_String::length(u_int theLength)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::length(u_int) - String is locked");

  myData.reallocate(theLength);
}

/* ------------------------------------------------------------------------- */
void OTC_String::resize(u_int theLength)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::resize(u_int) - String is locked");

  myData.reallocate(theLength);
}

/* ------------------------------------------------------------------------- */
void OTC_String::capacity(u_int theCapacity)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::capacity(u_int) - String is locked");

  u_int theLength = myData.length();

  if (myData.isShared())
  {
    if (theCapacity > theLength)
    {
      myData.reallocate(theCapacity);
      myData.reallocate(theLength);
    }
    else
    {
      myData.reallocate(theLength);
    }
  }
  else
  {
    if (theCapacity > myData.capacity())
    {
      myData.reallocate(theCapacity);
      myData.reallocate(theLength);
    }
  }
}

/* ------------------------------------------------------------------------- */
char* OTC_String::duplicate() const
{
  char* tmpString;
  tmpString = new char[myData.length()+1];
  OTCLIB_ASSERT(tmpString != 0);

  OTC_CopyActions<char>::copy(tmpString,myData.string(),myData.length()+1);
  return tmpString;
}

/* ------------------------------------------------------------------------- */
u_int OTC_String::length() const
{
  return myData.length();
}

/* ------------------------------------------------------------------------- */
u_int OTC_String::size() const
{
  return myData.length();
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_String::isEmpty() const
{
  return myData.length() == 0;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_String::isNull() const
{
  return myData.length() == 0;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_String::isUndefined() const
{
  return myData.string() == undefinedString().myData.string();
}

/* ------------------------------------------------------------------------- */
u_int OTC_String::capacity() const
{
  return myData.capacity();
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::assign(OTC_String const& theString)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::assign(OTC_String const&) - String is locked");

  myData = theString.myData;
  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::assign(OTC_String const& theString, u_int theNum)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::assign(OTC_String const&, u_int) - String is locked");

  OTCLIB_ENSURE((theNum <= theString.myData.length()),
   "OTC_String::assign(OTC_String const&, u_int) - Invalid length");

  _replace(0,myData.length(),theString.string(),theNum);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_TString operator+(OTC_String const& s1, OTC_String const& s2)
{
  OTC_RString tmpString(s1.myData.length()+s2.myData.length());
  char* tmpBuffer = tmpString.string();
  OTC_CopyActions<char>::copy(tmpBuffer,s1.string(),s1.myData.length());
  OTC_CopyActions<char>::copy(tmpBuffer+s1.myData.length(),s2.string(),
   s2.myData.length());
  return OTC_TString(tmpString);
}

/* ------------------------------------------------------------------------- */
OTC_TString operator+(OTC_String const& s1, char const* s2)
{
  u_int theLength = (s2 ? ::strlen(s2) : 0);
  OTC_RString tmpString(s1.myData.length()+theLength);
  char* tmpBuffer = tmpString.string();
  OTC_CopyActions<char>::copy(tmpBuffer,s1.string(),s1.myData.length());
  OTC_CopyActions<char>::copy(tmpBuffer+s1.myData.length(),s2,theLength);
  return OTC_TString(tmpString);
}

/* ------------------------------------------------------------------------- */
OTC_TString operator+(char const* s1, OTC_String const& s2)
{
  u_int theLength = (s1 ? ::strlen(s1) : 0);
  OTC_RString tmpString(theLength+s2.myData.length());
  char* tmpBuffer = tmpString.string();
  OTC_CopyActions<char>::copy(tmpBuffer,s1,theLength);
  OTC_CopyActions<char>::copy(tmpBuffer+theLength,s2.string(),
   s2.myData.length());
  return OTC_TString(tmpString);
}

/* ------------------------------------------------------------------------- */
OTC_TString operator+(OTC_String const& s1, OTC_SObject const& s2)
{
  OTC_String s2String = s2;
  OTC_RString tmpString(s1.myData.length()+s2String.myData.length());
  char* tmpBuffer = tmpString.string();
  OTC_CopyActions<char>::copy(tmpBuffer,s1.string(),s1.myData.length());
  OTC_CopyActions<char>::copy(tmpBuffer+s1.myData.length(),s2String.string(),
   s2String.myData.length());
  return OTC_TString(tmpString);
}

/* ------------------------------------------------------------------------- */
OTC_TString operator+(OTC_SObject const& s1, OTC_String const& s2)
{
  OTC_String s1String = s1;
  OTC_RString tmpString(s1String.myData.length()+s2.myData.length());
  char* tmpBuffer = tmpString.string();
  OTC_CopyActions<char>::copy(tmpBuffer,s1String.string(),
   s1String.myData.length());
  OTC_CopyActions<char>::copy(tmpBuffer+s1String.myData.length(),
   s2.string(),s2.myData.length());
  return OTC_TString(tmpString);
}

/* ------------------------------------------------------------------------- */
OTC_TString operator+(OTC_SObject const& s1, char const* s2)
{
  u_int theLength = (s2 ? ::strlen(s2) : 0);
  OTC_String s1String = s1;
  OTC_RString tmpString(s1String.myData.length()+theLength);
  char* tmpBuffer = tmpString.string();
  OTC_CopyActions<char>::copy(tmpBuffer,s1String.string(),
   s1String.myData.length());
  OTC_CopyActions<char>::copy(tmpBuffer+s1String.myData.length(),
   s2,theLength);
  return OTC_TString(tmpString);
}

/* ------------------------------------------------------------------------- */
OTC_TString operator+(OTC_String const& s1, char c1)
{
  OTC_RString tmpString(s1.myData.length()+1);
  char* tmpBuffer = tmpString.string();
  OTC_CopyActions<char>::copy(tmpBuffer,s1.string(),s1.myData.length());
  tmpBuffer[s1.myData.length()] = c1;
  return OTC_TString(tmpString);
}

/* ------------------------------------------------------------------------- */
OTC_TString operator+(char c1, OTC_String const& s1)
{
  OTC_RString tmpString(1+s1.myData.length());
  char* tmpBuffer = tmpString.string();
  tmpBuffer[0] = c1;
  OTC_CopyActions<char>::copy(tmpBuffer+1,s1.string(),s1.myData.length());
  return OTC_TString(tmpString);
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::truncate(u_int theIndex)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::truncate(u_int) - String is locked");

  if (myData.length() == 0 && theIndex == 0)
    return *this;

  int theNum = myData.length()-theIndex;

  if (theNum == 0)
    return *this;

  OTCLIB_ENSURE((theNum > 0),
   "OTC_String::truncate(u_int) - Index out of range");

  _replace(theIndex,theNum,EOS,0);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::rtrim()
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::rtrim() - String is locked");

  if (myData.length() == 0)
    return *this;

  u_int theIndex = myData.length();

  while (theIndex > 0 && isspace(myData[theIndex-1]))
    theIndex--;

  if (theIndex != myData.length())
    truncate(theIndex);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::rtrim(char theChar)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::rtrim(char) - String is locked");

  if (myData.length() == 0)
    return *this;

  u_int theIndex = myData.length();

  while (theIndex > 0 && myData[theIndex-1] == theChar)
    theIndex--;

  if (theIndex != myData.length())
    truncate(theIndex);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::ltrim()
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::ltrim() - String is locked");

  if (myData.length() == 0)
    return *this;

  u_int theIndex = 0;
  u_int theLength = myData.length();

  while (theIndex < theLength && isspace(myData[theIndex]))
    theIndex++;

  if (theIndex != 0)
    remove(0,theIndex);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::ltrim(char theChar)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::ltrim(char) - String is locked");

  if (myData.length() == 0)
    return *this;

  u_int theIndex = 0;
  u_int theLength = myData.length();

  while (theIndex < theLength && myData[theIndex] == theChar)
    theIndex++;

  if (theIndex != 0)
    remove(0,theIndex);

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::ljustify(u_int theWidth, char theChar)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::ljustify(u_int, char) - String is locked");

  if (myData.length() >= theWidth)
    return *this;

  _replace(myData.length(),0,theChar,theWidth-myData.length());

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::rjustify(u_int theWidth, char theChar)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::rjustify(u_int, char) - String is locked");

  if (myData.length() >= theWidth)
    return *this;

  _replace(0,0,theChar,theWidth-myData.length());

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_String& OTC_String::reverse()
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::reverse() - String is locked");

  if (myData.length() <= 1)
    return *this;

  u_int theLength = myData.length();

  if (myData.isShared())
  {
    OTC_RString tmpData(theLength,this);
    for (u_int i=theLength; i>0; i--)
      tmpData[theLength-i] = myData[i-1];

    myData = tmpData;
  }
  else
  {
    for (u_int i=(theLength/2); i>0; i--)
    {
      char c = myData[theLength-i];
      myData[theLength-i] = myData[i-1];
      myData[i-1] = c;
    }
  }

  return *this;
}

/* ------------------------------------------------------------------------- */
int OTC_String::_index(u_int theIndex, char theChar, u_int theNum) const
{
  OTCLIB_ENSURE((theNum > 0),
   "OTC_String::_index(u_int, char, u_int) const - Bad occurrence number");

  if (myData.length() == 0)
    return -1;

  OTCLIB_ENSURE((theIndex < myData.length()),
   "OTC_String::_index(u_int, char, u_int) const - Index out of range");

  char const* tmpString = myData.string()+theIndex;
  tmpString = (char const*)memchr(tmpString,theChar,myData.length()-theIndex);

  if (tmpString != 0)
  {
    int anIndex = tmpString - myData.string();
    if (theNum == 1)
    {
      return anIndex;
    }
    else
    {
      if (anIndex+1 == (int)myData.length())
        return -1;

      else
        return _index(anIndex+1,theChar,theNum-1);
    }
  }
  else
    return -1;
}

/* ------------------------------------------------------------------------- */
int OTC_String::_rindex(char theChar, u_int theNum) const
{
  OTCLIB_ENSURE((theNum > 0),
   "OTC_String::_rindex(char, u_int) const - Bad occurrence number");

  if (myData.length() == 0)
    return -1;

  int theIndex = myData.length() - 1;
  while (theIndex >= 0)
  {
    if (myData[theIndex] == theChar)
    {
      theNum--;
      if (theNum == 0)
        return theIndex;
    }
    theIndex--;
  }

  return -1;
}

/* ------------------------------------------------------------------------- */
int OTC_String::_index(
 u_int theIndex,
 OTC_BitSetC const& theCharSet,
 u_int theNum
) const
{
  OTCLIB_ENSURE((theNum > 0),
   "OTC_String::_index(u_int, OTC_BitSetC const&, u_int) const - "
   "Bad occurrence number");

  if (myData.length() == 0)
    return -1;

  OTCLIB_ENSURE((theIndex < myData.length()),
   "OTC_String::_index(u_int, OTC_BitSetC const&, u_int) const - "
   "Index out of range");

  char const* tmpString = myData.string() + theIndex;
  u_int theLength = myData.length() - theIndex;

  while (theLength != 0)
  {
    if (theCharSet.test(*tmpString) != 0)
      break;

    tmpString++;
    theLength--;
  }

  if (theLength != 0)
  {
    int anIndex = tmpString - myData.string();
    if (theNum == 1)
    {
      return anIndex;
    }
    else
    {
      if (anIndex+1 == (int)myData.length())
        return -1;

      else
        return _index(anIndex+1,theCharSet,theNum-1);
    }
  }
  else
    return -1;
}

/* ------------------------------------------------------------------------- */
int OTC_String::_rindex(OTC_BitSetC const& theCharSet, u_int theNum) const
{
  OTCLIB_ENSURE((theNum > 0),
   "OTC_String::_rindex(OTC_BitSetC const&, u_int) const - "
   "Bad occurrence number");

  if (myData.length() == 0)
    return -1;

  int theIndex = myData.length() - 1;
  while (theIndex >= 0)
  {
    if (theCharSet.test(myData[theIndex]) != 0)
    {
      theNum--;
      if (theNum == 0)
        return theIndex;
    }
    theIndex--;
  }

  return -1;
}

/* ------------------------------------------------------------------------- */
int OTC_String::rank(OTC_String const& theString) const
{
  return rank(myData.string(),myData.length(),theString.myData.string(),
   theString.myData.length());
}

/* ------------------------------------------------------------------------- */
int OTC_String::rank(OTC_String const& theString, u_int theLength) const
{
  OTCLIB_ENSURE((theLength <= theString.myData.length()),
   "OTC_String::rank(OTC_String const&, u_int) - Invalid length");

  return rank(myData.string(),myData.length(),theString.myData.string(),
   theLength);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_String::compare(
 u_int theIndex,
 char const* theString,
 OTC_CmpType theType
) const
{
  OTCLIB_ENSURE((theIndex <= myData.length()),
   "OTC_String::compare(u_int, char const*, OTC_CmpType) const - "
   "Index out of range");

  return rank(myData.string()+theIndex,myData.length()-theIndex,
   theString,(theString ? ::strlen(theString) : 0),theType) == 0;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_String::compare(
 u_int theIndex,
 char const* theString,
 u_int theNum,
 OTC_CmpType theType
) const
{
  OTCLIB_ENSURE((theIndex <= myData.length()),
   "OTC_String::compare(u_int, char const*, u_int, OTC_CmpType) - "
   "Index out of range");

  if (myData.length()-theIndex < theNum)
    return OTCLIB_FALSE;

  return rank(myData.string()+theIndex,theNum,theString,theNum,theType) == 0;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_String::compare(
 u_int theIndex,
 OTC_String const& theString,
 OTC_CmpType theType
) const
{
  OTCLIB_ENSURE((theIndex <= myData.length()),
   "OTC_String::compare(u_int, OTC_String const&, OTC_CmpType) const - "
   "Index out of range");

  return rank(myData.string()+theIndex,myData.length()-theIndex,
   theString.string(),theString.length(),theType) == 0;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_String::compare(
 u_int theIndex,
 OTC_String const& theString,
 u_int theNum,
 OTC_CmpType theType
) const
{
  OTCLIB_ENSURE((theIndex <= myData.length()),
   "OTC_String::compare(u_int, OTC_String const&, u_int, OTC_CmpType) const - "
   "Index out of range");

  OTCLIB_ENSURE((theNum <= theString.length()),
   "OTC_String::compare(u_int, OTC_String const&, u_int, OTC_CmpType) const - "
   "Invalid length");

  if (myData.length()-theIndex < theNum)
    return OTCLIB_FALSE;

  return rank(myData.string()+theIndex,theNum,
   theString.string(),theNum,theType) == 0;
}

/* ------------------------------------------------------------------------- */
OTC_String const& OTC_String::nullString()
{
  _mutex.lock();

  if (globNullString == 0)
  {
    OTC_CString tmpString;
    globNullString = new OTC_String(tmpString);
    OTCLIB_ASSERT(globNullString != 0);
  }

  _mutex.unlock();

  return *globNullString;
}

/* ------------------------------------------------------------------------- */
OTC_String const& OTC_String::undefinedString()
{
  _mutex.lock();

  if (globUndefinedString == 0)
  {
    OTC_CString tmpString;
    globUndefinedString = new OTC_String(tmpString);
    OTCLIB_ASSERT(globUndefinedString != 0);
  }

  _mutex.unlock();

  return *globUndefinedString;
}

/* ------------------------------------------------------------------------- */
int OTC_String::rank(
 char const* theString1,
 u_int theLength1,
 char const* theString2,
 u_int theLength2,
 OTC_CmpType theType
)
{
  // If both are zero length, they are equal.

  if (!theLength1 && !theLength2)
    return 0;

  // Work out which is shorter.

  u_int theLength = theLength1;
  if (theLength2 < theLength)
    theLength = theLength2;

  // If one is zero length, it is the least ranked.

  if (!theLength)
    return (theLength2 > 0) ? -1 : 1;

  // Check for zero pointers. If length arguments were correct,
  // there shouldn't be any.

  OTCLIB_ENSURE((theString1 != 0 && theString2 != 0),
   "OTC_String::rank(char const*, u_int, char const*, u_int, OTC_CmpType) - "
   "Invalid pointer");

  if (theType == OTCLIB_EXACTMATCH)
  {
    int theRank = memcmp(theString1,theString2,theLength);

    if (theRank != 0)
      return theRank;

//    while (theLength-- != 0)
//    {
//      if (*theString1++ != *theString2++)
//	  return (*--theString1 - *--theString2);
//    }
  }
  else
  {
    while (theLength-- != 0)
    {
      if (globCharMap[*theString1++] != globCharMap[*theString2++])
	return (globCharMap[*--theString1] - globCharMap[*--theString2]);
    }
  }

  return (theLength1 == theLength2) ? 0 : ((theLength1 < theLength2) ? -1 : 1);
}

/* ------------------------------------------------------------------------- */
void OTC_String::_replace(
 int theStart,
 u_int theLength,
 char theChar,
 u_int theNum
)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::_replace(int, u_int, char, u_int) - String is locked");

  OTCLIB_ENSURE((theStart >= 0 && theStart+theLength <= myData.length()),
   "OTC_String::_replace(int, u_int, char, u_int) - Invalid range");

  if (myData.isShared())
  {
    OTC_RString tmpData(myData.length()+theNum-theLength,this);
    OTC_CopyActions<char>::copy(tmpData.string(),myData.string(),theStart);
    memset(tmpData.string()+theStart,theChar,theNum);
    OTC_CopyActions<char>::copy(tmpData.string()+theStart+theNum,
     myData.string()+theStart+theLength,myData.length()-(theStart+theLength));
    myData = tmpData;
  }
  else
  {
    u_int oldLength = myData.length();
    if (theNum <= theLength)
    {
      memset(myData.string()+theStart,theChar,theNum);
      OTC_CopyActions<char>::copy(myData.string()+theStart+theNum,
       myData.string()+theStart+theLength,oldLength-(theStart+theLength));
      myData.reallocate(myData.length()+theNum-theLength);
    }
    else
    {
      myData.reallocate(myData.length()+theNum-theLength);
      OTC_CopyActions<char>::copy(myData.string()+theStart+theNum,
       myData.string()+theStart+theLength,oldLength-(theStart+theLength));
      memset(myData.string()+theStart,theChar,theNum);
    }
  }
}

/* ------------------------------------------------------------------------- */
void OTC_String::_replace(
 int theStart,
 u_int theLength,
 char const* theString,
 u_int theNum 
)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::_replace(int, u_int, char const*, u_int) - String is locked");

  OTCLIB_ENSURE((theStart >= 0 && theStart+theLength <= myData.length()),
   "OTC_String::_replace(int, u_int, char const*, u_int) - Invalid range");
  OTCLIB_ENSURE((theNum == 0 || theString != 0),
   "OTC_String::_replace(int, u_int, char const*, u_int) - "
   "Null string but non zero count");

  if (myData.isShared() || (theString >= myData.string() &&
   theString < myData.string()+myData.length()))
  {
    OTC_RString tmpData(myData.length()+theNum-theLength,this);
    OTC_CopyActions<char>::copy(tmpData.string(),myData.string(),theStart);
    OTC_CopyActions<char>::copy(tmpData.string()+theStart,theString,theNum);
    OTC_CopyActions<char>::copy(tmpData.string()+theStart+theNum,
     myData.string()+theStart+theLength,myData.length()-(theStart+theLength));
    myData = tmpData;
  }
  else
  {
    u_int oldLength = myData.length();
    if (theNum <= theLength)
    {
      OTC_CopyActions<char>::copy(myData.string()+theStart,theString,theNum);
      OTC_CopyActions<char>::copy(myData.string()+theStart+theNum,
       myData.string()+theStart+theLength,oldLength-(theStart+theLength));
      myData.reallocate(myData.length()+theNum-theLength);
    }
    else
    {
      myData.reallocate(myData.length()+theNum-theLength);
      OTC_CopyActions<char>::copy(myData.string()+theStart+theNum,
       myData.string()+theStart+theLength,oldLength-(theStart+theLength));
      OTC_CopyActions<char>::copy(myData.string()+theStart,theString,theNum);
    }
  }
}

/* ------------------------------------------------------------------------- */
void OTC_String::_replace(
 int theStart,
 u_int theLength,
 OTC_String const& theString,
 u_int theNum 
)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::_replace(int, u_int, OTC_String const&, u_int) - "
   "String is locked");

  OTCLIB_ENSURE((theNum <= theString.myData.length()),
   "OTC_String::_replace(int, u_int, OTC_String const&, u_int) - "
   "Invalid range");

  _replace(theStart,theLength,theString.myData.string(),theNum);
}

/* ------------------------------------------------------------------------- */
OTC_String OTC_String::_section(int theStart, u_int theLength) const
{
  OTCLIB_ENSURE((theStart >= 0 && theStart+theLength <= myData.length()),
   "OTC_String::_section(int, u_int) const - Invalid range");

  if (myData.length() == 0 && theLength == 0)
    return nullString();

  else
  {
    OTC_String tmpString(myData.string()+theStart,theLength);
    return tmpString;
  }
}

/* ------------------------------------------------------------------------- */
OTC_String OTC_String::except_(int theStart, u_int theLength) const
{
  OTCLIB_ENSURE((theStart >= 0 && theStart+theLength <= myData.length()),
   "OTC_String::except_(int, u_int) const - Invalid range");

  if (myData.length() == 0)
    return nullString();

  else if (theLength == 0)
    return *this;

  else
  {
    OTC_RString tmpString(myData.length()-theLength);
    char* tmpBuffer1 = tmpString.string();
    char const* tmpBuffer2 = myData.string();
    OTC_CopyActions<char>::copy(tmpBuffer1,tmpBuffer2,theStart);
    tmpBuffer1 += theStart;
    tmpBuffer2 += (theStart + theLength);
    OTC_CopyActions<char>::copy(tmpBuffer1,tmpBuffer2,
     myData.length()-(theStart+theLength)
    );
    return tmpString;
  }
}

/* ------------------------------------------------------------------------- */
void OTC_String::_upper(int theStart, u_int theLength)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::_upper(int, u_int) - String is locked");

  OTCLIB_ENSURE((theStart >= 0 && theStart+theLength <= myData.length()),
   "OTC_String::_upper(int, u_int) - Invalid range");

  if (theLength == 0)
    return;

  if (myData.isShared())
    myData.sync();

  for (u_int i=0; i<theLength; i++)
  {
    if (islower(myData[theStart+i]))
      myData[theStart+i] = toupper(myData[theStart+i]);
  }
}

/* ------------------------------------------------------------------------- */
void OTC_String::_lower(int theStart, u_int theLength)
{
  OTCLIB_ENSURE((myData.isLocked() == OTCLIB_FALSE),
   "OTC_String::_lower(int, u_int) - String is locked");

  OTCLIB_ENSURE((theStart >= 0 && theStart+theLength <= myData.length()),
   "OTC_String::_lower(int, u_int) - Invalid range");

  if (theLength == 0)
    return;

  if (myData.isShared())
    myData.sync();

  for (u_int i=0; i<theLength; i++)
  {
    if (isupper(myData[theStart+i]))
      myData[theStart+i] = tolower(myData[theStart+i]);
  }
}

/* ------------------------------------------------------------------------- */
int OTC_String::_index(
 u_int theIndex,
 char const* theString,
 u_int theLength
) const
{
  if (myData.length() == 0 || theLength == 0)
    return -1;

  OTCLIB_ENSURE((theIndex < myData.length()),
   "OTC_String::_index(u_int, char const*, u_int) const - "
   "Index out of range");

  u_int theLength1 = myData.length() - theIndex;

  if (theLength > theLength1)
    return -1;

  OTC_Boolean theEnd = OTCLIB_FALSE;
  char const* theStart = myData.string() + theIndex;

  char const* a;
  char const* b;

  b = theString;

  while (theEnd == OTCLIB_FALSE)
  {
    if (*theStart == *b)
    {
      a = theStart;

      u_int theLength2 = theLength;

      while (theLength2 != 0)
      {
	if (*a++ != *b++)
	  break;

	theLength2--;
      }

      if (theLength2 == 0)
	theEnd = OTCLIB_TRUE;

      b = theString;
    }

    if (theEnd == OTCLIB_FALSE)
    {
      theLength1--;

      // XXX
      //
      // if (theLength1 == 0)

      if (theLength > theLength1)
      {
	theStart = 0;
	theEnd = OTCLIB_TRUE;
      }
      else
	theStart++;
    }
  }

  if (theStart != 0)
  {
    u_int anIndex = theStart - myData.string();
    return anIndex;
  }

  return -1;
}

/* ------------------------------------------------------------------------- */
int OTC_String::_rindex(
 char const* theString,
 u_int theLength
) const
{
  if (myData.length() == 0 || theLength == 0)
    return -1;

  if (theLength > myData.length())
    return -1;

  OTC_Boolean theEnd = OTCLIB_FALSE;
  int theIndex = myData.length() - theLength;
  char const* theStart = myData.string() + theIndex;

  char const* a;
  char const* b;

  b = theString;

  while (theEnd == OTCLIB_FALSE)
  {
    if (*theStart == *b)
    {
      a = theStart;

      u_int theLength2 = theLength;

      while (theLength2 != 0)
      {
	if (*a++ != *b++)
	  break;

	theLength2--;
      }

      if (theLength2 == 0)
	theEnd = OTCLIB_TRUE;

      b = theString;
    }

    if (theEnd == OTCLIB_FALSE)
    {
      theIndex--;

      if (theIndex < 0)
      {
	theStart = 0;
	theEnd = OTCLIB_TRUE;
      }
      else
	theStart--;
    }
  }

  if (theStart != 0)
  {
    u_int anIndex = theStart - myData.string();
    return anIndex;
  }

  return -1;
}

#if defined(CXX_OS)
/* ------------------------------------------------------------------------- */
int OTC_RankActions<OTC_String>::rank(
 OTC_String const& s1,
 OTC_String const& s2
)
{
  return s1.rank(s2);
}

/* ------------------------------------------------------------------------- */
int OTC_HashActions<OTC_String>::hash(OTC_String const& s)
{
  return s.hash();
}
#endif

/* ------------------------------------------------------------------------- */
