/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     text/otcsymbol.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 TELSTRA CORPORATION LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/text/symbol.hh"
#endif

#include <OTC/text/symbol.hh>
#include <OTC/collctn/avltree.hh>
#include <OTC/memory/arena.hh>

#include <string.h>

/* ------------------------------------------------------------------------- */
OTC_SymbolData::~OTC_SymbolData()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_SymbolData::OTC_SymbolData(char* theString, u_int theLength)
{
  OTC_StringData::capacity = theLength+1;
  OTC_StringData::length = theLength;
  OTC_StringData::count = 1;
  OTC_StringData::locked = OTCLIB_FALSE;
  OTC_StringData::symbol = OTCLIB_TRUE;
  OTC_StringData::data = theString;
}

/* ------------------------------------------------------------------------- */
OTC_NRMutex OTC_Symbol::_mutex;
OTC_Arena* OTC_Symbol::globArena = 0;
OTC_AVLTree* OTC_Symbol::globTree = 0;
OTC_SymbolData* OTC_Symbol::globNullInfo = 0;
OTC_Symbol* OTC_Symbol::globNullSymbol = 0;

/* ------------------------------------------------------------------------- */
OTC_Symbol::~OTC_Symbol()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_Symbol::OTC_Symbol()
{
  myInfo = install(0,0);
}

/* ------------------------------------------------------------------------- */
OTC_Symbol::OTC_Symbol(char const* theString)
{
  myInfo = install(theString,(theString ? ::strlen(theString) : 0));
}

/* ------------------------------------------------------------------------- */
OTC_Symbol::OTC_Symbol(char const* theString, u_int theLength)
{
  myInfo = install(theString,theLength);
}

/* ------------------------------------------------------------------------- */
OTC_Symbol::OTC_Symbol(char theChar, u_int theNum)
{
  // This is inefficient, however, don't have much choice given
  // the way things have been done. At least use exact length
  // string.

  OTC_String theString = OTC_CString(theChar,theNum);

  myInfo = install(theString.string(),theString.length());
}

/* ------------------------------------------------------------------------- */
OTC_Symbol::OTC_Symbol(OTC_String const& theString)
{
  if (theString.rawString().isSymbol())
    myInfo = (OTC_SymbolData*)theString.rawString().data();
  else
    myInfo = install(theString.string(),theString.length());
}

/* ------------------------------------------------------------------------- */
OTC_Symbol::OTC_Symbol(OTC_String const& theString, u_int theLength)
{
  OTCLIB_ENSURE((theLength <= theString.length()),
   "OTC_Symbol(OTC_String const&, u_int) - Invalid range");

  if (theString.rawString().isSymbol() && theString.length() == theLength)
    myInfo = (OTC_SymbolData*)theString.rawString().data();
  else
    myInfo = install(theString.string(),theLength);
}

/* ------------------------------------------------------------------------- */
OTC_Symbol::OTC_Symbol(OTC_TString const& theString)
{
  myInfo = install(theString.myData.string(),theString.myData.length());
}

/* ------------------------------------------------------------------------- */
char OTC_Symbol::operator[](u_int theIndex) const
{
  OTCLIB_ENSURE((theIndex < myInfo->length()),
   "OTC_Symbol::operator[](u_int) const - Index out of range");

  return myInfo->string()[theIndex];
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Symbol::exists(char const* theString, u_int theLength)
{
  OTCLIB_ENSURE((theString != 0 || theLength == 0),
   "OTC_Symbol::exists(char const*, u_int) - "
   "Null pointer but non zero count");

  if (theString == 0 || theLength == 0)
    return OTCLIB_TRUE;

  OTC_Boolean theResult;
  theResult = OTCLIB_FALSE;

  _mutex.lock();

  if (globTree != 0)
  {
    OTC_SymbolData* theData = 0;
    OTC_AVLNode* theNode = globTree->root();
    OTC_AVLNode* theParent = 0;
    int theRank = 0;

    if (theNode != 0)
    {
      while (theNode != 0 && theResult == OTCLIB_FALSE)
      {
	theParent = theNode;

	theData = (OTC_SymbolData*)theNode;
	theRank = OTC_String::rank(theString,theLength,
	 theData->string(),theData->length());

	theParent = theNode;
	if (theRank < 0)
	  theNode = theNode->left();
	else if (theRank > 0)
	  theNode = theNode->right();
	else
	  theResult = OTCLIB_TRUE;
      }
    }
  }

  _mutex.unlock();

  return theResult;
}

/* ------------------------------------------------------------------------- */
OTC_Symbol const& OTC_Symbol::nullSymbol()
{
  _mutex.lock();

  if (globNullSymbol == 0)
  {
    globNullSymbol = new OTC_Symbol((char*)0,0);
    OTCLIB_ASSERT(globNullSymbol != 0);
  }

  _mutex.unlock();

  return *globNullSymbol;
}

/* ------------------------------------------------------------------------- */
OTC_RString OTC_Symbol::rawString() const
{
  return OTC_RString(myInfo->data());
}

/* ------------------------------------------------------------------------- */
OTC_SymbolData* OTC_Symbol::install(char const* theString, u_int theLength)
{
  OTCLIB_ENSURE((theString != 0 || theLength == 0),
   "OTC_Symbol::install(char const*, u_int) - "
   "Null pointer but non zero count");

  OTC_SymbolData* theData = 0; 

  _mutex.lock();

  if (globTree == 0)
  {
    globArena = new OTC_Arena(OTC_Alignment::ofInt());
    OTCLIB_ASSERT(globArena);

    globTree = new OTC_AVLTree;
    OTCLIB_ASSERT(globTree);

    globNullInfo = new OTC_SymbolData("",0);
    OTCLIB_ASSERT(globNullInfo != 0);

    globTree->addRoot(globNullInfo);
  }

  if (theString == 0 || theLength == 0)
    theData = globNullInfo;

  if (theData == 0)
  {
    OTC_SymbolData* tmpData = 0;
    OTC_AVLNode* theNode = globTree->root();
    OTC_AVLNode* theParent = 0;
    int theRank = 0;

    if (theNode != 0)
    {
      while (theNode != 0 && theData == 0)
      {
	theParent = theNode;

	tmpData = (OTC_SymbolData*)theNode;
	theRank = OTC_String::rank(theString,theLength,
	 tmpData->string(),tmpData->length());

	theParent = theNode;
	if (theRank < 0)
	  theNode = theNode->left();
	else if (theRank > 0)
	  theNode = theNode->right();
	else
	  theData = tmpData;
      }
    }

    if (theData == 0)
    {
      char* tmpString = (char*)globArena->allocate(theLength+1);
      OTC_CopyActions<char>::copy(tmpString,theString,theLength);
      tmpString[theLength] = EOS;

      theData = new OTC_SymbolData(tmpString,theLength);
      OTCLIB_ASSERT(theData != 0);

      if (theParent == 0)
	globTree->addRoot(theData);
      else if (theRank < 0)
	theParent->addBefore(theData);
      else
	theParent->addAfter(theData);
    }
  }

  _mutex.unlock();

  return theData;
}

#if defined(CXX_OS)
/* ------------------------------------------------------------------------- */
int OTC_RankActions<OTC_Symbol>::rank(
 OTC_Symbol const& s1,
 OTC_Symbol const& s2
)
{
  return s1.rank(s2);
}

/* ------------------------------------------------------------------------- */
int OTC_HashActions<OTC_Symbol>::hash(OTC_Symbol const& s)
{
  return s.hash();
}
#endif

/* ------------------------------------------------------------------------- */
ostream& operator<<(ostream& outs, OTC_Symbol const& theSymbol)
{
  OTC_String aString = theSymbol;
  outs << aString;
  return outs;
}

/* ------------------------------------------------------------------------- */
