/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     thread/otcmutex.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/thread/mutex.hh"
#endif

#include <OTC/thread/mutex.hh>

#if defined(HAVE_SOLARIS_THREADS)
/* ------------------------------------------------------------------------- */
OTC_Mutex::OTC_Mutex()
  : _nest(0), _thread(0)
{
  mutex_init(&_mutex,USYNC_THREAD,0);
  _valid = 1;
}

/* ------------------------------------------------------------------------- */
OTC_Mutex::~OTC_Mutex()
{
  _valid = 0;
  mutex_destroy(&_mutex);
}

/* ------------------------------------------------------------------------- */
void OTC_Mutex::lock()
{
  if (_valid == 0)
    return;

  thread_t theThread;

  theThread = thr_self();

  if (theThread == _thread)
  {
    _nest++;
  }
  else
  {
    if (mutex_lock(&_mutex) == 0)
    {
      _thread = theThread;
      _nest = 0;
    }
  }
}

/* ------------------------------------------------------------------------- */
void OTC_Mutex::unlock()
{
  if (_valid == 0)
    return;

  if (_nest > 0)
  {
    _nest--;
  }
  else
  {
    _thread = 0;
    mutex_unlock(&_mutex);
  }
}
#endif

#if defined(HAVE_POSIX_THREADS)
/* ------------------------------------------------------------------------- */
OTC_Mutex::OTC_Mutex()
{
  pthread_mutexattr_create(&_attr);
  pthread_mutexattr_setkind_np(&_attr,MUTEX_RECURSIVE_NP);
  pthread_mutex_init(&_mutex,_attr);
  _valid = 1;
}

/* ------------------------------------------------------------------------- */
OTC_Mutex::~OTC_Mutex()
{
  _valid = 0;
  pthread_mutex_destroy(&_mutex);
  pthread_mutexattr_delete(&_attr);
}

/* ------------------------------------------------------------------------- */
void OTC_Mutex::lock()
{
  if (_valid == 0)
    return;

  pthread_mutex_lock(&_mutex);
}

/* ------------------------------------------------------------------------- */
void OTC_Mutex::unlock()
{
  if (_valid == 0)
    return;

  pthread_mutex_unlock(&_mutex);
}
#endif

#if defined(HAVE_VXWORKS_THREADS)
/* ------------------------------------------------------------------------- */
OTC_Mutex::OTC_Mutex()
  : _nest(0), _thread(0)
{
  _mutex = semMCreate(SEM_Q_PRIORITY|SEM_DELETE_SAFE);
  _valid = 1;
}

/* ------------------------------------------------------------------------- */
OTC_Mutex::~OTC_Mutex()
{
  _valid = 0;
  semDelete(_mutex);
}

/* ------------------------------------------------------------------------- */
void OTC_Mutex::lock()
{
  if (_valid == 0)
    return;

  int theThread;

  theThread = taskIdSelf();

  if (theThread == _thread)
  {
    _nest++;
  }
  else
  {
    if (semTake(_mutex,WAIT_FOREVER) == 0)
    {
      _thread = theThread;
      _nest = 0;
    }
  }
}

/* ------------------------------------------------------------------------- */
void OTC_Mutex::unlock()
{
  if (_valid == 0)
    return;

  if (_nest > 0)
  {
    _nest--;
  }
  else
  {
    _thread = 0;
    semGive(_mutex);
  }
}
#endif

#if defined(HAVE_WIN32_THREADS)
/* ------------------------------------------------------------------------- */
OTC_Mutex::OTC_Mutex()
{
  InitializeCriticalSection(&_mutex);
  _valid = 1;
}

/* ------------------------------------------------------------------------- */
OTC_Mutex::~OTC_Mutex()
{
  _valid = 0;
  DeleteCriticalSection(&_mutex);
}

/* ------------------------------------------------------------------------- */
void OTC_Mutex::lock()
{
  if (_valid == 0)
    return;

  EnterCriticalSection(&_mutex);
}

/* ------------------------------------------------------------------------- */
void OTC_Mutex::unlock()
{
  if (_valid == 0)
    return;

  LeaveCriticalSection(&_mutex);
}
#endif

/* ------------------------------------------------------------------------- */
