/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     types/date.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/types/date.hh"
#endif

#include <OTC/types/date.hh>

#include <time.h>
#include <iostream.h>
#include <strstream.h>
#include <string.h>

#if defined(__GNUG__) && defined(HAVE_RTTI)
#define exception math_exception
#endif

#include <math.h>

#if defined(__GNUG__) && defined(HAVE_RTTI)
#undef exception
#endif

#if defined(ENV_OSTORE)
/* ------------------------------------------------------------------------- */
os_typespec* OTC_Date::typespec()
{
  static os_typespec ts("OTC_Date");
  return &ts;
}
#endif

/* ------------------------------------------------------------------------- */
OTC_NRMutex OTC_Date::_mutex;

/* ------------------------------------------------------------------------- */
OTC_Date::OTC_Date()
{
  setDate();
}

/* ------------------------------------------------------------------------- */
OTC_Date::OTC_Date(int theDay, int theMonth, int theYear)
{
  myDate = julianDate(theDay,theMonth,theYear);
}

/* ------------------------------------------------------------------------- */
OTC_Date::OTC_Date(OTC_String const& theDateString)
{
  int theDay = 0;
  int theMonth = 0;
  int theYear = 0;

  convertToDate(theDateString,theDay,theMonth,theYear);

  OTCLIB_ENSURE((theDay && theMonth && theYear),
   "OTC_Date::OTC_Date(OTC_String const&) - invalid format for date");

  myDate = julianDate(theDay,theMonth,theYear);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Date::isValidDate(int theDay, int theMonth, int theYear)
{
  if (theYear == 0)
    return OTCLIB_FALSE;

  int tmpDay;
  int tmpMonth;
  int tmpYear;

  long tmpDate = julianDate(theDay,theMonth,theYear);
  calendarDate(tmpDate,tmpDay,tmpMonth,tmpYear);

  return (theDay == tmpDay) && (theMonth == tmpMonth) && (theYear == tmpYear);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Date::isValidDate(OTC_String const& theDateString)
{
  int theDay = 0;
  int theMonth = 0;
  int theYear = 0;

  convertToDate(theDateString,theDay,theMonth,theYear);

  if (!theDay && !theMonth && !theYear)
    return OTCLIB_FALSE;

  return isValidDate(theDay,theMonth,theYear);
}

/* ------------------------------------------------------------------------- */
void OTC_Date::convertToDate(
 OTC_String const& theDateString,
 int& theDay,
 int& theMonth,
 int& theYear
)
{
  // Take a local copy as we want to cast away constness on the
  // internal buffer when using istrstream. The copy, even
  // though a delayed copy is sufficient. What we are trying to
  // protect against is access to the buffer from another thread.
  // What will happen is that since there is another reference to
  // the string internals, the other thread will make a copy of the
  // string if it wants to modify it.

  OTC_String tmpDateString = theDateString;

  istrstream ins((char*)tmpDateString.string(),tmpDateString.length());
  ins.unsetf(ios::skipws);

  theYear = 0;
  theMonth = 0;
  theDay = 0;

  char theSeparator1 = 0;
  char theSeparator2 = 0;

  ins >> dec >> theYear;
  ins >> theSeparator1;
  ins >> dec >> theMonth;
  ins >> theSeparator2;
  ins >> dec >> theDay;

  if (ins.fail() || theSeparator1 != '-' || theSeparator2 != '-')
  {
    theYear = theMonth = theDay = 0;
    return;
  }
}

/* ------------------------------------------------------------------------- */
int OTC_Date::day() const
{
  int theDay, theMonth, theYear;
  OTC_Date::calendarDate(myDate,theDay,theMonth,theYear);
  return theDay;
}

/* ------------------------------------------------------------------------- */
int OTC_Date::dayOfYear() const
{
  int tmpDay;
  int tmpMonth;
  int tmpYear;

  calendarDate(myDate,tmpDay,tmpMonth,tmpYear);
  OTC_Date tmpDate(1,1,tmpYear);
  return tmpDate.daysTill(*this) + 1;
}

/* ------------------------------------------------------------------------- */
int OTC_Date::month() const
{
  int theDay, theMonth, theYear;
  OTC_Date::calendarDate(myDate,theDay,theMonth,theYear);
  return theMonth;
}

/* ------------------------------------------------------------------------- */
int OTC_Date::year() const
{
  int theDay, theMonth, theYear;
  OTC_Date::calendarDate(myDate,theDay,theMonth,theYear);
  return theYear;
}

/* ------------------------------------------------------------------------- */
int OTC_Date::daysInYear() const
{
  if (isLeapYear())
    return 366;

  else
  {
    int tmpDay;
    int tmpMonth;
    int tmpYear;

    calendarDate(myDate,tmpDay,tmpMonth,tmpYear);

    if (tmpYear == 1582)
      return 355;

    else
      return 365;
  }
}

/* ------------------------------------------------------------------------- */
int OTC_Date::weekOfYear() const
{
  OTC_Date aDate(firstDayOfYear());
  aDate.subtractDays(1);
  aDate = aDate.following(OTCLIB_SUNDAY);

  if (*this < aDate)
    return 53;

  else
    return (dayOfYear() - aDate.dayOfYear()) / 7 + 1;
}

/* ------------------------------------------------------------------------- */
OTC_String const& OTC_Date::nameOfDayOfWeek() const
{
  // Use local statics, as we cannot rely on initialisation of
  // global statics in a shared library on certain platforms.
  // Also, do not assign OTC_CString directly in static declaration
  // as it triggers a bug in cfront compilers.

  static int gInit = 0;
  static OTC_String gSUN;
  static OTC_String gMON;
  static OTC_String gTUE;
  static OTC_String gWED;
  static OTC_String gTHU;
  static OTC_String gFRI;
  static OTC_String gSAT;

  _mutex.lock();

  if (gInit == 0)
  {
    gSUN = OTC_CString("Sunday");
    gMON = OTC_CString("Monday");
    gTUE = OTC_CString("Tuesday");
    gWED = OTC_CString("Wednesday");
    gTHU = OTC_CString("Thursday");
    gFRI = OTC_CString("Friday");
    gSAT = OTC_CString("Saturday");
    gInit = 1;
  }

  _mutex.unlock();

  switch (dayOfWeek())
  {
    case OTCLIB_SUNDAY:
      return gSUN;

    case OTCLIB_MONDAY:
      return gMON;

    case OTCLIB_TUESDAY:
      return gTUE;

    case OTCLIB_WEDNESDAY:
      return gWED;

    case OTCLIB_THURSDAY:
      return gTHU;

    case OTCLIB_FRIDAY:
      return gFRI;

    case OTCLIB_SATURDAY:
      return gSAT;
  }

  return OTC_String::undefinedString();
}

/* ------------------------------------------------------------------------- */
OTC_String const& OTC_Date::nameOfMonth() const
{
  // Use local statics, as we cannot rely on initialisation of 
  // global statics in a shared library on certain platforms.
  // Also, do not assign OTC_CString directly in static declaration
  // as it triggers a bug in cfront compilers.

  static int gInit = 0;
  static OTC_String gJAN;
  static OTC_String gFEB;
  static OTC_String gMAR;
  static OTC_String gAPR;
  static OTC_String gMAY;
  static OTC_String gJUN;
  static OTC_String gJUL;
  static OTC_String gAUG;
  static OTC_String gSEP;
  static OTC_String gOCT;
  static OTC_String gNOV;
  static OTC_String gDEC;

  _mutex.lock();

  if (gInit == 0)
  {
    gJAN = OTC_CString("January");
    gFEB = OTC_CString("February");
    gMAR = OTC_CString("March");
    gAPR = OTC_CString("April");
    gMAY = OTC_CString("May");
    gJUN = OTC_CString("June");
    gJUL = OTC_CString("July");
    gAUG = OTC_CString("August");
    gSEP = OTC_CString("September");
    gOCT = OTC_CString("October");
    gNOV = OTC_CString("November");
    gDEC = OTC_CString("December");
    gInit = 1;
  }

  _mutex.unlock();

  switch(monthOfYear())
  {
    case OTCLIB_JANUARY:
      return gJAN;

    case OTCLIB_FEBRUARY:
      return gFEB;

    case OTCLIB_MARCH:
      return gMAR;

    case OTCLIB_APRIL:
      return gAPR;

    case OTCLIB_MAY:
      return gMAY;

    case OTCLIB_JUNE:
      return gJUN;

    case OTCLIB_JULY:
      return gJUL;

    case OTCLIB_AUGUST:
      return gAUG;

    case OTCLIB_SEPTEMBER:
      return gSEP;

    case OTCLIB_OCTOBER:
      return gOCT;

    case OTCLIB_NOVEMBER:
      return gNOV;

    case OTCLIB_DECEMBER:
      return gDEC;
  }

  return OTC_String::undefinedString();
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Date::isBetween(
 OTC_Date const& theDate1,
 OTC_Date const& theDate2
) const
{
  if (theDate1.myDate < theDate2.myDate)
    return (myDate >= theDate1.myDate) && (myDate <= theDate2.myDate);

  return (myDate >= theDate2.myDate) && (myDate <= theDate1.myDate);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Date::isLeapYear() const
{
  int tmpDay;
  int tmpMonth;
  int tmpYear;

  calendarDate(myDate,tmpDay,tmpMonth,tmpYear);
  tmpDay = 29;
  tmpMonth = 2;
  return isValidDate(tmpDay,tmpMonth,tmpYear);
}

/* ------------------------------------------------------------------------- */
OTC_Date OTC_Date::following(int theDay) const
{
  OTCLIB_ASSERT(theDay >= 0 && theDay < 7);

  int theNum = theDay - dayOfWeek();
  if (theNum >= 0)
    return plusDays(theNum);

  else
    return plusDays(7+theNum);
}

/* ------------------------------------------------------------------------- */
OTC_Date OTC_Date::previous(int theDay) const
{
  OTCLIB_ASSERT(theDay >= 0 && theDay < 7);

  int theNum = dayOfWeek() - theDay;
  if (theNum >= 0)
    return plusDays(theNum);

  else
    return plusDays(7+theNum);
}

/* ------------------------------------------------------------------------- */
OTC_Date OTC_Date::firstDayOfWeek() const
{
  OTC_Date aDate(myDate+1);
  return aDate.previous(OTCLIB_SUNDAY);
}

/* ------------------------------------------------------------------------- */
OTC_Date OTC_Date::lastDayOfWeek() const
{
  OTC_Date aDate(myDate-1);
  return aDate.following(OTCLIB_SATURDAY);
}

/* ------------------------------------------------------------------------- */
OTC_Date OTC_Date::firstDayOfMonth() const
{
  int tmpDay;
  int tmpMonth;
  int tmpYear;

  calendarDate(myDate,tmpDay,tmpMonth,tmpYear);
  return OTC_Date(1,tmpMonth,tmpYear);
}

/* ------------------------------------------------------------------------- */
OTC_Date OTC_Date::lastDayOfMonth() const
{
  int tmpDay;
  int tmpMonth;
  int tmpYear;

  calendarDate(myDate,tmpDay,tmpMonth,tmpYear);

  OTC_Date tmpDate(1,tmpMonth+1,tmpYear);
  return tmpDate.minusDays(1);
}

/* ------------------------------------------------------------------------- */
OTC_String OTC_Date::asString() const
{
  OTC_String tmpString = OTC_Length(31);
  ostrstream theStream(tmpString.buffer(),tmpString.capacity());

  int tmpDay;
  int tmpMonth;
  int tmpYear;

  OTC_Date::calendarDate(myDate,tmpDay,tmpMonth,tmpYear);

  theStream << tmpYear << '-';
  theStream.width(2);
  theStream.fill('0');
  theStream << tmpMonth << '-';
  theStream.width(2);
  theStream.fill('0');
  theStream << tmpDay << ends;

  int theLength = strlen(tmpString.buffer());
  tmpString.length(theLength);

  return tmpString;
}

/* ------------------------------------------------------------------------- */
ostream& operator<<(ostream& theStream, OTC_Date const& theDate)
{
  int tmpDay;
  int tmpMonth;
  int tmpYear;

  OTC_Date::calendarDate(theDate.jday(),tmpDay,tmpMonth,tmpYear);

  theStream << tmpYear << '-';
  theStream.width(2);
  theStream.fill('0');
  theStream << tmpMonth << '-';
  theStream.width(2);
  theStream.fill('0');
  theStream << tmpDay;

  return theStream;
}

/* ------------------------------------------------------------------------- */
OTC_Date OTC_Date::firstDayOfYear() const
{
  int tmpDay;
  int tmpMonth;
  int tmpYear;

  calendarDate(myDate,tmpDay,tmpMonth,tmpYear);
  return OTC_Date(1,1,tmpYear);
}

/* ------------------------------------------------------------------------- */
OTC_Date OTC_Date::lastDayOfYear() const
{
  int tmpDay;
  int tmpMonth;
  int tmpYear;

  calendarDate(myDate,tmpDay,tmpMonth,tmpYear);

  OTC_Date tmpDate(1,1,tmpYear+1);
  return tmpDate.minusDays(1);
}

/* ------------------------------------------------------------------------- */
void OTC_Date::setDate()
{
  time_t theClock = time(0);
  tm* theDate = localtime(&theClock);
  myDate = julianDate(theDate->tm_mday,theDate->tm_mon+1,theDate->tm_year+1900);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Date::setDate(int theDay, int theMonth, int theYear)
{
  myDate = julianDate(theDay,theMonth,theYear);
  return isValidDate(theDay,theMonth,theYear);
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Date::operator-(OTC_Date const& theDate) const
{
  OTC_Duration theDuration(int(myDate - theDate.myDate));
  return theDuration;
}

/* ------------------------------------------------------------------------- */
/* jdn.c -- Julian Day Number computation
**
** public domain Julian Day Number functions
**
** Based on formulae originally posted by
**    Tom Van Flandern / Washington, DC / metares@well.sf.ca.us
**       in the UseNet newsgroup sci.astro.
**    Reposted 14 May 1991 in FidoNet C Echo conference by
**       Paul Schlyter (Stockholm)
** Minor corrections, added JDN to julian, and recast into C by
**    Raymond Gardner  Englewood, Colorado
**
** Synopsis:
**      long ymd_to_jdn(int year, int month, int day, int julian_flag)
**      void jdn_to_ymd(long jdn, int *year, int *month, int *day,
**                                                      int julian_flag)
**      year is negative if BC
**      if julian_flag is >  0, use Julian calendar
**      if julian_flag is == 0, use Gregorian calendar
**      if julian_flag is <  0, routines decide based on date
**
** These routines convert Gregorian and Julian calendar dates to and 
** from Julian Day Numbers.  Julian Day Numbers (JDN) are used by 
** astronomers as a date/time measure independent of calendars and 
** convenient for computing the elapsed time between dates.  The JDN 
** for any date/time is the number of days (including fractional 
** days) elapsed since noon, 1 Jan 4713 BC.  Julian Day Numbers were 
** originated by Joseph Scaliger in 1582 and named after his father 
** Julius, not after Julius Caesar.  They are not related to the 
** Julian calendar. 
** 
** For dates from 1 Jan 4713 BC thru 12 Dec Feb 32766 AD, ymd_to_jdn() 
** will give the JDN for noon on that date.  jdn_to_ymd() will compute 
** the year, month, and day from the JDN.  Years BC are given (and 
** returned) as negative numbers.  Note that there is no year 0 BC; 
** the day before 1 Jan 1 AD is 31 Dec 1 BC.  Note also that 1 BC, 
** 5 BC, etc. are leap years.
** 
** Pope Gregory XIII decreed that the Julian calendar would end on 
** 4 Oct 1582 AD and that the next day would be 15 Oct 1582 in the 
** Gregorian Calendar.  The only other change is that centesimal 
** years (years ending in 00) would no longer be leap years 
** unless divisible by 400.  Britain and its possessions and 
** colonies continued to use the Julian calendar up until 2 Sep 
** 1752, when the next day became 14 Sep 1752 in the Gregorian 
** Calendar.  These routines can be compiled to use either 
** convention.  By default, the British convention will be used.  
** Simply #define PAPAL to use Pope Gregory's convention. 
** 
** Each routine takes, as its last argument, a flag to indicate 
** whether to use the Julian or Gregorian calendar convention.  If 
** this flag is negative, the routines decide based on the date 
** itself, using the changeover date described in the preceding 
** paragraph.  If the flag is zero, Gregorian conventions will be used, 
** and if the flag is positive, Julian conventions will be used. 
*/

#define PAPAL 1

#ifdef PAPAL                    /* Pope Gregory XIII's decree */
#define LASTJULDATE 15821004L   /* last day to use Julian calendar */
#define LASTJULJDN  2299160L    /* jdn of same */
#else                           /* British-American usage */
#define LASTJULDATE 17520902L   /* last day to use Julian calendar */
#define LASTJULJDN  2361221L    /* jdn of same */
#endif


/* ------------------------------------------------------------------------- */
long OTC_Date::julianDate(int d, int m, int y)
// long ymd_to_jdn(int y, int m, int d, int julian)
{
	OTCLIB_ENSURE((y != 0),
	 "OTC_Date::julianDate() - invalid year zero");

	int julian = -1;

        long jdn;

        if (julian < 0)         /* set Julian flag if auto set */
                julian = (((y * 100L) + m) * 100 + d  <=  LASTJULDATE);

        if (y < 0)              /* adjust BC year */
                y++;

        if (julian)
                jdn = 367L * y - 7 * (y + 5001L + (m - 9) / 7) / 4
                + 275 * m / 9 + d + 1729777L;
        else
                jdn = (long)(d - 32076)
                + 1461L * (y + 4800L + (m - 14) / 12) / 4
                + 367 * (m - 2 - (m - 14) / 12 * 12) / 12
                - 3 * ((y + 4900L + (m - 14) / 12) / 100) / 4
                + 1;            /* correction by rdg */

        return jdn;
}

/* ------------------------------------------------------------------------- */
void OTC_Date::calendarDate(long jdn, int& dd, int& mm, int& yy)
// void jdn_to_ymd(long jdn, int *yy, int *mm, int *dd, int julian)
{
	int julian = -1;

        long x, z, m, d, y;
        long daysPer400Years = 146097L;
        long fudgedDaysPer4000Years = 1460970L + 31;

        if (julian < 0)                 /* set Julian flag if auto set */
                julian = (jdn <= LASTJULJDN);

        x = jdn + 68569L;
        if ( julian )
        {
                x += 38;
                daysPer400Years = 146100L;
                fudgedDaysPer4000Years = 1461000L + 1;
        }
        z = 4 * x / daysPer400Years;
        x = x - (daysPer400Years * z + 3) / 4;
        y = 4000 * (x + 1) / fudgedDaysPer4000Years;
        x = x - 1461 * y / 4 + 31;
        m = 80 * x / 2447;
        d = x - 2447 * m / 80;
        x = m / 11;
        m = m + 2 - 12 * x;
        y = 100 * (z - 49) + y + x;

        yy = (int)y;
        mm = (int)m;
        dd = (int)d;

        if (yy <= 0)                   /* adjust BC years */
                (yy)--;
}

#if defined(CXX_OS)
/* ------------------------------------------------------------------------- */
int OTC_RankActions<OTC_Date>::rank(
 OTC_Date const& theDate1,
 OTC_Date const& theDate2
)
{
  return theDate1.rank(theDate2);
}

/* ------------------------------------------------------------------------- */
int OTC_HashActions<OTC_Date>::hash(OTC_Date const& theDate)
{
  return theDate.hash();
}
#endif

/* ------------------------------------------------------------------------- */
