/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     types/otcduration.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OTC/types/duration.hh"
#endif

#include <OTC/types/duration.hh>

#include <iostream.h>
#include <strstream.h>
#include <string.h>

#if defined(ENV_OSTORE)
/* ------------------------------------------------------------------------- */
os_typespec* OTC_Duration::typespec()
{
  static os_typespec ts("OTC_Duration");
  return &ts;
}
#endif

/* ------------------------------------------------------------------------- */
OTC_Duration::OTC_Duration()
  : mySign(1),
    myDays(0),
    myHours(0),
    myMinutes(0),
    mySeconds(0)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_Duration::OTC_Duration(
 int theDays,
 int theHours,
 int theMinutes,
 int theSeconds
)
  : mySign(1),
    myDays(theDays),
    myHours(theHours),
    myMinutes(theMinutes),
    mySeconds(theSeconds)
{
  normalise();
}

/* ------------------------------------------------------------------------- */
OTC_Duration::OTC_Duration(OTC_Duration const& theDuration)
  : mySign(theDuration.mySign),
    myDays(theDuration.myDays),
    myHours(theDuration.myHours),
    myMinutes(theDuration.myMinutes),
    mySeconds(theDuration.mySeconds)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Duration::isZeroLength() const
{
  return !myDays && !myHours && !myMinutes && !mySeconds;
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Duration::plusDuration(OTC_Duration const& theDuration) const
{
  OTC_Duration tmpDuration(*this);
  tmpDuration.addDuration(theDuration);
  return tmpDuration;
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Duration::minusDuration(OTC_Duration const& theDuration) const
{
  OTC_Duration tmpDuration(*this);
  tmpDuration.subtractDuration(theDuration);
  return tmpDuration;
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Duration::plusDuration(
 int theDays,
 int theHours,
 int theMinutes,
 int theSeconds
) const
{
  OTC_Duration tmpDuration(*this);
  tmpDuration.addDuration(theDays,theHours,theMinutes,theSeconds);
  return tmpDuration;
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Duration::minusDuration(
 int theDays,
 int theHours,
 int theMinutes,
 int theSeconds
) const
{
  OTC_Duration tmpDuration(*this);
  tmpDuration.subtractDuration(theDays,theHours,theMinutes,theSeconds);
  return tmpDuration;
}

/* ------------------------------------------------------------------------- */
OTC_String OTC_Duration::asString() const
{
  OTC_String tmpString = OTC_Length(31);
  ostrstream theStream(tmpString.buffer(),tmpString.capacity());

  theStream << 'P';
  theStream << days() << "DT";
  theStream << hours() << 'H';
  theStream << minutes() << 'M';
  theStream << seconds() << 'S' << ends;

  int theLength = strlen(tmpString.buffer());
  tmpString.length(theLength);

  return tmpString;
}

/* ------------------------------------------------------------------------- */
ostream& operator<<(ostream& theStream, OTC_Duration const& theDuration)
{
  theStream << 'P';
  theStream << theDuration.days() << "DT";
  theStream << theDuration.hours() << 'H';
  theStream << theDuration.minutes() << 'M';
  theStream << theDuration.seconds() << 'S' << ends;

  return theStream;
}

/* ------------------------------------------------------------------------- */
void OTC_Duration::setDuration(OTC_Duration const& theDuration)
{
  if (this != &theDuration)
  {
    mySign = theDuration.mySign;
    myDays = theDuration.myDays;
    myHours = theDuration.myHours;
    myMinutes = theDuration.myMinutes;
    mySeconds = theDuration.mySeconds;
  }
}

/* ------------------------------------------------------------------------- */
void OTC_Duration::setDuration(
 int theDays,
 int theHours,
 int theMinutes,
 int theSeconds
)
{
  myDays = theDays;
  myHours = theHours;
  myMinutes = theMinutes;
  mySeconds = theSeconds;

  normalise();
}

/* ------------------------------------------------------------------------- */
OTC_Duration& OTC_Duration::addDuration(OTC_Duration const& theDuration)
{
  int theSign = theDuration.mySign;

  myDays = (mySign * myDays) + (theSign * theDuration.myDays);
  myHours = (mySign * myHours) + (theSign * theDuration.myHours);
  myMinutes = (mySign * myMinutes) + (theSign * theDuration.myMinutes);
  mySeconds = (mySign * mySeconds) + (theSign * theDuration.mySeconds);

  normalise();

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_Duration& OTC_Duration::subtractDuration(OTC_Duration const& theDuration)
{
  int theSign = theDuration.mySign;

  myDays = (mySign * myDays) - (theSign * theDuration.myDays);
  myHours = (mySign * myHours) - (theSign * theDuration.myHours);
  myMinutes = (mySign * myMinutes) - (theSign * theDuration.myMinutes);
  mySeconds = (mySign * mySeconds) - (theSign * theDuration.mySeconds);

  normalise();

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_Duration& OTC_Duration::addDuration(
 int theDays,
 int theHours,
 int theMinutes,
 int theSeconds
)
{
  OTC_Duration tmpDuration(theDays,theHours,theMinutes,theSeconds);
  return addDuration(tmpDuration);
}

/* ------------------------------------------------------------------------- */
OTC_Duration& OTC_Duration::subtractDuration(
 int theDays,
 int theHours,
 int theMinutes,
 int theSeconds
)
{
  OTC_Duration tmpDuration(theDays,theHours,theMinutes,theSeconds);
  return subtractDuration(tmpDuration);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Duration::operator==(OTC_Duration const& theDuration) const
{
  return (myDays == theDuration.myDays) &&
	 (myHours == theDuration.myHours) &&
	 (myMinutes == theDuration.myMinutes) &&
	 (mySeconds == theDuration.mySeconds);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Duration::operator!=(OTC_Duration const& theDuration) const
{
  return !OTC_Duration::operator==(theDuration);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Duration::operator<=(OTC_Duration const& theDuration) const
{
  return !OTC_Duration::operator>(theDuration);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Duration::operator<(OTC_Duration const& theDuration) const
{
  if (myDays < theDuration.myDays)
    return OTCLIB_TRUE;

  else if (myDays > theDuration.myDays)
    return OTCLIB_FALSE;

  if (myHours < theDuration.myHours)
    return OTCLIB_TRUE;

  else if (myHours > theDuration.myHours)
    return OTCLIB_FALSE;

  if (myMinutes < theDuration.myMinutes)
    return OTCLIB_TRUE;

  else if (myMinutes > theDuration.myMinutes)
    return OTCLIB_TRUE;

  if (mySeconds < theDuration.mySeconds)
    return OTCLIB_TRUE;

  return OTCLIB_FALSE;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Duration::operator>=(OTC_Duration const& theDuration) const
{
  return !OTC_Duration::operator<(theDuration);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Duration::operator>(OTC_Duration const& theDuration) const
{
  if (myDays > theDuration.myDays)
    return OTCLIB_TRUE;

  else if (myDays < theDuration.myDays)
    return OTCLIB_FALSE;

  if (myHours > theDuration.myHours)
    return OTCLIB_TRUE;

  else if (myHours < theDuration.myHours)
    return OTCLIB_FALSE;

  if (myMinutes > theDuration.myMinutes)
    return OTCLIB_TRUE;

  else if (myMinutes < theDuration.myMinutes)
    return OTCLIB_TRUE;

  if (mySeconds > theDuration.mySeconds)
    return OTCLIB_TRUE;

  return OTCLIB_FALSE;
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Duration::operator-() const
{
  OTC_Duration tmpDuration(*this);
  tmpDuration.mySign = -tmpDuration.mySign;
  return tmpDuration;
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Duration::operator*(int theNum) const
{
  OTC_Duration tmpDuration(*this);
  tmpDuration *= theNum;
  return tmpDuration;
}

/* ------------------------------------------------------------------------- */
OTC_Duration& OTC_Duration::operator*=(int theNum)
{
  myDays *= (mySign * theNum);
  myHours *= (mySign * theNum);
  myMinutes *= (mySign * theNum);
  mySeconds *= (mySign * theNum);

  normalise();

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Duration::operator/(u_int theNum) const
{
  OTC_Duration tmpDuration(*this);
  tmpDuration /= theNum;
  return tmpDuration;
}

/* ------------------------------------------------------------------------- */
OTC_Duration& OTC_Duration::operator/=(u_int theNum)
{
  OTCLIB_ENSURE((theNum != 0),
   "OTC_Duration::operator/=(u_int) - zero divisor");

  double theRes;

  theRes = (double)mySeconds / theNum;

  mySeconds = int(theRes);

  theRes = (double)myMinutes / theNum;

  myMinutes = int(theRes);
  mySeconds += int((theRes - int(theRes)) * 60);

  theRes = (double)myHours / theNum;

  myHours = int(theRes);
  myMinutes += int((theRes - int(theRes)) * 60);

  theRes = (double)myDays / theNum;

  myDays = int(theRes);
  myHours += int((theRes - int(theRes)) * 24);

  normalise();

  return *this;
}

/* ------------------------------------------------------------------------- */
int OTC_Duration::operator/(OTC_Duration const& theDuration) const
{
  OTCLIB_ENSURE((!theDuration.isZeroLength()),
   "OTC_Duration::operator/(OTC_Duration const& - zero length duration");

  int theExtra = 0;

  if (theDuration.myDays != 0)
    return myDays / theDuration.myDays;

  if (myDays != 0)
    theExtra += (24 * myDays);

  if (theDuration.myHours != 0)
    return (myHours + theExtra) / theDuration.myHours;

  theExtra *= 60;
  if (myHours != 0)
    theExtra += (60 * myHours);

  if (theDuration.myMinutes != 0)
    return (myMinutes + theExtra) / theDuration.myMinutes;

  theExtra *= 60;
  if (myMinutes != 0)
    theExtra += (60 * myMinutes);

  return (mySeconds + theExtra) / theDuration.mySeconds;
}

/* ------------------------------------------------------------------------- */
void OTC_Duration::normalise()
{
  int theDiv;

  // Default sign.

  mySign = 1;

  // Bring values within range.
  // Values may still be negative.

  theDiv = mySeconds / 60;
  mySeconds %= 60;
  myMinutes += theDiv;

  theDiv = myMinutes / 60;
  myMinutes %= 60;
  myHours += theDiv;

  theDiv = myHours / 24;
  myHours %= 24;
  myDays += theDiv;

  if (myDays || myHours || myMinutes || mySeconds)
  {
    // Not all zero.

    if (myDays <= 0 && myHours <= 0 && myMinutes <= 0 && mySeconds <= 0)
    {
      // All negative or zero.

      mySign = -1;
      myDays = -myDays;
      myHours = -myHours;
      myMinutes = -myMinutes;
      mySeconds = -mySeconds;
    }
    else if (myDays < 0 || myHours < 0 || myMinutes < 0 || mySeconds < 0)
    {
      // Mixture of negative and positive.

      if (mySeconds < 0)
      {
	myMinutes -= 1;
	mySeconds += 60;
      }
      if (myMinutes < 0)
      {
	myHours -= 1;
	myMinutes += 60;
      }
      if (myHours < 0)
      {
	myDays -= 1;
	myHours += 24;
      }

      if (myDays < 0)
      {
	// Negative days, other positive.
	// Need to convert to all negative.

	myDays += 1;
	myHours -= 24;

	if (myMinutes > 0 || mySeconds > 0)
	{
	  myHours += 1;
	  myMinutes -= 60;

	  if (mySeconds > 0)
	  {
	    myMinutes += 1;
	    mySeconds -= 60;
	  }
	}

	mySign = -1;

	myDays = -myDays;
	myHours = -myHours;
	myMinutes = -myMinutes;
	mySeconds = -mySeconds;
      }
    }
  }
}

/* ------------------------------------------------------------------------- */
int OTC_Duration::rank(OTC_Duration const& theDuration) const
{
  int theRank;

  theRank = myDays - theDuration.myDays;

  if (theRank != 0)
    return theRank;

  theRank = myHours - theDuration.myHours;

  if (theRank != 0)
    return theRank;

  theRank = myMinutes - theDuration.myMinutes;

  if (theRank != 0)
    return theRank;

  theRank = mySeconds - theDuration.mySeconds;

  return theRank;
}

/* ------------------------------------------------------------------------- */
int OTC_Duration::hash() const
{
  return myDays ^ myHours ^ myMinutes ^ mySeconds;
}

#if defined(CXX_OS)
/* ------------------------------------------------------------------------- */
int OTC_RankActions<OTC_Duration>::rank(
 OTC_Duration const& theDuration1,
 OTC_Duration const& theDuration2
)
{
  return theDuration1.rank(theDuration2);
}

/* ------------------------------------------------------------------------- */
int OTC_HashActions<OTC_Duration>::hash(OTC_Duration const& theDuration)
{
  return theDuration.hash();
}
#endif

/* ------------------------------------------------------------------------- */
