/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     types/time.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/types/time.hh>

#include <time.h>
#include <iostream.h>
#include <strstream.h>
#include <string.h>

#if defined(ENV_OSTORE)
/* ------------------------------------------------------------------------- */
os_typespec* OTC_Time::typespec()
{
  static os_typespec ts("OTC_Time");
  return &ts;
}
#endif

/* ------------------------------------------------------------------------- */
OTC_Time::OTC_Time()
  : OTC_Date(0)
{
  setTime();
}

/* ------------------------------------------------------------------------- */
OTC_Time::OTC_Time(OTC_Time const& theTime)
 : OTC_Date(theTime)
{
  mySeconds = theTime.mySeconds;
}

/* ------------------------------------------------------------------------- */
OTC_Time::OTC_Time(int theHour, int theMin, int theSec)
  : OTC_Date(0)
{
  setTime(theHour,theMin,theSec);
}

/* ------------------------------------------------------------------------- */
OTC_Time::OTC_Time(
 OTC_Date const& theDate,
 int theHour,
 int theMin,
 int theSec
)
  : OTC_Date(0)
{
  setTime(theDate,theHour,theMin,theSec);
}

/* ------------------------------------------------------------------------- */
OTC_Time::OTC_Time(OTC_String const& theTimeString)
  : OTC_Date(0)
{
  int theDay = 0;
  int theMonth = 0;
  int theYear = 0;
  int theHour = 0;
  int theMin = 0;
  int theSec = 0;

  convertToTime(theTimeString,theDay,theMonth,theYear,theHour,theMin,theSec);

  OTCLIB_ENSURE((theDay && theMonth && theYear),
   "OTC_Time::OTC_Time(OTC_String const&) - invalid format for time");

  OTC_Date tmpDate(theDay,theMonth,theYear);
  setTime(tmpDate,theHour,theMin,theSec);
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Time::isValidTime(int theHour, int theMin, int theSec)
{
  if (theHour < 0 || theHour >= 24)
    return OTCLIB_FALSE;

  if (theMin < 0 || theMin >= 60)
    return OTCLIB_FALSE;

  if (theSec < 0 || theSec >= 60)
    return OTCLIB_FALSE;

  return OTCLIB_TRUE;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Time::isValidTime(OTC_String const& theTimeString)
{
  int theDay = 0;
  int theMonth = 0;
  int theYear = 0;
  int theHour = 0;
  int theMin = 0;
  int theSec = 0;

  convertToTime(theTimeString,theDay,theMonth,theYear,theHour,theMin,theSec);

  if (!theDay && !theMonth && !theYear && !theHour && !theMin && !theSec)
    return OTCLIB_FALSE;

  OTC_Boolean theResult = OTCLIB_TRUE;

  theResult &= isValidDate(theDay,theMonth,theYear);
  theResult &= isValidTime(theHour,theMin,theSec);

  return theResult;
}

/* ------------------------------------------------------------------------- */
void OTC_Time::convertToTime(
 OTC_String const& theTimeString,
 int& theDay,
 int& theMonth,
 int& theYear,
 int& theHour,
 int& theMin,
 int& theSec
)
{
  // Take a local copy as we want to cast away constness on the
  // internal buffer when using istrstream. The copy, even
  // though a delayed copy is sufficient. What we are trying to
  // protect against is access to the buffer from another thread.
  // What will happen is that since there is another reference to
  // the string internals, the other thread will make a copy of the
  // string if it wants to modify it.

  OTC_String tmpTimeString = theTimeString;

  istrstream ins((char*)tmpTimeString.string(),tmpTimeString.length());
  ins.unsetf(ios::skipws);

  theYear = 0;
  theMonth = 0;
  theDay = 0;
  theHour = 0;
  theMin = 0;
  theSec = 0;

  char theSeparator1 = 0;
  char theSeparator2 = 0;
  char theSeparator3 = 0;
  char theSeparator4 = 0;
  char theSeparator5 = 0;

  ins >> dec >> theYear;
  ins >> theSeparator1;
  ins >> dec >> theMonth;
  ins >> theSeparator2;
  ins >> dec >> theDay;
  ins >> theSeparator3;
  ins >> dec >> theHour;
  ins >> theSeparator4;
  ins >> dec >> theMin;
  ins >> theSeparator5;
  ins >> dec >> theSec;

  if (ins.fail() || theSeparator1 != '-' || theSeparator2 != '-'
   || theSeparator3 != 'T' || theSeparator4 != ':' || theSeparator5 != ':'
  )
  {
    theYear = theMonth = theDay = 0;
    theHour = theMin = theSec = 0;
    return;
  }
}

/* ------------------------------------------------------------------------- */
int OTC_Time::hour() const
{
  return mySeconds / (60*60);
}

/* ------------------------------------------------------------------------- */
int OTC_Time::minute() const
{
  return (mySeconds / 60) % 60;
}

/* ------------------------------------------------------------------------- */
int OTC_Time::second() const
{
  return mySeconds % 60;
}

/* ------------------------------------------------------------------------- */
OTC_Time OTC_Time::plusSeconds(int theNum) const
{
  OTC_Time theTime(*this);
  theTime.addSeconds(theNum);
  return theTime;
}

/* ------------------------------------------------------------------------- */
OTC_Time OTC_Time::plusDuration(OTC_Duration const& theDuration) const
{
  OTC_Time theTime(*this);
  theTime.addDuration(theDuration);
  return theTime;
}

/* ------------------------------------------------------------------------- */
OTC_String OTC_Time::asString() const
{
  OTC_String tmpString = OTC_Length(47);
  ostrstream theStream(tmpString.buffer(),tmpString.capacity());

  int tmpDay;
  int tmpMonth;
  int tmpYear;

  OTC_Date::calendarDate(jday(),tmpDay,tmpMonth,tmpYear);

  theStream << tmpYear << '-';
  theStream.width(2);
  theStream.fill('0');
  theStream << tmpMonth << '-';
  theStream.width(2);
  theStream.fill('0');
  theStream << tmpDay << 'T';
  theStream.width(2);
  theStream.fill('0');
  theStream << hour() << ':';
  theStream.width(2);
  theStream.fill('0');
  theStream << minute() << ':';
  theStream.width(2);
  theStream.fill('0');
  theStream << second() << ends;

  int theLength = strlen(tmpString.buffer());
  tmpString.length(theLength);

  return tmpString;
}

/* ------------------------------------------------------------------------- */
ostream& operator<<(ostream& theStream, OTC_Time const& theTime)
{
  int tmpDay;
  int tmpMonth;
  int tmpYear;

  OTC_Date::calendarDate(theTime.jday(),tmpDay,tmpMonth,tmpYear);

  theStream << tmpYear << '-';
  theStream.width(2);
  theStream.fill('0');
  theStream << tmpMonth << '-';
  theStream.width(2);
  theStream.fill('0');
  theStream << tmpDay << 'T';
  theStream.width(2);
  theStream.fill('0');
  theStream << theTime.hour() << ':';
  theStream.width(2);
  theStream.fill('0');
  theStream << theTime.minute() << ':';
  theStream.width(2);
  theStream.fill('0');
  theStream << theTime.second();

  return theStream;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Time::operator<(OTC_Time const& theTime) const
{
  if (OTC_Date::operator<(theTime))
    return OTCLIB_TRUE;

  else if (OTC_Date::operator==(theTime))
    return mySeconds < theTime.mySeconds;

  else
    return OTCLIB_FALSE;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Time::operator<=(OTC_Time const& theTime) const
{
  if (OTC_Date::operator<(theTime))
    return OTCLIB_TRUE;

  else if (OTC_Date::operator==(theTime))
    return mySeconds <= theTime.mySeconds;

  else
    return OTCLIB_FALSE;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Time::operator>(OTC_Time const& theTime) const
{
  if (OTC_Date::operator>(theTime))
    return OTCLIB_TRUE;

  else if (OTC_Date::operator==(theTime))
    return mySeconds > theTime.mySeconds;

  else
    return OTCLIB_FALSE;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Time::operator>=(OTC_Time const& theTime) const
{
  if (OTC_Date::operator>(theTime))
    return OTCLIB_TRUE;

  else if (OTC_Date::operator==(theTime))
    return mySeconds >= theTime.mySeconds;

  else
    return OTCLIB_FALSE;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Time::operator==(OTC_Time const& theTime) const
{
  if (OTC_Date::operator!=(theTime))
    return OTCLIB_FALSE;

  else
    return mySeconds == theTime.mySeconds;
}

/* ------------------------------------------------------------------------- */
OTC_Boolean OTC_Time::operator!=(OTC_Time const& theTime) const
{
  if (OTC_Date::operator!=(theTime))
    return OTCLIB_TRUE;

  else
    return mySeconds != theTime.mySeconds;
}

/* ------------------------------------------------------------------------- */
void OTC_Time::setTime()
{
  time_t theClock = ::time(0);
  tm* theTime = localtime(&theClock);
  setDate(theTime->tm_mday,theTime->tm_mon+1,theTime->tm_year+1900);
  mySeconds = (theTime->tm_hour*60*60)+(theTime->tm_min*60)+theTime->tm_sec;
}

/* ------------------------------------------------------------------------- */
void OTC_Time::setTime(OTC_Time const& theTime)
{
  setDate(theTime);
  mySeconds = theTime.mySeconds;
}

/* ------------------------------------------------------------------------- */
void OTC_Time::setTime(int theHour, int theMin, int theSec)
{
  setDate();
  mySeconds = (theHour*60*60)+(theMin*60)+theSec;
}

/* ------------------------------------------------------------------------- */
void OTC_Time::setTime(
 OTC_Date const& theDate,
 int theHour,
 int theMin,
 int theSec
)
{
  setDate(theDate);
  mySeconds = (theHour*60*60)+(theMin*60)+theSec;
}

/* ------------------------------------------------------------------------- */
OTC_Time& OTC_Time::addSeconds(int theNum)
{
  mySeconds += theNum;
  if (mySeconds >= (24*60*60))
  {
    addDays(mySeconds/(24*60*60));
    mySeconds = mySeconds % (24*60*60);
  }
  else if (mySeconds < 0)
  {
    addDays(mySeconds/(24*60*60));
    mySeconds = mySeconds - (mySeconds % (24*60*60));
  }
  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_Time& OTC_Time::addMinutes(int theNum)
{
  return addSeconds(60*theNum);
}

/* ------------------------------------------------------------------------- */
OTC_Time& OTC_Time::addHours(int theNum)
{
  return addSeconds(60*60*theNum);
}

/* ------------------------------------------------------------------------- */
OTC_Time& OTC_Time::addDuration(OTC_Duration const& theDuration)
{
  int theSign = theDuration.sign();

  addSeconds(theSign*theDuration.seconds());
  addMinutes(theSign*theDuration.minutes());
  addHours(theSign*theDuration.hours());
  addDays(theSign*theDuration.days());

  return *this;
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Time::operator-(OTC_Time const& theTime) const
{
  OTC_Duration d(0);

  if (this == &theTime || *this == theTime)
    return d;

  d = OTC_Date::operator-(theTime);
  d.subtractDuration(0,theTime.hour(),theTime.minute(),theTime.second());
  d.addDuration(0,hour(),minute(),second());
  return d;
}

/* ------------------------------------------------------------------------- */
OTC_Duration OTC_Time::operator-(OTC_Date const& theDate) const
{
  OTC_Duration d(0);

  d = OTC_Date::operator-(theDate);
  d.addDuration(0,hour(),minute(),second());

  return d;
}

/* ------------------------------------------------------------------------- */
int OTC_Time::rank(OTC_Time const& theTime) const
{
  int theRank = OTC_Date::rank(theTime);

  if (theRank == 0)
    theRank = mySeconds - theTime.mySeconds;

  return theRank;
}

#if defined(CXX_OS)
/* ------------------------------------------------------------------------- */
int OTC_RankActions<OTC_Time>::rank(
 OTC_Time const& theTime1,
 OTC_Time const& theTime2
)
{
  return theTime1.rank(theTime2);
}

/* ------------------------------------------------------------------------- */
int OTC_HashActions<OTC_Time>::hash(OTC_Time const& theTime)
{
  return theTime.hash();
}
#endif

/* ------------------------------------------------------------------------- */
