/*
// ============================================================================
//
// = LIBRARY
//     OUX
// 
// = FILENAME
//     dispatch/ouxjobqueue.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma implementation "OUX/dispatch/jobqueue.hh"
#endif
#endif

#include <OUX/dispatch/jobqueue.hh>
#include <OUX/dispatch/signal.hh>
#include <OTC/dispatch/timeout.hh>
#include <OTC/dispatch/alarm.hh>
#include <OTC/dispatch/ioevent.hh>

#include <stdlib.h>
#include <string.h>

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#ifdef ENV_VXWORKS
#include <selectLib.h>
#endif

// Use poll() instead of select(), unless we are on SunOS 4.1.X.

#if defined(sun) && defined(HAVE_SELECT) && defined(HAVE_POLL)
#if defined(HAVE_SYS_RESOURCE_H) && !defined(HAVE_SYS_ELF_H)
#undef HAVE_POLL
#endif
#endif

#if defined(HAVE_SELECT) && defined(HAVE_POLL)
#undef HAVE_SELECT
#endif

#if defined(HAVE_POLL)
#if defined(HAVE_STROPTS_H)
#include <stropts.h>
#endif
#define reqevents events
#define rtnevents revents
#include <poll.h>
#undef reqevents
#undef rtnevents
#undef events
#undef revents
#endif

#if defined(HAVE_SELECT)
#include <sys/types.h>
#include <unistd.h>
#ifndef M_XENIX
#include <sys/time.h>
#endif
#if defined(HAVE_SYS_SELECT_H)
#include <sys/select.h>
#endif
#if defined(CXX_DEC)
extern "C" void bzero(char*, int);
extern "C" int select(int, fd_set*, fd_set*, fd_set*, const timeval*);
#endif
#endif

#if defined(HAVE_POLL)
#if defined(__osf__)
extern "C" int poll(struct pollfd[], unsigned int, int);
#else
extern "C" int poll(pollfd*, unsigned long, int);
#endif
#endif

#ifdef _MSC_VER
#include <windows.h>
#include <winsock.h>
#define HAVE_SELECT
#endif

#include <stdio.h>
#include <errno.h>

#ifndef ENV_VXWORKS
#if defined(HAVE_SYS_PARAM_H)
#include <sys/param.h>
#endif
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef NOFILE
#ifdef OPEN_MAX
#define NOFILE OPEN_MAX
#else
#ifdef FOPEN_MAX
#define NOFILE FOPEN_MAX
#else
#ifdef NFDS
#define NOFILE NFDS
#else
#ifdef _NFILE
#define NOFILE _NFILE
#else
#ifdef _NFILE_
#define NOFILE _NFILE_
#endif
#endif
#endif
#endif
#endif
#endif

#ifdef HAVE_SYSCONF
extern "C" long sysconf(int);
#endif

// If we do not have sysconf() and NOFILE still isn't set then tough. :-)

#if defined(NEED_MEMCPY)
extern "C" void* memcpy(void*, void const*, int);
#endif

/* ------------------------------------------------------------------------- */
OUX_JobQueue::OUX_JobQueue()
{
  myPriorityJobs = new OTC_JobQueue;
  OTCLIB_ASSERT(myPriorityJobs != 0);

  myIdleJobs = new OTC_JobQueue;
  OTCLIB_ASSERT(myIdleJobs != 0);

  myPendingFdEvents = new int[nfds()];
  OTCLIB_ASSERT(myPendingFdEvents != 0);

  for (int i=nfds(); i>0; i--)
    myPendingFdEvents[i-1] = 0;

  myNextFd = 0;
  myMaxFd = -1;
}

/* ------------------------------------------------------------------------- */
OUX_JobQueue::~OUX_JobQueue()
{
  delete myIdleJobs;
  delete [] myPendingFdEvents;
}

/* ------------------------------------------------------------------------- */
void OUX_JobQueue::add(OTC_Job* theJob, int theOptions)
{
  OTCLIB_ENSURE((theJob != 0),
   "OUX_JobQueue::add(OTC_Job*, int) - invalid job");

  if (theOptions == OUXLIB_IDLE_JOB)
    OTC_JobQueue::add(theJob,0);
  else
    myPriorityJobs->add(theJob,0);
}

/* ------------------------------------------------------------------------- */
OTC_Job* OUX_JobQueue::next(int theActions, int)
{
  theActions = theActions & (OUXLIB_ALL_JOBS | OUXLIB_DONT_WAIT);
  if ((theActions & OUXLIB_ALL_JOBS) == 0)
    theActions |= OUXLIB_ALL_JOBS;

  OTC_Job* theJob = 0;

  while (theJob == 0)
  {
    // Check for signals.

    if (theActions & OUXLIB_SIGNAL_JOBS)
    {
      theJob = OUXEV_Signal::pending();

      if (theJob != 0)
	return theJob;
    }

    // Check for alarms.

    if (theActions & OUXLIB_ALARM_JOBS)
    {
      theJob = OTCEV_Alarm::pending();

      if (theJob != 0)
	return theJob;
    }

    // Check for priority jobs.

    if (theActions & OUXLIB_PRIORITY_JOBS)
    {
      theJob = myPriorityJobs->next();

      if (theJob != 0)
       return theJob;
    }

    // Check if we still have idle jobs to process.

    if (theActions & OUXLIB_IDLE_JOBS)
    {
      theJob = myIdleJobs->next();

      if (theJob != 0)
	return theJob;
    }

    // Check for io events which haven't yet been processed.

    if (theActions & OUXLIB_IO_JOBS)
    {
      while (myNextFd <= myMaxFd)
      {
	int theFd = myNextFd++;
	int theEvents = myPendingFdEvents[theFd];

	if (theEvents)
	{
	  myPendingFdEvents[theFd] = 0;
	  theJob = OTCEV_IOEvent::job(theFd,theEvents);

	  if (theJob != 0)
	    return theJob;
	}
      }
    }

    // Check for timeouts.

    if (theActions & OUXLIB_TIMEOUT_JOBS)
    {
      theJob = OTCEV_Timeout::pending();

      if (theJob != 0)
	return theJob;
    }

    // Work out period of timeout.

    long alarmPeriod = -1;
    long timeoutPeriod = -1;
    long period = -1;

    if (theActions & OUXLIB_ALARM_JOBS)
      alarmPeriod = 1000 * OTCEV_Alarm::period();

    if (theActions & OUXLIB_TIMEOUT_JOBS)
      timeoutPeriod = OTCEV_Timeout::period();

    if (alarmPeriod < 0 && timeoutPeriod >= 0)
      period = timeoutPeriod;
    else if (timeoutPeriod < 0 && alarmPeriod >= 0)
      period = alarmPeriod;
    else if (alarmPeriod >= 0 && timeoutPeriod >= 0)
      period = (timeoutPeriod < alarmPeriod) ? timeoutPeriod : alarmPeriod;

    // Set timeout period to zero if we shouldn't wait.

    long rawPeriod = period;

    if (theActions & OUXLIB_DONT_WAIT)
      period = 0;

    // Find out how many file descriptors we should check.

    int maxFd = -1;

    if (theActions & OUXLIB_IO_JOBS)
      maxFd = OTCEV_IOEvent::maxFd();

#if defined(HAVE_POLL)

    // Calculate timeout for <poll()>.

    int timeout = int(period);

    // Calculate io events we are interested in.

    int numFd = 0;
    static pollfd* fds = new pollfd[nfds()];

    if (theActions & OUXLIB_IO_JOBS)
    {
      for (int i=0; i<=maxFd; i++)
      {
	fds[numFd].events = OTCEV_IOEvent::events(i);
	if (fds[numFd].events != 0)
	{
	  fds[numFd].fd = i;
	  fds[numFd].revents = 0;
	  numFd++;
	}
      }
    }

    if (theActions & OUXLIB_SIGNAL_JOBS)
    {
      int signalFd = OUXEV_Signal::fd();
      if (signalFd != -1)
      {
	fds[numFd].fd = signalFd;
	fds[numFd].events = POLLIN;
	fds[numFd].revents = 0;
	numFd++;
      }
    }

    // Do the <poll()> call.

    int res = 0;

    if ((theActions & OUXLIB_IDLE_JOBS) && !OTC_JobQueue::isEmpty())
    {
      if (numFd > 0)
	res = poll(fds,numFd,0);

      if (res == 0)
      {
	if (rawPeriod != 0)
	{
	  theJob = OTC_JobQueue::next();
	  while (theJob != 0)
	  {
	    myIdleJobs->add(theJob);
	    theJob = OTC_JobQueue::next();
	  }

	  theJob = myIdleJobs->next();

	  if (theJob != 0)
	    return theJob;
	}
      }
    }

    res = 0;

    if (numFd != 0 || period != -1)
      res = poll(fds,numFd,timeout);
    else
      return 0;

    // Generate io events.

    if (res > 0)
    {
      maxFd = -1;
      for (int i=0; i<numFd; i++)
      {
	if (fds[i].revents != 0)
	{
	  if (fds[i].fd > maxFd)
	    maxFd = fds[i].fd;

	  myPendingFdEvents[fds[i].fd] = fds[i].revents;
	}
      }

      if (maxFd != -1)
      {
	myNextFd = 0;
	myMaxFd = maxFd;
      }
    }
    else if (res == 0)
    {
      if (theActions & OUXLIB_DONT_WAIT)
      {
	if (rawPeriod != 0)
	  return 0;
      }
    }

#else
#if defined(HAVE_SELECT)

    // Calculate timeout for <select()>.

    timeval timeout;
    timeval* timeoutp = 0;

    if (period != -1)
    {
      timeoutp = &timeout;
      timeout.tv_sec = period / 1000;
      timeout.tv_usec = 1000 * (period % 1000);
    }

    // Calculate ioevents we are interested in.

    int width = maxFd + 1;

    fd_set readfds;
    fd_set writefds;
    fd_set exceptfds;

    FD_ZERO(&readfds);
    FD_ZERO(&writefds);
    FD_ZERO(&exceptfds);

    if (theActions & OUXLIB_IO_JOBS)
    {
      for (int i=0; i<width; i++)
      {
	int theEvents = OTCEV_IOEvent::events(i);

	if (theEvents & OTCLIB_POLLIN)
	  FD_SET(i,&readfds);
	if (theEvents & OTCLIB_POLLOUT)
	  FD_SET(i,&writefds);
	if (theEvents & OTCLIB_POLLPRI)
	  FD_SET(i,&exceptfds);
      }
    }

    if (theActions & OUXLIB_SIGNAL_JOBS)
    {
      int signalFd = OUXEV_Signal::fd();
      if (signalFd != -1)
      {
	FD_SET(signalFd,&readfds);
	if (signalFd >= width)
	  width = signalFd+1;
      }
    }

    // Do the <select()> call.

    int res = 0;

    if ((theActions & OUXLIB_IDLE_JOBS) && !OTC_JobQueue::isEmpty())
    {
      timeval timeout;
      timeout.tv_sec = 0;
      timeout.tv_usec = 0;

      fd_set t_readfds;
      fd_set t_writefds;
      fd_set t_exceptfds;

      memcpy(&t_readfds,&readfds,sizeof(fd_set));
      memcpy(&t_writefds,&writefds,sizeof(fd_set));
      memcpy(&t_exceptfds,&exceptfds,sizeof(fd_set));

      if (width > 0)
      {
#if defined(hpux)
	res = select(width,(int*)&t_readfds,(int*)&t_writefds,
	 (int*)&t_exceptfds,&timeout);
#else
	res = select(width,&t_readfds,&t_writefds,&t_exceptfds,&timeout);
#endif

	// If we have a bad file descriptor, cycle through all file
	// descriptors, doing a select on each, such that we can find
	// which is the bad file descriptor and generate a OTCLIB_POLLNVAL
	// event for it. We can only assume that the recipient will take
	// that event and unsubscribe the file descriptor. If they do not
	// do this, the select() call will most likely loop. Note that
	// we can do select() just on input events as whether the type of
	// event is ready or not is not important but whether select() likes
	// the file descriptor.

	if (res == -1 && errno == EBADF)
	{
	  fd_set nullfds;
	  fd_set checkfds;

	  FD_ZERO(&nullfds);
	  FD_ZERO(&checkfds);

	  for (int i=0; i<width; i++)
	  {
	    int theEvents = OTCEV_IOEvent::events(i);

	    if (theEvents != 0)
	    {
	      FD_SET(i,&checkfds);

#if defined(hpux)
	      res = select(width,(int*)&checkfds,(int*)&nullfds,
	       (int*)&nullfds,&timeout);
#else
	      res = select(width,&checkfds,&nullfds,&nullfds,&timeout);
#endif

	      if (res == -1 && errno == EBADF)
	      {
		theJob = OTCEV_IOEvent::job(i,OTCLIB_POLLNVAL);

		// Should always get a job back, but just in case.

		if (theJob != 0)
		  return theJob;
	      }

	      FD_CLR(i,&checkfds);
	    }
	  }

	  // If we get here and haven't found a bad file descriptor
	  // then we have a problem. All we can do is set result back
	  // to -1 and keep going and hope that everything is okay.

          res = -1;
	}
      }

      if (res == 0)
      {
	if (rawPeriod != 0)
	{
	  theJob = OTC_JobQueue::next();
	  while (theJob != 0)
	  {
	    myIdleJobs->add(theJob);
	    theJob = OTC_JobQueue::next();
	  }

	  theJob = myIdleJobs->next();

	  if (theJob != 0)
	    return theJob;
	}
      }
    }

    res = 0;

    if (width != 0 || period != -1)
    {
#if defined(hpux)
      res = select(width,(int*)&readfds,(int*)&writefds,
       (int*)&exceptfds,timeoutp);
#else
      res = select(width,&readfds,&writefds,&exceptfds,timeoutp);
#endif
    }
    else
      return 0;

    // Generate io events.

    if (res > 0)
    {
      maxFd = -1;
      for (int i=0; i<width; i++)
      {
	int theEvents = 0;

	if (FD_ISSET(i,&readfds))
	  theEvents |= OTCLIB_POLLIN;
	if (FD_ISSET(i,&writefds))
	  theEvents |= OTCLIB_POLLOUT;
	if (FD_ISSET(i,&exceptfds))
	  theEvents |= OTCLIB_POLLPRI;

	if (theEvents != 0)
	{
	  maxFd = i;
	  myPendingFdEvents[i] = theEvents;
	}
      }

      if (maxFd != -1)
      {
	myNextFd = 0;
	myMaxFd = maxFd;
      }
    }
    else if (res == 0)
    {
      if (theActions & OUXLIB_DONT_WAIT)
      {
	if (rawPeriod != 0)
	  return 0;
      }
    }

#else

    OTCLIB_EXCEPTION("UNIX dispatcher not supported");

#endif /* HAVE_SELECT */
#endif /* HAVE_POLL */
  }

  return theJob;
}

/* ------------------------------------------------------------------------- */
int OUX_JobQueue::nfds()
{
  // Assumed that this is always called from inside a lock.
  // Not really supposed to cache these values as they could
  // change if resource limits of a program are changed. But
  // then we base the size of an array on the first call and
  // thus probably do not want this to change, ie., get larger.
  // Can only assume that when first called that limit is at
  // the max it will for the program.

#ifdef HAVE_SYSCONF_SC_OPEN_MAX
  static int _nfds = (int)sysconf(_SC_OPEN_MAX);
#else
#ifdef HAVE_GETDTABLESIZE
  static int _nfds = getdtablesize();
#else
  static int _nfds = NOFILE;
#endif
#endif
  return _nfds;
}

/* ------------------------------------------------------------------------- */
