/*
// ============================================================================
//
// = LIBRARY
//     OUX
// 
// = FILENAME
//     dispatch/ouxsignal.cc
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#ifdef __GNUG__
#pragma implementation "OUX/dispatch/signal.hh"
#endif

#include <OUX/dispatch/signal.hh>
#include <OUX/system/sigblock.hh>
#include <OTC/dispatch/jobqueue.hh>
#include <OTC/dispatch/eventjob.hh>
#include <OTC/debug/logstrm.hh>
#include <OUX/system/sighndle.hh>

#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <iostream.h>
#if defined(HAVE_UNISTD_H)
#include <unistd.h>
#endif
#if defined(CXX_ATT2_1) || defined(CXX_CL1_1)
#include <osfcn.h>
#endif
#ifdef _MSC_VER
#include <io.h>
#endif

/* ------------------------------------------------------------------------- */
class OUX_SignalSubscription
{
  public:

    int			agentId;

    OUX_SignalSubscription*	next;

    void*		operator new(size_t theSize)
				{ return OTC_CommonPool::allocate(theSize); }

    void		operator delete(void* theMem, size_t theSize)
				{ OTC_CommonPool::release(theMem,theSize); }
};

class OUX_SignalInfo
{
  public:

    int			signal;

    int			count;
};

/* ------------------------------------------------------------------------- */
OTC_NRMutex OUXEV_Signal::_mutex;
OUX_SignalSubscription** OUXEV_Signal::globSubscriptions = 0;
OUX_SignalInfo* OUXEV_Signal::globPendingSignals = 0;
u_int OUXEV_Signal::globPendingSize = 0;
u_int OUXEV_Signal::globNextAvailable = 0;
OTC_Boolean OUXEV_Signal::globSignalsLost = OTCLIB_FALSE;
u_int OUXEV_Signal::globSigCount = 0;
OTC_JobQueue* OUXEV_Signal::globSignalJobs = 0;
int OUXEV_Signal::globSignalFds[2] = { -1, -1 };
int OUXEV_Signal::globTypeId = 0;

/* ------------------------------------------------------------------------- */
OUXEV_Signal::~OUXEV_Signal()
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
void* OUXEV_Signal::type() const
{
  return typeId();
}

/* ------------------------------------------------------------------------- */
void OUXEV_Signal::dump(ostream& outs) const
{
  outs << "<OUX> SIGNAL - number = " << signal();
}

/* ------------------------------------------------------------------------- */
void OUXEV_Signal::subscribe(int theAgentId, int theSignal)
{
  OTCLIB_ENSURE((theSignal > 0 && theSignal < NSIG),
   "OUXEV_Signal::subscribe() - invalid signal number");
  OTCLIB_ENSURE((theAgentId != 0),
   "OUXEV_Signal::subscribe() - invalid agent ID");

  _mutex.lock();

  OUX_SignalBlock sigblock;

  if (globPendingSize == 0)
  {
    globPendingSize = 64;

    globPendingSignals = new OUX_SignalInfo[globPendingSize];
    OTCLIB_ASSERT(globPendingSignals != 0);
  }

  if (globSignalFds[0] == -1)
  {
#ifndef _MSC_VER
    if (pipe(globSignalFds) == -1)
    {
      globSignalFds[0] = -1;
      globSignalFds[1] = -1;
    }
    else
    {
      fcntl(globSignalFds[0],F_SETFD,FD_CLOEXEC);
      fcntl(globSignalFds[1],F_SETFD,FD_CLOEXEC);
    }
#endif
  }

  OTC_Boolean isSubscribed = OTCLIB_FALSE;

  if (globSubscriptions == 0)
  {
    globSubscriptions = new OUX_SignalSubscription*[NSIG];
    OTCLIB_ASSERT(globSubscriptions != 0);
    for (int i=0; i<NSIG; i++)
      globSubscriptions[i] = 0;
  }
  else
  {
    OUX_SignalSubscription* aSubscription;
    aSubscription = globSubscriptions[theSignal];
    while (aSubscription != 0)
    {
      if (aSubscription->agentId == theAgentId)
      {
	isSubscribed = OTCLIB_TRUE;
	break;
      }
      else
	aSubscription = aSubscription->next;
    }
  }

  if (isSubscribed == OTCLIB_FALSE)
  {
    OUX_SignalSubscription* aSubscription;
    aSubscription = new OUX_SignalSubscription;
    OTCLIB_ASSERT(aSubscription != 0);
    aSubscription->agentId = theAgentId;
    aSubscription->next = globSubscriptions[theSignal];
    globSubscriptions[theSignal] = aSubscription;

    if (aSubscription->next == 0)
    {
      OUX_SignalHandler::install(theSignal,sighandler);
      globSigCount++;
    }
  }

  sigblock.release();

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OUXEV_Signal::unsubscribe(int theAgentId, int theSignal)
{
  if (theSignal <= 0 || theSignal >= NSIG || theAgentId == 0)
    return;

  _mutex.lock();

  if (globSubscriptions != 0 && globSigCount != 0)
  {
    OUX_SignalSubscription* aSubscription;
    aSubscription = globSubscriptions[theSignal];

    if (aSubscription != 0)
    {
      OUX_SignalBlock sigblock;

      if (aSubscription->agentId == theAgentId)
      {
	globSubscriptions[theSignal] = aSubscription->next;
	delete aSubscription;
      }
      else
      {
	OUX_SignalSubscription* tmpSubscription;
	tmpSubscription = aSubscription;
	aSubscription = aSubscription->next;
	while (aSubscription != 0)
	{
	  if (aSubscription->agentId == theAgentId)
	  {
	    tmpSubscription->next = aSubscription->next;
	    delete aSubscription;
	    break;
	  }
	  else
	  {
	    tmpSubscription = aSubscription;
	    aSubscription = aSubscription->next;
	  }
	}
      }

      if (globSubscriptions[theSignal] == 0)
      {
	OUX_SignalHandler::install(theSignal,0);
	globSigCount--;
      }
    }
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
void OUXEV_Signal::unsubscribeAgent(int theAgentId)
{
  if (theAgentId == 0)
    return;

  for (int i=1; i<NSIG; i++)
    unsubscribe(theAgentId,i);
}

/* ------------------------------------------------------------------------- */
void OUXEV_Signal::capacity(u_int theCapacity)
{
  _mutex.lock();

  if (globPendingSize == 0)
  {
    OTCLIB_ENSURE((theCapacity != 0),
     "OEXEV_Signal::capacity() - capacity must be non zero");

    globPendingSize = theCapacity;
    globPendingSignals = new OUX_SignalInfo[globPendingSize];
    OTCLIB_ASSERT(globPendingSignals != 0);
  }

  _mutex.unlock();
}

/* ------------------------------------------------------------------------- */
OTC_Job* OUXEV_Signal::pending()
{
  OTC_Job* theJob;
  theJob = 0;

  _mutex.lock();

  if (globSignalJobs == 0)
  {
    globSignalJobs = new OTC_JobQueue;
    OTCLIB_ASSERT(globSignalJobs != 0);
  }

  fill(globSignalJobs);
  theJob = globSignalJobs->next();

  _mutex.unlock();

  return theJob;
}

/* ------------------------------------------------------------------------- */
void OUXEV_Signal::fill(OTC_JobQueue* theJobQueue)
{
  // Assumed that this is called from within thread lock. This is
  // the case where called from pending().

  // Check whether we have any subscriptions.

  if (globPendingSize == 0 || globSigCount == 0)
    return;

  // Mask further signals.

  OUX_SignalBlock sigblock;

  // Check whether we have any signals pending.

  if (globNextAvailable == 0)
    return;

  // Remove signal byte from pipe.

  if (fd() != -1)
  {
    char buf[1];
    read(fd(),buf,1);
  }

  // Check for discarded signals.

  if (globSignalsLost != OTCLIB_FALSE)
  {
    globSignalsLost = OTCLIB_FALSE;
    OTCLIB_LOGGER(OTCLIB_LOG_WARNING)
     << "OUXEV_Signal::fill() - overflow, signals were lost" << flush;
  }

  // Fill the queue.

  u_int i = 0;
  while (i != globNextAvailable)
  {
    OUX_SignalInfo& theInfo = globPendingSignals[i];

    OUXEV_Signal* theSignal;
    theSignal = new OUXEV_Signal(theInfo.signal);
    OTCLIB_ASSERT(theSignal);

    while (theInfo.count != 0)
    {
      OUX_SignalSubscription* aSubscription;
      aSubscription = globSubscriptions[theInfo.signal];

      while (aSubscription != 0)
      {
	OTC_Event* aEvent;
	aEvent = theSignal->clone();

	OTC_EventJob* theJob;
	theJob = new OTC_EventJob(aSubscription->agentId,aEvent);
	OTCLIB_ASSERT(theJob);

	theJobQueue->add(theJob);

	aSubscription = aSubscription->next;
      }
      theInfo.count--;
    }

    theSignal->destroy();

    i++;
  }

  globNextAvailable = 0;
}

/* ------------------------------------------------------------------------- */
void OUXEV_Signal::sighandler(int theSignal)
{
  // Micro$oft C++ unregisters signal handler, must reregister
  // it ourselves.

#ifdef _MSC_VER
  OUX_SignalHandler::install(theSignal,sighandler);
#endif

  // This may get called while inside a thread lock. This is okay as
  // critical parts of code within thread lock are protected against
  // signals occuring. Therefore we do not want to add thread lock in
  // here as it would cause all sorts of problems.

  OUX_SignalBlock sigblock;

  if (globNextAvailable == globPendingSize)
  {
    globSignalsLost = OTCLIB_TRUE;
    return;
  }

  if (globNextAvailable != 0)
  {
    OUX_SignalInfo& theInfo = globPendingSignals[globNextAvailable-1];
    if (theInfo.signal == theSignal)
    {
      theInfo.count++;
      return;
    }
  }
  else
  {
    int num = errno;
    if (globSignalFds[1] != -1)
      write(globSignalFds[1],"X",1);
    errno = num;
  }

  OUX_SignalInfo& theInfo = globPendingSignals[globNextAvailable];
  theInfo.signal = theSignal;
  theInfo.count = 1;

  globNextAvailable++;
}

/* ------------------------------------------------------------------------- */
void OUXEV_Signal::cancelSource(int theAgentId)
{
  OUXEV_Signal::unsubscribeAgent(theAgentId);

  OTCLIB_LOGGER(OTCLIB_LOG_WARNING) <<
   "OUXEV_Signal::cancelSource() - signal subscription cancelled" << flush;
  OTCLIB_LOGGER(OTCLIB_LOG_DEBUG) <<
   "Agent " << theAgentId << flush;
}

/* ------------------------------------------------------------------------- */
