/*
-----------------------------------------------------------------------------
| Source File   | X10_EXMP.C                                                |
|               |                                                           |
| System        | X-10 Professional Developers Library for C, v. 3.1        |
|               |                                                           |
| Purpose       | Example of a working X10 schedule.                        |
|               |                                                           |
| Author        | Dale Morrison                                             |
| Copyright     | Smart Home Technology, 1993                               |
-----------------------------------------------------------------------------


-----------------------------------------------------------------------------
| Load the headers for standard C functions used in this example program    |
----------------------------------------------------------------------------- */
#include <conio.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/*
-----------------------------------------------------------------------------
| Load the file containing declarations and constants for the X10 Library   |
----------------------------------------------------------------------------- */
#include "X10_C.H"

/*
-----------------------------------------------------------------------------
| Define your own unique constants and functions.                           |
----------------------------------------------------------------------------- */
#define OUTAGE_FILE         "outage.sta"    /* Name of the recovery file    */
#define SAVE_STATUS_TIMER   0               /* Timer number to use          */
#define SAVE_STATUS_TIME    "00:30:00"      /* How often to save status     */
#define RECV_TIMEOUT        .5              /* Seconds to wait for a cmd    */
#define LONGITUDE           122.2           /* Longitude of your city       */
#define LATITUDE            47.37           /* Latitude of your city        */
#define TIME_ZONE           8               /* Time zone for your city      */
#define START_DAYLITE       ""              /* Have the library calculate   */
#define END_DAYLITE         ""              /* Have the library calculate   */

/*
-----------------------------------------------------------------------------
| Define your own unique functions.                                         |
----------------------------------------------------------------------------- */
void startup(void);
void chk_time_event(void);
void chk_ctrler_event(void);
void chk_timer_event(void);

/*
-----------------------------------------------------------------------------
| Define your custom variables that are initialized in the Startup procedure|
----------------------------------------------------------------------------- */
int JrIsWorking, CancelBedtime, ResetMotion;
char hc[2] = " ";

/*
-----------------------------------------------------------------------------
| Start of your program                                                     |
----------------------------------------------------------------------------- */
void main(void)
{
    char KeyHit, NewTime[6] = "";

    /*
    -------------------------------------------------------------------------
    | Your basic clear the screen command                                   |
    ------------------------------------------------------------------------- */
    #ifdef __TURBOC__
        clrscr();
    #else
        system("cls");
    #endif

    /*
    -------------------------------------------------------------------------
    | Open communications to the TW523 module                               |
    ------------------------------------------------------------------------- */
    open_tw523(OUTAGE_FILE);

    /*
    -------------------------------------------------------------------------
    | Do you want to display what is being transmitted and received?        |
    ------------------------------------------------------------------------- */
	ShowTransmit = YES;

    /*
    -------------------------------------------------------------------------
    | Number of seconds you want the system to wait for a complete x10 cmd  |
    | The system defaults to 1 second.  This in no way slows the system down|
    ------------------------------------------------------------------------- */
    set_recv_timeout(RECV_TIMEOUT);

    /*
    -------------------------------------------------------------------------
    | Initialize the internal variables for sunrise/sunset calculations.    |
    ------------------------------------------------------------------------- */
    set_map_coord(LONGITUDE, LATITUDE, TIME_ZONE, START_DAYLITE, END_DAYLITE);

    /*
    -------------------------------------------------------------------------
    | Do your Startup procedure to initialize your modules and variables    |
    ------------------------------------------------------------------------- */
    startup();

    /*
    -------------------------------------------------------------------------
    | Start a timer to be used for saving the status tables for outages.    |
    ------------------------------------------------------------------------- */
    start_timer(SAVE_STATUS_TIMER, SAVE_STATUS_TIME);

    /*
    -------------------------------------------------------------------------
    | Check to see if there was a power outage since the last boot up.      |
    ------------------------------------------------------------------------- */
    if (PowerOutage) {
        printf("\n!!! There was a power outage.... ");
        printf("one moment while I catch up!\n\n");
        play_catchup(OUTAGE_FILE);
    }

    /*
    -------------------------------------------------------------------------
    | Now start the loop to monitor your schedule until a keypress          |
    ------------------------------------------------------------------------- */
    do {

        /*
        ---------------------------------------------------------------------
        | Get the current system time based on the 24 hour clock, ie: 23:00 |
        --------------------------------------------------------------------- */
        if (get_x10time())
            chk_time_event();

        /*
        ---------------------------------------------------------------------
        | Get the next received X-10 command from the driver, Null if none  |
        --------------------------------------------------------------------- */
        if (get_x10cmd(&X10Cmd))
            chk_ctrler_event();
 
        /*
        ---------------------------------------------------------------------
        | If you want to see the time and date when the system catches up   |
        --------------------------------------------------------------------- */
        #ifdef __TURBOC__
        if (PowerOutage) {
            gotoxy(1, wherey() );
            printf("    %-8s  %s", X10Time, X10Date);
        }
        #endif

        /*
        ---------------------------------------------------------------------
        | The part of your schedule that would deal with any expired timers |
        --------------------------------------------------------------------- */
        chk_timer_event();

        /*
        ---------------------------------------------------------------------
        | Put any other conditions in this area.  Its wide open             |
        --------------------------------------------------------------------- */
        if (kbhit()) {
            KeyHit = toupper(getch());

            if (KeyHit >= 'A' && KeyHit <= 'P') {
                hc[0] = KeyHit;
                display_status(hc);
                KeyHit = '\0';
            }
            else if (KeyHit == 'V') {
                KeyHit = '\0';
                display_x10_globals();
                printf("    Jr is working: %s\n\n", (JrIsWorking) ? "Yes" : "No");
            }
            else if (KeyHit == 'T') {
                KeyHit = '\0';
                display_x10_globals();
                printf("    Jr is working: %s\n\n", (JrIsWorking) ? "Yes" : "No");
                printf("    Enter new time (hh:mm): ");
                gets(NewTime);
                if (strlen(NewTime) == 5) {
                    set_time(NewTime);
                    printf("    Time has been reset to %s\n\n", NewTime);
                }
            }
        }

    /*
    -------------------------------------------------------------------------
    | End of the schedule loop.  If the user pressed the x key then exit    |
    ------------------------------------------------------------------------- */
    } while (KeyHit != 'X');

    /*
    -------------------------------------------------------------------------
    | Make sure you close communications to the TW523 module before you quit|
    ------------------------------------------------------------------------- */
    close_tw523(OUTAGE_FILE);
}

/*
-----------------------------------------------------------------------------
| Function      | startup                                                   |
|               |                                                           |
| Purpose       | To initialize all of your X-10 modules to a known state.  |
|               | Usually this would be to turn all modules to Off.         |
|               | This is where you would also do any first time            |
|               | initialization of any variables you will be using.        |
|               |                                                           |
| Syntax        | startup()                                                 |
|               |                                                           |
| Parameters    | None                                                      |
|               |                                                           |
| Example       | startup();                                                |
|               |                                                           |
| Assumes       | Nothing                                                   |
|               |                                                           |
| Side Effects  | Depends on what you do inside the procedure               |
|               |                                                           |
| Returns       | Nothing                                                   |
|               |                                                           |
|---------------------------------------------------------------------------|
| Notes:                                                                    |
|                                                                           |
----------------------------------------------------------------------------- */

void startup(void)
{
    /*
    -------------------------------------------------------------------------
    | Define the addresses of your modules... REQUIRED !                    |
    |                                                                       |
    | Parameters:      Addr     Type     Level      Refresh     Udp         |
    ------------------------------------------------------------------------- */

    initialize_module("E1",     APPL,       0,      NO_REFRESH, NULL);
    initialize_module("E2",     LAMP,       0,      REFRESH,    NULL);
    initialize_module("E3",     SWITCH,     0,      REFRESH,    NULL);
    initialize_module("E4",     LAMP,       0,      REFRESH,    NULL);
    initialize_module("E5",     SWITCH,     0,      REFRESH,    NULL);
    initialize_module("E6",     SWITCH,     0,      REFRESH,    NULL);

    initialize_module("F1",     APPL,       0,      NO_REFRESH, NULL);

    initialize_module("G3",     SWITCH,     0,      REFRESH,    NULL);

    initialize_module("O1",     SWITCH,     0,      REFRESH,    NULL);
    initialize_module("O2",     SWITCH,     0,      REFRESH,    NULL);

    /*
    -------------------------------------------------------------------------
    | Do your custom variable initializations, ie: your days off            |
    ------------------------------------------------------------------------- */
    if (dow_is_between(WED, THU)) {
		JrIsWorking = NO;
		CancelBedtime = YES;
    }
    else {
		JrIsWorking = YES;
		CancelBedtime = NO;
    }

    display_x10_globals();
	printf("    Jr is working: %s", (JrIsWorking) ? "Yes\n\n" : "No\n\n");

}

/*
-----------------------------------------------------------------------------
| Function      | chk_time_event                                            |
|               |                                                           |
| Purpose       | Respond to preset time of day events.                     |
|               |                                                           |
| Syntax        | chk_time_event(void)                                      |
|               |                                                           |
| Parameters    | None.                                                     |
|               |                                                           |
| Example       | if (get_x10time())                                        |
|               |     chk_time_event();                                     |
|               |                                                           |
| Assumes       | Nothing.                                                  |
|               |                                                           |
| Side Effects  | Whatever you preprogram.                                  |
|               |                                                           |
| Returns       | Nothing.                                                  |
|               |                                                           |
|---------------------------------------------------------------------------|
| Notes: This is just an example of how you can break down the program      |
|        into smaller, more manageable modules.                             |
|                                                                           |
|        Since get_x10time() will return TRUE only when a new minute has    |
|        come around, you can use that logical to branch to this type of    |
|        function.  This will not only make your program easier but will    |
|        also speed it up since it does not have to check all of your if's  |
|        on every loop, now it will only check when a new minute arrives.   |
|                                                                           |
----------------------------------------------------------------------------- */

void chk_time_event(void)
{
    if (time_is("05:00")) {                     /* Dales wakeup time                */
        turn_module("F1", ON);                  /*      Coffee pot on               */
        if (time_is_before(Dawn)) {
            turn_all_modules("O", OFF);         /*      Outside lights off          */
            dim_module("E5", 8);                /*      Stove light to lvl 8        */
        }
    }

    if (time_is(Dawn)) {                        /* First morning light              */
        turn_all_modules("E", OFF);             /*      Inside lights off           */
        turn_all_modules("O", OFF);             /*      Outside lights off          */
    }

    if (time_is(Dusk))                          /* Its now dark outside             */
        turn_all_modules("O", ON);              /*      Outside lights on           */

    if (time_is("19:30"))                       /* Boys pajamas time                */
        flash_module("E3", 15);                 /*      Flash living rm table lamp  */

    if (time_is("19:45"))                       /* Boys get toys for bed            */
        turn_module("E6", ON);                  /*      Playroom light on           */

    if (time_is("20:15")) {                     /* Boys already in bed              */
        turn_module("E6", OFF);                 /*      Playroom light off          */
        turn_module("F1", OFF);                 /*      Coffee pot off              */
    }

    if (time_is("21:30") && !CancelBedtime) {   /* Dales Bedtime                    */
        dim_module("E2", 10);                   /*      Living table lamp to lvl 10 */
        select_modules("E1",                    /*      Living room flourescents    */
                       "E3",                    /*      Living room ovhd lights     */
                       "E4",                    /*      Kitchen stove light         */
                       "E5",                    /*      Kitchen ovhd lights         */
                       "E6", NULL);             /*      Playroom lights             */
        turn_selected_modules("E", OFF);        /*      Turn them all off           */
        turn_module("E5", ON);                  /*      Kitchen ovhd lights on      */
        dim_module("G3", 5);                    /*      Master bed lights to lvl 5  */
    }

    if ((time_is("22:10") && dow_is(FRI)) ||                /* If its time for JR to come home  */
        (time_is("21:59") && dow_is_between(SAT, SUN)) ||
		(time_is("22:59") && dow_is_between(MON, TUE))) {
            dim_module("E2", 8);                            /*      Living table lamp to lvl 8  */
            dim_module("E5", 9);                            /*      Kitchen ovhd lights to lvl 9*/
            turn_module("O1", OFF);                         /*      Front porch light off       */
            pause(3);                                       /*      Force motion light on       */
            turn_module("O1", ON);                          /*      Front porch light on        */
            ResetMotion = YES;
    }

    if (time_is("23:30") && ResetMotion) {      /* Jr is home by now                */
        turn_module("O1", OFF);                 /*      Front porch light off       */
        pause(10);                              /*      Force back to sense motion  */
        turn_module("O1", ON);                  /*      Front porch light on        */
        ResetMotion = NO;
    }

    if (time_is("00:01")) {                     /* Its a new day so set variables   */
        if (dow_is_between(WED, THU)) {       /*      If Jr does not work today   */
            JrIsWorking = NO;                   /*         set flag saying such     */
            CancelBedtime = YES;                /*         and cancel dales bedtime */
        }
        else {                                  /*      If Jr does work today       */
            JrIsWorking = YES;                  /*         set flag saying such     */
            CancelBedtime = NO;                 /*         dont cancel dales bedtime*/
        }
        display_x10_globals();
        printf("    Jr is working: %s", (JrIsWorking) ? "Yes\n\n" : "No\n\n");
    }

    if (time_is("02:00")) {                     /* Safety reasons                   */
        turn_all_modules("E", OFF);             /*      All inside lights off       */
        turn_all_modules("F", OFF);             /*      All appliance modules off   */
        turn_all_modules("G", OFF);             /*      All bedroom lights off      */
    }
}

/*
-----------------------------------------------------------------------------
| Function      | chk_ctrler_event                                          |
|               |                                                           |
| Purpose       | Respond to preset events based on signals generated by    |
|               | X10 controllers.                                          |
|               |                                                           |
| Syntax        | chk_ctrler_event(void)                                    |
|               |                                                           |
| Parameters    | None.                                                     |
|               |                                                           |
| Example       | if (get_x10cmd(&X10Cmd))                                  |
|               |     chk_ctrler_event(void);                               |
|               |                                                           |
| Assumes       | You are using the global X10Cmd to hold incoming signals. |
|               |                                                           |
| Side Effects  | Whatever you preprogram.                                  |
|               |                                                           |
| Returns       | Nothing.                                                  |
|               |                                                           |
|---------------------------------------------------------------------------|
| Notes: This is just an example of how you can break down the program      |
|        into smaller, more manageable modules.                             |
|                                                                           |
|        Since get_x10cmd() will return TRUE only when a new X10 Command is |
|        received, you can use that logical to branch to this type of       |
|        function.  This will not only make your program easier but will    |
|        also speed it up since it does not have to check all of your if's  |
|        on every loop, now it will only check when a new signal arrives.   |
|                                                                           |
----------------------------------------------------------------------------- */

void chk_ctrler_event(void)
{
    if (x10cmd_is(&X10Cmd, "G1", OFF)) {        /* Master bedroom ctrler            */
        turn_all_modules("E", OFF);             /*      All modules off             */
        turn_all_modules("G", OFF);             /*      Master bed lights off       */
    }

    if (x10cmd_is(&X10Cmd, "G1", ON)) {         /* Master bedroom ctrler            */
        turn_all_modules("E", ON);              /*      All lights on               */
        turn_all_modules("G", ON);              /*      All lights on               */
        turn_all_modules("O", ON);              /*      All lights on               */
    }

    if (x10cmd_is(&X10Cmd, "G4", OFF)) {        /* Master bedroom ctrler            */
		JrIsWorking = NO;                       /*      Jr is not working today     */
        CancelBedtime = YES;                    /*      Cancel dales bedtime        */
    }

    if (x10cmd_is(&X10Cmd, "G4", ON)) {         /* Master bedroom ctrler            */
        JrIsWorking = YES;                      /*      Jr is working today         */
        CancelBedtime = NO;                     /*      Cancel dales bedtime        */
    }

    if (x10cmd_is(&X10Cmd, "H4", ON)) {         /* Living room sundowner ctrler     */
        dim_module("E2", 8);                    /*      Living table lamp to lvl 8  */
		start_timer(1, "00:10:00");             /*      Start a 10 minute timer     */
    }

    if (x10cmd_is(&X10Cmd, "H3", ON))           /* Living room sundowner ctrler     */
        turn_all_modules("O", ON);              /*      Outside lights on           */

    if (x10cmd_is(&X10Cmd, "H3", OFF))          /* Living room sundowner ctrler     */
        turn_all_modules("O", OFF);             /*      Outside lights off          */

}

/*
-----------------------------------------------------------------------------
| Function      | chk_timer_event                                           |
|               |                                                           |
| Purpose       | Respond to your preset timers.                            |
|               |                                                           |
| Syntax        | chk_timer_event(void)                                     |
|               |                                                           |
| Parameters    | None.                                                     |
|               |                                                           |
| Example       | chk_timer_event(void);                                    |
|               |                                                           |
| Assumes       | Nothing.                                                  |
|               |                                                           |
| Side Effects  | Whatever you preprogram.                                  |
|               |                                                           |
| Returns       | Nothing.                                                  |
|               |                                                           |
|---------------------------------------------------------------------------|
| Notes: This is just an example of how you can break down the program      |
|        into smaller, more manageable modules.                             |
|                                                                           |
|        You would have this called with every iteration of the main loop.  |
|                                                                           |
----------------------------------------------------------------------------- */

void chk_timer_event(void)
{
    if (timer_expired(SAVE_STATUS_TIMER)) {                     /* Time to save status of modules   */
        reset_timer(SAVE_STATUS_TIMER);                         /*      reset timer for next save   */
        printf("\n   Saving status table at %s\n", X10Time);
        save_status(OUTAGE_FILE);                               /*      save status to disk         */
    }

    if (timer_status(1) == EXPIRED)                             /* Our 10 minute timer expired      */
        dim_module("E2", 30);                                   /*      Living table lamp to full   */

}

/*
-----------------------------------------------------------------------------
| End of file: X10_EXMP.C                                                   |
----------------------------------------------------------------------------- */
