/*
-----------------------------------------------------------------------------
| Source File   | X10_SHEL.C                                                |
|               |                                                           |
| System        | X-10 Professional Developers Library for C, v. 3.1        |
|               |                                                           |
| Purpose       | Example program shell for Home Automation using X-10      |
|               | modules and a computer serial port interface.             |
|               |                                                           |
| Author        | Dale Morrison                                             |
| Copyright     | Smart Home Technology, 1993, 1994                         |
-----------------------------------------------------------------------------


-----------------------------------------------------------------------------
| Load the headers for standard C functions used in this shell of a program |
----------------------------------------------------------------------------- */
#include <conio.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

/*
-----------------------------------------------------------------------------
| Load the file containing declarations and constants for the X10 Library   |
----------------------------------------------------------------------------- */
#include "X10_C.H"

/*
-----------------------------------------------------------------------------
| Define your own unique constants.                                         |
----------------------------------------------------------------------------- */
#define OUTAGE_FILE         "outage.sta"      /* Name of the recovery file    */
#define SAVE_STATUS_TIMER   0                 /* Timer number to use          */
#define SAVE_STATUS_TIME    "00:30:00"        /* How often to save status     */
#define RECV_TIMEOUT        .5                /* Seconds to wait for a cmd    */
#define LONGITUDE           122.2             /* Longitude of your city       */
#define LATITUDE            47.37             /* Latitude of your city        */
#define TIME_ZONE           8                 /* Time zone for your city      */
#define START_DAYLITE       ""                /* Have the library calculate   */
#define END_DAYLITE         ""                /* Have the library calculate   */

/*
-----------------------------------------------------------------------------
| Define your own unique functions.                                         |
----------------------------------------------------------------------------- */
void startup(void);                           /* Initialize modules and variables   */
void chk_time_event(void);                    /* Events dealing with time           */
void chk_ctrler_event(void);                  /* Events dealing with X10 Ctrlers    */
void chk_timer_event(void);                   /* Events dealing with expired timers */

/*
-----------------------------------------------------------------------------
| Identify your own global variables.                                       |
----------------------------------------------------------------------------- */
int DayOff;

/*
-----------------------------------------------------------------------------
| Main body of your application.                                            |
----------------------------------------------------------------------------- */
void main()
{
	char KeyHit, hc[2] = " ", NewTime[9] = "";

    /*
    -------------------------------------------------------------------------
    | Your basic clear the screen command                                   |
    ------------------------------------------------------------------------- */
    #ifdef __TURBOC__
        clrscr();
    #else
        system("cls");
    #endif

    /*
    -------------------------------------------------------------------------
    | Open communications to the TW523 module                               |
    ------------------------------------------------------------------------- */
    open_tw523(OUTAGE_FILE);

    /*
    -------------------------------------------------------------------------
    | Do you want to display what is being transmitted and received?        |
    ------------------------------------------------------------------------- */
	ShowTransmit = YES;

    /*
    -------------------------------------------------------------------------
    | Number of seconds you want the system to wait for a complete x10 cmd  |
    | The system defaults to 1 second.  This in no way slows the system down|
    ------------------------------------------------------------------------- */
    set_recv_timeout(RECV_TIMEOUT);

    /*
    -------------------------------------------------------------------------
    | Initialize the internal variables for sunrise/sunset calculations.    |
    ------------------------------------------------------------------------- */
    set_map_coord(LONGITUDE, LATITUDE, TIME_ZONE, START_DAYLITE, END_DAYLITE);

    /*
    -------------------------------------------------------------------------
    | Do your Startup procedure to initialize your modules and variables    |
    ------------------------------------------------------------------------- */
    startup();

    /*
    -------------------------------------------------------------------------
    | Start a timer to be used for saving the status tables for outages.    |
    ------------------------------------------------------------------------- */
    start_timer(SAVE_STATUS_TIMER, SAVE_STATUS_TIME);

    /*
    -------------------------------------------------------------------------
    | Check to see if there was a power outage since the last boot up.      |
    ------------------------------------------------------------------------- */
    if (PowerOutage) {
        printf("\n!!! There was a power outage.... ");
        printf("one moment while I catch up!\n\n");
        play_catchup(OUTAGE_FILE);
    }

    /*
    -------------------------------------------------------------------------
    | Now start the loop to monitor your schedule until an 'X' is pressed   |
    ------------------------------------------------------------------------- */
    do {

        /*
        ---------------------------------------------------------------------
        | Get the current system time based on the 24 hour clock, ie: 23:00 |
        --------------------------------------------------------------------- */
        if (get_x10time())
            chk_time_event();

        /*
        ---------------------------------------------------------------------
        | Get the next received X-10 command from the buffer.               |
        --------------------------------------------------------------------- */
        if (get_x10cmd(&X10Cmd))
            chk_ctrler_event();

        /*
        ---------------------------------------------------------------------
        | If you want to see the time and date as the system catches up     |
        --------------------------------------------------------------------- */
        #ifdef __TURBOC__
        if (PowerOutage) {
            gotoxy(1, wherey() );
            printf("    %-8s  %s", X10Time, X10Date);
        }
        #endif

        /*
        ---------------------------------------------------------------------
        | The part of your schedule that would deal with any expired timers |
        --------------------------------------------------------------------- */
        chk_timer_event();

        /*
        ---------------------------------------------------------------------
        | Put any other conditions in this area.  Its wide open             |
        --------------------------------------------------------------------- */

        /*
        ---------------------------------------------------------------------
        | End of the schedule loop. If the user pressed the x key then exit |
        --------------------------------------------------------------------- */
        if (kbhit()) {
            KeyHit = toupper(getch());

            if (KeyHit >= 'A' && KeyHit <= 'P') {       /* House code status display */
                hc[0] = KeyHit;
				display_status(hc);
                KeyHit = '\0';
            }
            else if (KeyHit == 'V') {                   /* Display internal globals  */
                display_x10_globals();
                KeyHit = '\0';
            }
            else if (KeyHit == '1') {                   /* Just a test, flash a module */
                flash_module("P1", 5);
                KeyHit = '\0';
            }
            else if (KeyHit == 'T') {
                KeyHit = '\0';
                display_x10_globals();
				printf("    Enter new time (hh:mm:ss): ");
				gets(NewTime);
				if (strlen(NewTime) == 8) {
					set_time(NewTime);
					printf("    Time has been reset to %s\n\n", NewTime);
                }
            }
        }

    } while (KeyHit != 'X');

    /*
    -------------------------------------------------------------------------
    | Make sure you close communications to the TW523 module before you quit|
    ------------------------------------------------------------------------- */
    close_tw523(OUTAGE_FILE);
}

/*
-----------------------------------------------------------------------------
| Function      | startup                                                   |
|               |                                                           |
| Purpose       | To initialize all of your X-10 modules.                   |
|               | This is where you would also do any first time            |
|               | initialization of any variables you will be using.        |
|               |                                                           |
| Syntax        | startup()                                                 |
|               |                                                           |
| Parameters    | None                                                      |
|               |                                                           |
| Example       | startup();                                                |
|               |                                                           |
| Assumes       | Nothing                                                   |
|               |                                                           |
| Side Effects  | Depends on what you do inside the procedure               |
|               |                                                           |
| Returns       | Nothing                                                   |
|               |                                                           |
|---------------------------------------------------------------------------|
| Notes:                                                                    |
|                                                                           |
----------------------------------------------------------------------------- */

void startup(void)
{
    /*
    -------------------------------------------------------------------------
    | Define the addresses of your modules... REQUIRED !                    |
    |                                                                       |
    | Parameters:      Addr     Type     Level      Refresh     Udp         |
    ------------------------------------------------------------------------- */
    initialize_module("P1",     APPL,      0,       REFRESH,    NULL);
    
    /*
    -------------------------------------------------------------------------
    | If you want identify your controllers, you might do so here...        |
    ------------------------------------------------------------------------- */

    /*
    -------------------------------------------------------------------------
    | You might want to create your groups at this point.                   |
    ------------------------------------------------------------------------- */

    /*
    -------------------------------------------------------------------------
    | Do your personal variable initializations, ie: your days off          |
    ------------------------------------------------------------------------- */
    if (dow_is(MON) || dow_is(TUE))
		DayOff = TRUE;

}

/*
-----------------------------------------------------------------------------
| Function      | chk_time_event                                            |
|               |                                                           |
| Purpose       | Respond to preset time of day events.                     |
|               |                                                           |
| Syntax        | chk_time_event(void)                                      |
|               |                                                           |
| Parameters    | None.                                                     |
|               |                                                           |
| Example       | if (get_x10time())                                        |
|               |     chk_time_event();                                     |
|               |                                                           |
| Assumes       | Nothing.                                                  |
|               |                                                           |
| Side Effects  | Whatever you preprogram.                                  |
|               |                                                           |
| Returns       | Nothing.                                                  |
|               |                                                           |
|---------------------------------------------------------------------------|
| Notes: This is just an example of how you can break down the program      |
|        into smaller, more manageable modules.                             |
|                                                                           |
|        Since get_x10time() will return TRUE only when a new minute has    |
|        come around, you can use that logical to branch to this type of    |
|        function.  This will not only make your program easier but will    |
|        also speed it up since it does not have to check all of your if's  |
|        on every loop, now it will only check when a new minute arrives.   |
|                                                                           |
----------------------------------------------------------------------------- */

void chk_time_event(void)
{
    if (time_is("00:01"))
        display_x10_globals();

}

/*
-----------------------------------------------------------------------------
| Function      | chk_ctrler_event                                          |
|               |                                                           |
| Purpose       | Respond to preset events based on signals generated by    |
|               | X10 controllers.                                          |
|               |                                                           |
| Syntax        | chk_ctrler_event(void)                                    |
|               |                                                           |
| Parameters    | None.                                                     |
|               |                                                           |
| Example       | if (get_x10cmd(&X10Cmd))                                  |
|               |     chk_ctrler_event(void);                               |
|               |                                                           |
| Assumes       | You are using the global X10Cmd to hold incoming signals. |
|               |                                                           |
| Side Effects  | Whatever you preprogram.                                  |
|               |                                                           |
| Returns       | Nothing.                                                  |
|               |                                                           |
|---------------------------------------------------------------------------|
| Notes: This is just an example of how you can break down the program      |
|        into smaller, more manageable modules.                             |
|                                                                           |
|        Since get_x10cmd() will return TRUE only when a new X10 Command is |
|        received, you can use that logical to branch to this type of       |
|        function.  This will not only make your program easier but will    |
|        also speed it up since it does not have to check all of your if's  |
|        on every loop, now it will only check when a new signal arrives.   |
|                                                                           |
----------------------------------------------------------------------------- */

void chk_ctrler_event(void)
{
    if (x10cmd_is(&X10Cmd, "C1", ON))
        flash_module("P1", 5);

}

/*
-----------------------------------------------------------------------------
| Function      | chk_timer_event                                           |
|               |                                                           |
| Purpose       | Respond to your preset timers.                            |
|               |                                                           |
| Syntax        | chk_timer_event(void)                                     |
|               |                                                           |
| Parameters    | None.                                                     |
|               |                                                           |
| Example       | chk_timer_event(void);                                    |
|               |                                                           |
| Assumes       | Nothing.                                                  |
|               |                                                           |
| Side Effects  | Whatever you preprogram.                                  |
|               |                                                           |
| Returns       | Nothing.                                                  |
|               |                                                           |
|---------------------------------------------------------------------------|
| Notes: This is just an example of how you can break down the program      |
|        into smaller, more manageable modules.                             |
|                                                                           |
|        You would have this called with every iteration of the main loop.  |
|                                                                           |
----------------------------------------------------------------------------- */

void chk_timer_event(void)
{
    if (timer_expired(SAVE_STATUS_TIMER)) {
        reset_timer(SAVE_STATUS_TIMER);
        printf("\n    Saving status table at %s\n", X10Time);
        save_status(OUTAGE_FILE);
    }

}

/*
-----------------------------------------------------------------------------
| End of file: X10_SHEL.C                                                   |
----------------------------------------------------------------------------- */
